% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_infections.R
\name{estimate_infections}
\alias{estimate_infections}
\title{Estimate Infections, the Time-Varying Reproduction Number and the Rate of Growth}
\usage{
estimate_infections(
  reported_cases,
  model = NULL,
  samples = 1000,
  stan_args = NULL,
  method = "exact",
  family = "negbin",
  generation_time,
  CrIs = c(0.2, 0.5, 0.9),
  delays = list(),
  horizon = 7,
  gp = list(basis_prop = 0.3, boundary_scale = 2, lengthscale_alpha = 4.5,
    lengthscale_beta = 21.5, alpha_sd = 0.1),
  rt_prior = list(mean = 1, sd = 1),
  week_effect = TRUE,
  use_breakpoints = TRUE,
  stationary = FALSE,
  future_rt = "project",
  burn_in = 0,
  prior_smoothing_window = 7,
  future = FALSE,
  max_execution_time = Inf,
  return_fit = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{reported_cases}{A data frame of confirmed cases (confirm) by date (date). confirm must be integer and date must be
in date format.}

\item{model}{A compiled stan model. By default uses the internal package model.}

\item{samples}{Numeric, defaults to 1000. Number of samples post warmup.}

\item{stan_args}{A list of stan arguments to be passed to \code{rstan::sampling} or \code{rstan::vb} (when using the "exact"
or "approximate" method). For \code{method = approximate} an additional argument \code{trials} indicates the number of attempts to make
using variational inference before returning an error (as stochastic failure is possible). The default for this is 5.}

\item{method}{A character string defaults to "exact". Also accepts "approximate". Indicates the fitting method to be used
this can either be "exact" (NUTs sampling) or "approximate" (variational inference). The exact approach returns samples
from the posterior whilst the approximate method returns approximate samples. The approximate method is likely to return results
several order of magnitudes faster than the exact method.}

\item{family}{A character string indicating the reporting model to use. Defaults to negative
binomial ("negbin") with poisson ("poisson") also supported.}

\item{generation_time}{A list containing the mean, standard deviation of the mean (mean_sd),
standard deviation (sd), standard deviation of the standard deviation and the maximum allowed value for the
generation time (assuming a gamma distribution).}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{delays}{A list of delays (i.e incubation period/reporting delay) between infection and report.
Each list entry must also be a list containing the mean, standard deviation of the mean (mean_sd),
standard deviation (sd), standard deviation of the standard deviation and the maximum allowed value for the
that delay (assuming a lognormal distribution with all parameters excepting the max allowed value
on the log scale). To use no delays set this to \code{list()}.}

\item{horizon}{Numeric, defaults to 7. Number of days into the future to forecast.}

\item{gp}{List controlling the Gaussian process approximation if set to \code{list()} then \code{Rt} is assumed to be constant unless
other settings introduce variation. If set must contain the \code{basis_prop} (number of basis functions based on scaling the time points)
which defaults to 0.3 and must be between 0 and 1 (increasing this increases the accuracy of the approximation and the cost of
additional compute. Must also contain the \code{boundary_scale} (multiplied by half the range of the input time series). Increasing this
increases the accuracy of the approximation at the cost of additional compute.
See here: https://arxiv.org/abs/2004.11408 for more information on setting these parameters.
Must also contain the  \code{lengthscale_alpha} and \code{lengthscale_beta}. These tune the prior of the lengthscale. Principled
values can be obtained using \code{tune_inv_gamma} which optimises based on the desired truncation (which should be based on the scale
of the observed data). The default is tuned to have 98\% of the density of the distribution between 2 and 21 days. Finally the list must
contain \code{alpha_sd} the standard deviation for the alpha parameter of the gaussian process. This defaults to 0.1.}

\item{rt_prior}{A list contain the mean and standard deviation (sd) of the lognormally distributed prior for
Rt. By default this is assumed to be mean 1 with a standard deviation of 1 (note in model these will be mapped to
log space). To infer infections only using non-parametric backcalculation set this to \code{list()}.}

\item{week_effect}{Logical, defaults TRUE. Should weekly reporting effects be estimated.}

\item{use_breakpoints}{Logical, defaults to TRUE but only active if a \code{breakpoint} variable is present in the input data.
Breakpoints should be defined as 1 if present and otherwise 0. By default breakpoints are fit jointly with
a global non-parametric effect and so represent a conservative estimate of breakpoint changes. To specify a random walk define
breakpoints every n days (so every 7 days for a weekly random walk) and disable the gaussian process using \code{gp = list()}.}

\item{stationary}{Logical, defaults to FALSE. Should Rt be estimated with a global mean. When estimating Rt
this should substantially improve run times but will revert to the global average for real time and forecasted estimates.
This setting is most appropriate when estimating historic Rt or when combined with breakpoints.}

\item{future_rt}{A character string or integer. This argument indicates how to set future Rt values. Supported
options are to project using the Rt model ("project"), to use the latest estimate based on partial data ("latest"),
to use the latest estimate based on data that is over 50\% complete ("estimate"). If an integer is supplied then the Rt estimate
from this many days into the future (or past if negative) past will be used forwards in time.}

\item{burn_in}{Numeric, defaults to 0. The number of initial Rt estimates to discard. This argument may be used to reduce
spurious findings when running \code{estimate_infections} on a partial timeseries (as the earliest estimates will not be informed by
all cases that occurred only those supplied to \code{estimate_infections}). The combined delays used will inform the appropriate length
of this burn in but 7 days is likely a sensible starting point.}

\item{prior_smoothing_window}{Numeric defaults to 7. The number of days over which to take a rolling average
for the prior based on reported cases.}

\item{future}{Logical, defaults to \code{FALSE}. Should stan chains be run in parallel using \code{future}. This allows users to have chains
fail gracefully (i.e when combined with \code{max_execution_time}). Should be combined with a call to \code{future::plan}}

\item{max_execution_time}{Numeric, defaults to Inf (seconds). If set will kill off processing of each chain if not finished within the specified timeout.
When more than 2 chains finish successfully estimates will still be returned. If less than 2 chains return within the allowed time then estimation
will fail with an informative error.}

\item{return_fit}{Logical, defaults to FALSE. Should the fitted stan model be returned.}

\item{verbose}{Logical, defaults to \code{FALSE}. Should verbose debug progress messages be printed. Corresponds to the "DEBUG" level from
\code{futile.logger}. See \code{setup_logging} for more detailed logging options.}
}
\description{
This function uses a non-parametric approach to reconstruct cases by date of infection from reported
cases. It can optionally then estimate the time-varying reproduction number and the rate of growth.
}
\examples{
\donttest{
# get example case counts
reported_cases <- EpiNow2::example_confirmed[1:50]

# add a dummy breakpoint (used only when optionally estimating breakpoints)
reported_cases_bp <- data.table::copy(reported_cases)[,
              breakpoint := data.table::fifelse(date == as.Date("2020-03-16"), 1, 0)]
# set up example generation time
generation_time <- get_generation_time(disease = "SARS-CoV-2", source = "ganyani")
# set delays between infection and case report 
incubation_period <- get_incubation_period(disease = "SARS-CoV-2", source = "lauer")
reporting_delay <- list(mean = log(5), mean_sd = log(2),
                        sd = log(2), sd_sd = log(1.5), max = 30)
      
# Note: all examples below have been tuned to reduce the runtimes of examples
# these settings are not suggesed for real world use.                   
# run model with default setting
def <- estimate_infections(reported_cases, generation_time = generation_time,
                           delays = list(incubation_period, reporting_delay), 
                           stan_args = list(warmup = 200, control = list(adapt_delta = 0.8),
                                            cores = ifelse(interactive(), 4, 1)))

plots <- report_plots(summarised_estimates = def$summarised, reported = reported_cases)
plots$summary

# run the model using the approximate method (variational inference)
approx <- estimate_infections(reported_cases, generation_time = generation_time,
                              delays = list(incubation_period, reporting_delay),
                              method = "approximate")

plots <- report_plots(summarised_estimates = approx$summarised, reported = reported_cases)
plots$summary

# run the model with default settings using the future backend 
## (combine with a call to future::plan to make this parallel).
def_future <- estimate_infections(reported_cases, generation_time = generation_time,
                                  delays = list(incubation_period, reporting_delay),
                                  stan_args = list(warmup = 200, 
                                                   control = list(adapt_delta = 0.9),
                                                   cores = ifelse(interactive(), 4, 1)))

plots <- report_plots(summarised_estimates = def_future$summarised, reported = reported_cases)
plots$summary                          
                           
# run model with Rt fixed into the future using the latest estimate
fixed_rt <- estimate_infections(reported_cases, generation_time = generation_time,
                                delays = list(incubation_period, reporting_delay),
                                stan_args = list(warmup = 200, 
                                                 control = list(adapt_delta = 0.9),
                                                 cores = ifelse(interactive(), 4, 1)),
                                future_rt = "latest")

plots <- report_plots(summarised_estimates = fixed_rt$summarised, reported = reported_cases)
plots$summary

# run the model with default settings on a later snapshot of 
# data (use burn_in here to remove the first week of estimates that may
# be impacted by this most).
snapshot_cases <- EpiNow2::example_confirmed[80:130]
snapshot <- estimate_infections(reported_cases, generation_time = generation_time,
                                delays = list(incubation_period, reporting_delay),
                                stan_args = list(warmup = 200, 
                                                 control = list(adapt_delta = 0.9),
                                                 cores = ifelse(interactive(), 4, 1)),
                                burn_in = 7)

plots <- report_plots(summarised_estimates = snapshot$summarised, reported = snapshot_cases)
plots$summary    

# run model with stationary Rt assumption (likely to provide biased real-time estimates)
stat <- estimate_infections(reported_cases, generation_time = generation_time,
                            delays = list(incubation_period, reporting_delay),
                            stan_args = list(warmup = 200, cores = ifelse(interactive(), 4, 1),
                                             control = list(adapt_delta = 0.9)),
                            stationary = TRUE)

plots <- report_plots(summarised_estimates = stat$summarised, reported = reported_cases)
plots$summary
       
# run model with fixed Rt assumption 
fixed <- estimate_infections(reported_cases, generation_time = generation_time,
                             delays = list(incubation_period, reporting_delay),
                             stan_args = list(warmup = 200, cores = ifelse(interactive(), 4, 1),
                                              control = list(adapt_delta = 0.9)),
                             gp = list())

plots <- report_plots(summarised_estimates = fixed$summarised, reported = reported_cases)
plots$summary

# run model with no delays 
no_delay <- estimate_infections(reported_cases, generation_time = generation_time,
                                stan_args = list(warmup = 200,
                                                 cores = ifelse(interactive(), 4, 1),
                                                 control = list(adapt_delta = 0.9)))

plots <- report_plots(summarised_estimates = no_delay$summarised, reported = reported_cases)
plots$summary         
              
# run model with breakpoints                                                                      
bkp <- estimate_infections(reported_cases_bp, generation_time = generation_time,
                           delays = list(incubation_period, reporting_delay),
                           stan_args = list(warmup = 200, 
                                            cores = ifelse(interactive(), 4, 1),
                                            control = list(adapt_delta = 0.9)))

plots <- report_plots(summarised_estimates = bkp$summarised, reported = reported_cases)
plots$summary
             
# run model with breakpoints but with constrained non-linear change over time 
# rhis formulation may increase the apparent effect of the breakpoint but needs to be tested using
# model fit criteria (i.e LFO). 
cbkp <- estimate_infections(reported_cases_bp, generation_time = generation_time,
                           delays = list(incubation_period, reporting_delay),
                           gp = list(basis_prop = 0.3, boundary_scale = 2, 
                                     lengthscale_mean = 20, lengthscale_sd = 1),
                           stan_args = list(warmup = 200, 
                           cores = ifelse(interactive(), 4, 1),
                           control = list(adapt_delta = 0.9)))                                                                   

plots <- report_plots(summarised_estimates = cbkp$summarised, reported = reported_cases)
plots$summary
# breakpoint effect
cbkp$summarised[variable == "breakpoints"]

# run model with breakpoints but otherwise static Rt
# This formulation may increase the apparent effect of the breakpoint but needs to be tested using
# model fit criteria (i.e LFO).           
fbkp <- estimate_infections(reported_cases_bp, generation_time = generation_time,
                            delays = list(incubation_period, reporting_delay),
                            stan_args = list(warmup = 200, 
                                             cores = ifelse(interactive(), 4, 1),
                                             control = list(adapt_delta = 0.9)),
                            gp = list())                                                         

plots <- report_plots(summarised_estimates = fbkp$summarised, reported = reported_cases)
plots$summary
# breakpoint effect
fbkp$summarised[variable == "breakpoints"]

# run model without Rt estimation (just backcalculation)
backcalc <- estimate_infections(reported_cases, generation_time = generation_time,
                               delays = list(incubation_period, reporting_delay),
                               stan_args = list(warmup = 200, 
                                                cores = ifelse(interactive(), 4, 1),
                                                control = list(adapt_delta = 0.9)),
                               rt_prior = list())

# plot just infections as report_plots does not support the backcalculation only model
plot_estimates(estimate = backcalc$summarised[variable == "infections"],
               reported = reported_cases, ylab = "Cases")
}                                
}
