% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.fossils.R
\name{sim.fossils.environment}
\alias{sim.fossils.environment}
\title{Simulate fossils under an environment-dependent model of preservation (Holland, 1995)}
\usage{
sim.fossils.environment(
  tree = NULL,
  taxonomy = NULL,
  interval.ages = NULL,
  max.age = NULL,
  strata = NULL,
  proxy.data = NULL,
  PD = 0.5,
  DT = 0.5,
  PA = 0.5,
  root.edge = TRUE,
  use.rates = FALSE,
  pr.1.approx = 0.999,
  use.exact.times = TRUE
)
}
\arguments{
\item{tree}{Phylo object.}

\item{taxonomy}{Taxonomy object.}

\item{interval.ages}{Vector of stratigraphic interval ages, starting with the minimum age of the youngest interval and ending with the maximum age of the oldest interval.}

\item{max.age}{Maximum age of the oldest stratigraphic interval or age at the base of the basin.}

\item{strata}{Number of stratigraphic intervals.}

\item{proxy.data}{Vector of relative water depth or other proxy data. The first number corresponds to the youngest interval. The length of the vector should be 1 less than the length of interval.ages.}

\item{PD}{Preferred depth parameter value or a vector of values.}

\item{DT}{Depth tolerance parameter value or a vector of values.}

\item{PA}{Peak abundance parameter value or a vector of values.}

\item{root.edge}{If TRUE include the root edge. Default = TRUE.}

\item{use.rates}{If TRUE convert per interval sampling probability into a per interval Poisson rate. Default = FALSE.}

\item{pr.1.approx}{Value used to approximate sampling probabilities = 1 when use.rates = TRUE.}

\item{use.exact.times}{If TRUE use exact sampling times. If FALSE \code{hmin} and \code{hmax} will equal the start and end times of the corresponding interval. Default = TRUE.}
}
\value{
An object of class fossils.
}
\description{
This function uses a three parameter Gaussian model to simulate non-uniform fossil recovery along a specified phylogeny.
Preservation varies with respect to water depth, which is a useful for proxy for changes in the depositional environment.
The per interval probability of sampling is \deqn{P(collection) = PA e ^ (-(d - PD)^2 / 2*DT^2 ) }
where \emph{PA} is species peak abundance, \emph{PD} is preferred depth, \emph{DT} is depth tolerance and \emph{d} is current water depth.
\emph{PD} is the depth at which the species is most likely to be found and is equivalent to the mean of the distribution.
\emph{PA} is the probability of sampling an occurrence at this depth.
\emph{DT} is the potential of a species to be found at a range of depths and is equivalent to the standard deviation.
Although here fossil recovery is described with respect to water depth, the model could be applied in the context of any environmental gradient. \cr \cr
The model uses a probability of collecting a fossil within a given interval, rather than a rate.

To simulate discrete fossil sampling events and times within each interval we need to convert the probability into a rate
(\code{use.rates = TRUE}). This is done using the formula \deqn{rate = -ln(1 - P(collection)/t) } where \emph{t} is the interval length.
One caveat of this approach is that the model cannot use a probability of 1, as it would correspond to rate = infinity.
In this instance we use an approximation for probabilities = 1 (e.g. \code{pr.1.approx = 0.999}). \cr \cr

Non-uniform interval ages can be specified as a vector (\code{interval.ages}) or a uniform set of interval ages can be specified using
maximum interval age (\code{max.age}) and the number of intervals (\code{strata}), where interval length \eqn{= max.age/strata}. \cr \cr
A vector of values can be specified for the model parameters \emph{PA}, \emph{PD} and \emph{DT} to allow for variation across lineages.
If a vector is provided, each entry will apply to each unique species in the order in which they appear in the taxonomy object (if taxonomy is provided),
or to each unique edge in the order in which they appear in the tree object.
If the tree object has a root edge (\code{root.edge}), the first entry in the vector will apply to this edge. \cr \cr
Fossils can be simulated for a phylo (\code{tree}) or taxonomy (\code{taxonomy}) object.
If both are specified, the function uses taxonomy.
If no taxonomic information is provided, the function assumes all speciation is symmetric (i.e. bifurcating, \code{beta = 1}).
}
\examples{
# simulate tree
t = ape::rtree(6)

# assign a max age based on tree height
max.age = tree.max(t)

# generate water depth profile
strata = 7
wd = sim.gradient(strata)

# simulate fossils using tree & max.age and strata
f = sim.fossils.environment(t, max.age = max.age, strata = strata,
proxy.data = wd, PD = 0.5, DT = 1, PA = 1)
plot(f, t, show.proxy = TRUE, proxy.data = wd, strata = strata, show.strata = TRUE)

# simulate fossils using taxonomy & interval.ages
s = sim.taxonomy(t, 0.1, 0.1, 1)
times = seq(0, max.age, length.out = strata + 1)
f = sim.fossils.environment(taxonomy = s, interval.ages = times,
     proxy.data = wd, PD = 0.5, DT = 1, PA = 1)
plot(f, t, strata = strata, binned = TRUE)

# simulate fossils with variable preservation across lineages
dist = function() {runif(1)}
PD = sim.trait.values(1, taxonomy = s, model = "independent", dist = dist,
                     change.pr = 0.1)
f = sim.fossils.environment(taxonomy = s, interval.ages = times,
     proxy.data = wd, PD = PD, DT = 1, PA = 1)
plot(f, t, strata = strata, binned = TRUE)

}
\references{
Holland, S.M. 1995. The stratigraphic distribution of fossils. Paleobiology 21: 92-109.
}
\seealso{
\code{\link{sim.fossils.poisson}}, \code{\link{sim.fossils.intervals}}, \code{\link{sim.trait.values}}
}
\keyword{fossil}
\keyword{non-uniform}
\keyword{preseravtion}
