% --- Source file: predict.GAMens.Rd ---
\name{predict.GAMens}
\alias{predict.GAMens}

\title{Predicts from a fitted GAMens object (i.e., GAMbag, GAMrsm or GAMens classifier).}

\description{
  Generates predictions (classes and class membership probabilities) for observations in a dataframe 
  using a GAMens object (i.e., GAMens, GAMrsm or GAMbag classifier).
}
\usage{
\method{predict}{GAMens}(object, data,...)}

\arguments{
  \item{object}{fitted model object of \code{GAMens} class.}
  \item{data}{data frame with observations to genenerate predictions for. }
  \item{...}{ further arguments passed to or from other methods.}
}

\value{
   An object of class \code{predict.GAMens}, which is a list with the following components:
  \item{pred}{the class membership probabilities generated by the ensemble classifier. }
  \item{class}{the classes predicted by the ensemble classifier. }
  \item{conf}{the confusion matrix which compares the real versus predicted class memberships, based on the \code{class} object. Obtains value \code{NULL} if the testdata is unlabeled. }
}

\references{De Bock, K. W., Coussement, K. and Van den Poel, D. (2010): "Ensemble Classification based on generalized additive models". Computational Statistics & Data Analysis, Vol 54, 6, pp. 1535--1546.

  Breiman, L. (1996): "Bagging predictors". Machine Learning, Vol 24, 2, pp. 123--140.

  Hastie, T. and Tibshirani, R. (1990): "Generalized Additive Models", Chapman and Hall, London.

  Ho, T. K. (1998): "The random subspace method for constructing decision forests". IEEE Transactions on Pattern Analysis and Machine Intelligence, Vol 20, 8, pp. 832--844.  }

\author{Koen W. De Bock \email{K.DeBock@ieseg.fr}, Kristof Coussement \email{K.Coussement@ieseg.fr} and Dirk Van den Poel \email{Dirk.VandenPoel@ugent.be} }


\seealso{ 
	\code{\link{GAMens}},
	\code{\link{GAMens.cv}} }

\examples{
## Load data, mlbench library should be loaded!)
library(mlbench)
data(Sonar)

## Select indexes for training set observations
idx <- c(sample(1:97,60),sample(98:208,70))

## Train GAMrsm using all variables in Sonar dataset. Generate predictions
## for test set observations.
Sonar.GAMrsm <- GAMens(Class~.,Sonar[idx,], autoform=TRUE, iter=10, 
bagging=FALSE, rsm=TRUE)
Sonar.GAMrsm.predict <- predict.GAMens(Sonar.GAMrsm,Sonar[-idx,])


## Load data mlbench library should be loaded!)
library(mlbench)
data(Ionosphere)
Ionosphere_s <- Ionosphere[order(Ionosphere[,35]),]

## Select indexes for training set observations
idx <- c(sample(1:97,60),sample(98:208,70))


## Compare test set classification performance of GAMens, GAMrsm and 
## GAMbag ensembles, using using 4 nonparametric terms and 2 linear terms in the
## Ionosphere dataset
Ionosphere.GAMens <- GAMens(Class~s(V3,4)+s(V4,4)+s(V5,3)+s(V6,5)+V7+V8,
Ionosphere[idx,], autoform=FALSE, iter=10, bagging=TRUE, rsm=TRUE)

Ionosphere.GAMens.predict <- predict.GAMens(Ionosphere.GAMens,
Ionosphere[-idx,])

Ionosphere.GAMrsm <- GAMens(Class~s(V3,4)+s(V4,4)+s(V5,3)+s(V6,5)+V7+V8,
Ionosphere[idx,], autoform=FALSE, iter=10, bagging=FALSE, rsm=TRUE)

Ionosphere.GAMrsm.predict <- predict.GAMens(Ionosphere.GAMrsm,
Ionosphere[-idx,])

Ionosphere.GAMbag <- GAMens(Class~s(V3,4)+s(V4,4)+s(V5,3)+s(V6,5)+V7+V8,
Ionosphere[idx,], autoform=FALSE, iter=10, bagging=TRUE, rsm=FALSE)

Ionosphere.GAMbag.predict <- predict.GAMens(Ionosphere.GAMbag,
Ionosphere[-idx,])

## Calculate AUCs(for function colAUC, load caTools library)
library(caTools)
GAMens.auc <- colAUC(Ionosphere.GAMens.predict[[1]], 
Ionosphere[-idx,"Class"]=="good", plotROC=FALSE)

GAMrsm.auc <- colAUC(Ionosphere.GAMrsm.predict[[1]], 
Ionosphere[-idx,"Class"]=="good", plotROC=FALSE)

GAMbag.auc <- colAUC(Ionosphere.GAMbag.predict[[1]], 
Ionosphere[-idx,"Class"]=="good", plotROC=FALSE)

}

\keyword{models}
\keyword{classif}
