% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select.mtd.comb.kb.R
\name{select.mtd.comb.kb}
\alias{select.mtd.comb.kb}
\title{Maximum Tolerated Dose (MTD) Selection for Drug-combination Trials}
\usage{
select.mtd.comb.kb(
  target,
  npts,
  ntox,
  cutoff.eli = 0.95,
  extrasafe = FALSE,
  offset = 0.05
)
}
\arguments{
\item{target}{The target dose-limiting toxicity (DLT) rate.}

\item{npts}{A \code{J*K} matrix \code{(J<=K)} containing the number of
patients treated at each dose combination.}

\item{ntox}{A \code{J*K} matrix \code{(J<=K)} containing the number of
patients who experienced a dose-limiting toxicity at each dose
combination.}

\item{cutoff.eli}{The cutoff value to eliminate an overly toxic dose and all
higher doses for safety.\cr
The default value is 0.95.}

\item{extrasafe}{Set \code{extrasafe=TRUE} to impose a stricter
stopping rule.\cr
The default is FALSE.}

\item{offset}{A small positive number (between 0 and 0.5) to control how
strict the stopping rule is when \code{extrasafe=TRUE}. A
larger value leads to a stricter stopping rule.\cr
The default value is 0.05.}
}
\value{
The function returns a list with: \cr
\enumerate{
  \item the target toxicity probability (\code{$target}),\cr
  \item the selected MTD (\code{$MTD}),\cr
  \item a matrix with the isotonic estimates of the DLT probability at each
      dose (\code{$p_est}).
}
}
\description{
This function selects the maximum tolerated dose (MTD) after the drug-combination trial is
completed.
}
\details{
The Keyboard design starts by specifying a target toxicity interval
(referred to as the "target key") such that any dose with a toxicity
probability within that interval can be practically viewed as the MTD.
Based on this interval's width, the Keyboard design forms a series of
equally wide keys that span the rest of the range from 0 to 1.

Given all the observed data once the trial is completed, this function
selects the MTD based on matrix isotonic estimates of the toxicity
probability of each dose combination, selecting the dose whose estimate is
closest to the target. When there are ties, one is randomly chosen.
These (matrix) isotonic estimates are obtained with
\code{\link[Iso]{biviso}}.

For patient safety, the following dose elimination rule is evaluated after
each cohort:
if at least 3 patients have been treated at the given dose and the
observed data indicate that there is more than a 95\% chance that the current
dose is above the MTD, then we eliminate this dose and beyond from the trial to
prevent exposing future patients to these overly toxic doses. The
probability threshold for elimination can be specified with
\code{cutoff.eli}. When a dose is eliminated, the design recommends the next
lower dose for treating the next patient. If the lowest dose combination
(1, 1) is overly toxic, then the trial terminates early and no dose is selected
as the MTD.

For some applications, investigators may prefer a stricter stopping rule
for extra safety when the lowest dose is overly toxic. This can be achieved
by setting \code{extrasafe=TRUE}, which imposes the following stricter
safety stopping rule:\cr
Stop the trial if \cr
(i) the number of patients treated at the lowest dose \eqn{\ge 3}, and \cr
(ii) \deqn{Pr[(toxicity rate of the lowest dose > target) | data)]
           > cutoff.eli - offset}
As a tradeoff, the strong stopping rule will decrease the MTD selection
percentage when the lowest dose is the true MTD.
}
\note{
The MTD selection and dose escalation/de-escalation rules are two
  independent components of the trial design. When appropriate, another dose
  selection procedure (e.g., one based on a fitted logistic model) can be used
  to select the MTD after completing the trial using the Keyboard design.
}
\section{Uses}{

This function uses \code{\link[Iso]{biviso}}.
}

\examples{
### Drug-combination trial ###

## Select the MTD based on the data from a 3 x 5 combination trial
n <- matrix(c(3, 5, 0, 0, 0,
              7, 6, 15, 0, 0,
              0, 0, 4, 0, 0), ncol=5, byrow=TRUE)
y <- matrix(c(0, 1, 0, 0, 0,
              1, 1, 4, 0, 0,
              0, 0, 2, 0, 0), ncol=5, byrow=TRUE)

sel.comb <- select.mtd.comb.kb(target=0.3, npts=n, ntox=y)

summary.kb(sel.comb)
plot.kb(sel.comb)

}
\references{


Yan F, Mandrekar SJ, Yuan Y. Keyboard: A Novel Bayesian Toxicity Probability
Interval Design for Phase I Clinical Trials.
\emph{Clinical Cancer Research}. 2017; 23:3994-4003.
http://clincancerres.aacrjournals.org/content/23/15/3994.full-text.pdf

Pan H, Lin R, Yuan Y. Keyboard design for phase I drug-combination trials.
\emph{Contemporary Clinical Trials}. 2020.
https://doi.org/10.1016/j.cct.2020.105972
}
\seealso{
Other drug-combination functions: 
\code{\link{get.boundary.comb.kb}()},
\code{\link{get.oc.comb.kb}()},
\code{\link{next.comb.kb}()}
}
\author{
Hongying Sun, Li Tang, and Haitao Pan
}
\concept{drug-combination functions}
