\name{LDheatmap}
\alias{LDheatmap}
\title{ This function produces a pairwise LD plot. }
\description{ \code{LDheatmap()} is used to produce a graphical display,
  as a heat map, of pairwise linkage disequilibrium (LD) measurements for SNPs.
  The heat map is a false color image in the upper-left diagonal of a square plot.
     Optionally, a line parallel to the diagonal of the image indicating
     the physical or genetic map positions of the SNPs may be added, along
     with text reporting the total length of the genomic region considered.
}
\usage{
LDheatmap(gdat, genetic.distances=NULL, distances="physical",
	LDmeasure="r", title="Pairwise LD", add.map=TRUE, add.key=TRUE,
	geneMapLocation=0.15, geneMapLabelX=NULL, geneMapLabelY=NULL,
	SNP.name=NULL, color=NULL, newpage=TRUE,
	name="ldheatmap", vp.name=NULL, pop=FALSE, flip=NULL, text=FALSE)
}
\arguments{
  \item{gdat}{ SNP data: a data frame of genotype objects, a square matrix of
  	pairwise linkage disequilibrium measurements or an object of
	class \code{"LDheatmap"} (the returned object of this function).
}
  \item{genetic.distances}{ A numeric vector of map locations of the SNPs,
        in the same order as SNPs listed in \code{gdat},
  	in terms of genetic or physical distances.
	Physical distances should be in bases, genetic distances should be in
	centiMorgans (cM).
        When \code{gdat} is \emph{not} an object of class \code{LDheatmap}, the
        default is a vector that represents equi-spaced markers, 1kb (1000 bases)
        apart. When \code{gdat} \emph{is} an object of class \code{LDheatmap},
        the \code{genetic.distances} argument is taken to be the
        \code{genetic.distances} list item of \code{gdat}. }
  \item{distances}{ A character string to specify whether the provided map locations
  	are in physical or genetic distances.
	If \code{distances="physical"} (default), the text describing the total
	length of the region will be \dQuote{Physical Length:XXkb} where XX is the
	length of the region in kilobases. If \code{distances="genetic"}, the
        text will be \dQuote{Genetic Map Length:YYcM} where YY is
        the length of the region in centiMorgans.
        If \code{gdat} is an object of class LDheatmap,
        \code{distances} is taken from \code{gdat}. }
  \item{LDmeasure}{ A character string specifying the measure of LD
  	- either allelic correlation \eqn{r^2} or Lewontin's
	|D\eqn{'}|; default = \code{"r"} for \eqn{r^2};
	type \code{"D'"} for |D\eqn{'}|.  This argument is ignored when the user has already
	supplied calculated LD measurements through \code{gdat} (i.e., when \code{gdat}
	is a matrix of pairwise LD measurements or an object of class \code{"LDheatmap"}).}
  \item{title}{ A character string for the main title of the plot.
        Default is \dQuote{Pairwise LD}. }
  \item{add.map}{ If \code{TRUE} (default), a diagonal line indicating the physical
    	or genetic map positions of the SNPs will be added to the plot,
    	along with text indicating the total length of the genetic region. }
  \item{add.key}{ If \code{TRUE} (default) the color legend is drawn. }
  \item{geneMapLocation}{ A numeric value specifying the position of the line
    	parallel to the diagonal of the matrix; the larger the value, the
    	farther it lies from the matrix diagonal. Ignored when \code{add.map=FALSE}.}
  \item{geneMapLabelX}{ A numeric value specifying the x-coordinate
    	of the text indicating the total length of the genomic region
    	being considered. Ignored when \code{add.map=FALSE}.}
  \item{geneMapLabelY}{ A numeric value specifying the y-coordinate
    	of the text indicating the total length of the genomic region
    	being considered. Ignored when \code{add.map=FALSE}.}
  \item{SNP.name}{ A vector of character string(s) of SNP name(s) to
        be labelled. Should match the names of SNPs in the provided object
        \code{gdat}, otherwise nothing is done. }
  \item{color}{ A range of colors to be used for drawing the heat map. Default
is \code{grey.colors(20)}.}
  \item{newpage}{ If \code{TRUE} (default), the heat map will be drawn on a new page. }
  \item{name}{ A character string specifying the name of the LDheatmap
               graphical object (\code{grob}) to be produced. }
  \item{vp.name}{ A character string specifying the name of the viewport
where the heat map is going to be drawn. }
  \item{pop}{ If \code{TRUE}, the viewport where the heat map is drawn is
\code{pop}ped (i.e. removed) from the viewport tree after drawing. Default=\code{FALSE}. }
  \item{flip}{ If \code{TRUE}, the LDheatmap plot is rotated 270 degrees. Default is \code{FALSE}.}
  \item{text}{ If \code{TRUE}, the LD measurements are printed on each cell. }
} 
\details{
For \code{gdat}, any square matrix with values between 0 and 1
inclusive can be passed, and the values above the diagonal will be plotted.
In the display of LD, SNPs appear in the order supplied by the 
user as the horizontal and vertical coordinates are increased 
and one moves along the off-diagonal line, from the bottom-left
to the top-right corner.  To achieve this, the conventions of 
the \code{image()} function have been adopted, in which horizontal
coordinates correspond to the rows of the matrix and vertical coordinates 
correspond to columns, and vertical coordinates are indexed in increasing 
order from bottom to top.

For the argument \code{color}, an appropriate
color palette for quantitative data is recommended, 
as outlined in the help page of 
the \code{\link[RColorBrewer:ColorBrewer]{brewer.pal}()} function of 
the 
\pkg{RColorBrewer} package.

See the package vignette, of the same name as the package, for more 
details.
}
\value{An object of class \code{"LDheatmap"} which contains the
following components:
\item{LDmatrix}{ The matrix of pairwise LD measurements plotted in the heat map. }
\item{LDheatmapGrob}{ A grid graphical object (grob) representing the produced heat map. }
\item{heatmapVP}{ The viewport in which the heat map is drawn. See
\link[grid:viewport]{viewport}.}
\item{genetic.distances}{The vector of the supplied physical or
genetic map locations, or the vector of equispaced marker distances
when no distance vector is supplied.}
\item{distances}{ A character string specifying whether the provided map
distances are physical or genetic. }
\item{color}{ The range of colors used for drawing the heat map. }

The \code{grob} \code{LDheatmapGrob} has three \code{grob}s as its children (components).
They are listed below along with their own children and respectively represent
the color image with main title, genetic map and color key of the heat map:

\code{"heatMap"} - \code{"heatmap"}, \code{"title"};

\code{"geneMap"} - \code{"diagonal"}, \code{"segments"},

\code{"title"}, \code{"symbols"}, \code{"SNPnames"}; and

\code{"Key"} - \code{"colorKey"}, \code{"title"}, \code{"labels"},

\code{"ticks"}, \code{"box"}.
}

\references{  Shin J-H, Blay S, McNeney B and Graham J (2006). LDheatmap: 
	      An R Function for Graphical Display of Pairwise Linkage 
	      Disequilibria Between Single Nucleotide Polymorphisms. 
	      Journal of Statistical Software, \bold{16} Code Snippet 3
	   }
 
\note{  The produced heat map can be modified in two ways.
	First, it is possible to edit \emph{interactively} the grob components of the heat map,
	by using the function \code{\link[grid:grid.edit]{grid.edit}}; 
the function will not work if there is no
	open graphical device showing the heat map.
	Alternatively, the user can use the function 
\code{\link[grid:grid.edit]{editGrob}} and work with
	the grob \code{LDheatmapGrob} returned by \code{LDheatmap}.
	See Examples for usage.
	
	\code{LDheatmap()} uses \code{\link[grid:Grid]{Grid}}, which 
does not respond to \code{par()} settings.
Hence modifying \code{par()} settings of \code{mfrow} and \code{mfcol} 
will not work with \code{LDheatmap()}. The Examples section shows how to
	display multiple heat maps on one plot without the use
of \code{par()}.
	}

\author{ Ji-hyung Shin <shin@sfu.ca>, Sigal Blay <sblay@sfu.ca>,  Nicholas
Lewin-Koh <nikko@hailmail.net>, Brad McNeney <mcneney@stat.sfu.ca>, Jinko
Graham <jgraham@cs.sfu.ca> }
\seealso{ \code{\link[genetics]{LD}}, \code{\link[genetics]{genotype}},
\code{\link[grid:Grid]{Grid}}, \code{\link{LDheatmap.highlight}},
\code{\link{LDheatmap.marks}} }
\examples{
#Load the package's data set
data(CEUData)
#Creates a data frame "CEUSNP" of genotype data and a vector "CEUDist"
#of physical locations of the SNPs

	#### Produce a heat map in a grey color scheme ####

MyHeatmap <- LDheatmap(CEUSNP, genetic.distances = CEUDist,
  color = grey.colors(20))

# Prompt the user before starting a new page of graphics output
# and save the original prompt settings in old.prompt.
old.prompt <- devAskNewPage(ask = TRUE)

# Highlight a certain LD block of interest:
LDheatmap.highlight(MyHeatmap, i = 3, j = 8, col = "black", fill = "grey" )
# Plot a symbol in the center of the pixel which represents LD between
# the fourth and seventh SNPs:
LDheatmap.marks(MyHeatmap,  4,  7,  gp=gpar(cex=2),  pch = "*")


        #### Use an RGB pallete for the color scheme ####
rgb.palette <- colorRampPalette(rev(c("blue", "orange", "red")), space = "rgb")
LDheatmap(MyHeatmap, color=rgb.palette(18))


	#### Modify the plot by using 'grid.edit' function ####
#Draw a heat map where the SNPs "rs2283092" and "rs6979287" are labelled.
LDheatmap(MyHeatmap, SNP.name = c("rs2283092", "rs6979287"))

#Find the names of the top-level graphical objects (grobs) on the current display
getNames()
#[1] "ldheatmap"

# Find the names of the component grobs of "ldheatmap"
childNames(grid.get("ldheatmap"))
#[1] "heatMap" "geneMap" "Key"

#Find the names of the component grobs of heatMap
childNames(grid.get("heatMap"))
#[1] "heatmap" "title"

#Find the names of the component grobs of geneMap
childNames(grid.get("geneMap"))
#[1] "diagonal" "segments" "title"    "symbols"  "SNPnames"

#Find the names of the component grobs of Key
childNames(grid.get("Key"))
#[1] "colorKey" "title"    "labels"   "ticks"    "box"

#Change the plotting symbols that identify SNPs rs2283092 and rs6979287
#on the plot to bullets
grid.edit("symbols", pch = 20, gp = gpar(cex = 1))

#Change the color of the main title
grid.edit(gPath("ldheatmap", "heatMap", "title"), gp = gpar(col = "red"))

#Change size of SNP labels
grid.edit(gPath("ldheatmap", "geneMap","SNPnames"), gp = gpar(cex=1.5))

#Add a grid of white lines to the plot to separate pairwise LD measures
grid.edit(gPath("ldheatmap", "heatMap", "heatmap"), gp = gpar(col = "white",
    lwd = 2))


	#### Modify a heat map using 'editGrob' function ####
MyHeatmap <- LDheatmap(MyHeatmap, color = grey.colors(20))

new.grob <- editGrob(MyHeatmap$LDheatmapGrob, gPath("geneMap", "segments"),
	gp=gpar(col="orange"))

 ##Clear the old graphics object from the display before drawing the modified heat map:
grid.newpage()

grid.draw(new.grob)
# now the colour of line segments connecting the SNP
# positions to the LD heat map has been changed from black to orange.


	#### Draw a resized heat map (in a 'blue-to-red' color scale ####
grid.newpage()

pushViewport(viewport(width=0.5, height=0.5))
LDheatmap(MyHeatmap, SNP.name = c("rs2283092", "rs6979287"), newpage=FALSE,
color="blueToRed")
popViewport()


	#### Draw and modify two heat maps on one plot ####
grid.newpage()

  ##Draw and the first heat map on the left half of the graphics device
pushViewport(viewport(x=0, width=0.5, just="left"))
LD1<-LDheatmap(MyHeatmap, color=grey.colors(20), newpage=FALSE,
	title="Pairwise LD in grey.colors(20)",
	SNP.name="rs6979572", geneMapLabelX=0.6,
	geneMapLabelY=0.4, name="ld1")
 upViewport()

 ##Draw the second heat map on the right half of the graphics device
pushViewport(viewport(x=1,width=0.5,just="right"))
LD2<-LDheatmap(MyHeatmap, newpage=FALSE, title="Pairwise LD in heat.colors(20)",
	SNP.name="rs6979572", geneMapLabelX=0.6, geneMapLabelY=0.4, name="ld2")
upViewport()

 ##Modify the text size of main title of the first heat map.
grid.edit(gPath("ld1", "heatMap","title"), gp=gpar(cex=1.5))

 ##Modify the text size and color of the SNP label of the second heat map.
 grid.edit(gPath("ld2", "geneMap","SNPnames"), gp=gpar(cex=1.5, col="DarkRed"))

	#### Draw a lattice-like plot with heat maps in panels ####
# Load CHBJPTSNP and CHBJPTDist
data(CHBJPTData)
# Make a variable which indicates Chinese vs. Japanese
pop <- factor(c(rep("chinese",45), rep("japanese",45)))
require(lattice)

xyplot(1:nrow(CHBJPTSNP) ~ 1:nrow(CHBJPTSNP) | pop,
type="n", scales=list(draw=FALSE), xlab="", ylab="",
panel=function(x, y, subscripts,...) {
          LDheatmap(CHBJPTSNP[subscripts,], CHBJPTDist, newpage=FALSE) })

data(GIMAP5)
require(lattice)
n<-nrow(GIMAP5$snp.data)
xyplot(1:n ~ 1:n | GIMAP5$subject.support$pop,
type="n", scales=list(draw=FALSE), xlab="", ylab="",
panel=function(x, y, subscripts,...) {
          LDheatmap(GIMAP5$snp.data[subscripts,], 
          GIMAP5$snp.support$Position, SNP.name="rs6598", newpage=FALSE) })



#Reset the user's setting for prompting on the graphics output
#to the original value before running these example commands.
 devAskNewPage(old.prompt)
}
\keyword{ hplot }
