% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.R
\name{residuals.lmm}
\alias{residuals.lmm}
\title{Extract The Residuals From a Linear Mixed Model}
\usage{
\method{residuals}{lmm}(
  object,
  type = "response",
  format = "long",
  data = NULL,
  p = NULL,
  keep.data = FALSE,
  var = NULL,
  plot = "none",
  engine.qqplot = "ggplot2",
  add.smooth = TRUE,
  digits.cor = 2,
  size.text = 16,
  scales = "free",
  ...
)
}
\arguments{
\item{object}{a \code{lmm} object.}

\item{type}{[character] type of residual to output: raw residuals (\code{"response"}), Pearson residuals (\code{"pearson"}), normalized residuals (\code{"normalized"}, scaled residual \code{"scaled"}), or partial residuals (\code{"partial"} or \code{"partial-center"}). Can also be \code{"all"} to output all except partial residuals. See detail section.}

\item{format}{[character] Should the residuals be output relative as a vector (\code{"long"}), or as a matrix with in row the clusters and in columns the outcomes (\code{"wide"}).}

\item{data}{[data.frame] dataset relative to which the residuals should be computed. Only relevant if differs from the dataset used to fit the model.}

\item{p}{[numeric vector] value of the model coefficients at which to evaluate the residuals. Only relevant if differs from the fitted values.}

\item{keep.data}{[logical] Should the argument \code{data} be output along side the residuals? Only possible in the long format.}

\item{var}{[character vector] name of the variable relative to which the partial residuals should be computed.}

\item{plot}{[character] Should a qqplot (\code{"qqplot"}), or a heatmap of the correlation between residuals  (\code{"correlation"}, require wide format), or a plot of residuals along the fitted values (\code{"scatterplot"}, require long format) be displayed?}

\item{engine.qqplot}{[character] Should ggplot2 or qqtest be used to display quantile-quantile plots? Only used when argument \code{plot} is \code{"qqplot"}.}

\item{add.smooth}{[logical] should a local smoother be used to display the mean of the residual values across the fitted values. Only relevant for \code{plot="scatterplot"}.}

\item{digits.cor}{[integer, >0] Number of digit used to display the correlation coefficients? No correlation coefficient is displayed when set to 0. Only used when argument \code{plot} is \code{"correlation"}.}

\item{size.text}{[numeric, >0] Size of the font used to displayed text when using ggplot2.}

\item{scales}{[character] Passed to \code{ggplot2::facet_wrap}.}

\item{...}{Not used. For compatibility with the generic method.}
}
\value{
When argument format is \code{"long"} and type.oobject is \code{"lmm"}, a vector containing the value of the residual realtive to each observation.
It is a matrix if the argument \code{type} contains several values.
When argument format is \code{"wide"} and type.oobject is \code{"lmm"}, a data.frame with the value of the residual relative to each cluster (in rows) at each timepoint (in columns).
}
\description{
Extract or compute the residuals of a linear mixed model.
}
\details{
The argument \code{type} defines how the residuals are computed:
\itemize{
\item \code{"fitted"}: fitted value \eqn{X_{ij} \hat{\beta}}.
\item \code{"response"}: raw residual, i.e. observed outcome minus fitted value \eqn{\varepsilon_{ij} = Y_{ij} - X_{ij} \hat{\beta}}.
\item \code{"pearson"}: each raw residual is divided by its modeled standard deviation \eqn{\varepsilon_{ij} = \frac{Y_{ij} - X_{ij} \hat{\beta}}{\sqrt{\hat{\omega}_{ij}}}}.
\item \code{"studentized"}: same as \code{"pearson"} but excluding the contribution of the cluster in the modeled standard deviation  \eqn{\varepsilon_{ij} = \frac{Y_{ij} - X_{ij} \hat{\beta}}{\sqrt{\hat{\omega}_{ij}-\hat{q}_{ij}}}}.
\item \code{"normalized"}: raw residuals are multiplied, within clusters, by the inverse of the (upper) Cholesky factor of the modeled residual variance covariance matrix \eqn{\varepsilon_{ij} = ( Y_{i} - X_{i} \hat{\beta} )\hat{C}^{-1}}.
\item \code{"normalized2"}: same as \code{"normalized"} but excluding the contribution of the cluster in the modeled residual variance covariance matrix \eqn{\varepsilon_{ij} = ( Y_{i} - X_{i} \hat{\beta} ) \hat{D}_i^{-1}}.
\item \code{"scaled"}: scaled residuals (see PROC MIXED in SAS). Numerically identical to \code{"normalized"} but computed by sequentially scaling and centering the residuals, to make them conditionally independent of previous residuals from the same cluster at previous repetitions.
\item \code{"partial"}: partial residuals (\eqn{\gamma E + \hat{\varepsilon}}). A reference level can be also be specified via the attribute \code{"reference"} to change the absolute level of the partial residuals.
\code{"partial-center"}: partial residuals with centered covariates (\eqn{\gamma E + \hat{\varepsilon}} where \eqn{E} has been centered, i.e., has 0-mean)
}
where
\itemize{
\item \eqn{X=(E,W)} the design matrix. For partial residuals, it is split according to the variable(s) in argument \code{var} (\eqn{E}) and the rest (\eqn{W}).
\item \eqn{Y} the outcome
\item \eqn{\hat{\beta}=(\hat{\gamma},\hat{\delta})} the estimated mean coefficients relative to \eqn{X=(E,W)}
\item \eqn{\hat{\Omega}} the modeled variance-covariance of the residuals and \eqn{\hat{\omega}} its diagonal elements
\item \eqn{\hat{C}} the upper Cholesky factor of \eqn{\hat{\Omega}}, i.e. upper triangular matrix satisfying \eqn{\hat{C}^{t} \hat{C} = \hat{\Omega}}
\item \eqn{\hat{Q}_i= X_i (X^{t}\hat{\Omega}X)^{-1}X_i^{t}} a cluster specific correction factor, approximating the contribution of cluster i to \eqn{\hat{\Omega}}. Its diagonal elements are denoted \eqn{\hat{q}_i}.
\item \eqn{\hat{D}_i} the upper Cholesky factor of \eqn{\hat{\Omega}-\hat{Q}_i}
}
}
\examples{
#### simulate data in the long format ####
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")

#### Linear Model ####
e.lm <- lmm(Y ~ visit + X1 + X2 + X5, data = dL)

## partial residuals
residuals(e.lm, type = "partial", var = "X1")
## residuals(e.lm) + dL$X1 * coef(e.lm)["X1"]
residuals(e.lm, type = "partial", var = "X1", keep.data = TRUE)

## partial residuals
type <- "partial"
attr(type,"reference") <- data.frame(visit=factor(2,1:3),X2=0,X5=0)
residuals(e.lm, type = type, var = "X1")
## residuals(e.lm) + dL$X1 * coef(e.lm)["X1"] + coef(e.lm)["visit2"]

## partial residuals with centered covariates
residuals(e.lm, type = "partial-center", var = "X1")
## residuals(e.lm) + (dL$X1-mean(dL$X1)) * coef(e.lm)["X1"]

#### Linear Mixed Model ####
eUN.lmm <- lmm(Y ~ visit + X1 + X2 + X5 + X6,
               repetition = ~visit|id, structure = "UN", data = dL)

## residuals
residuals(eUN.lmm, format = "long", type = c("normalized","pearson"))
residuals(eUN.lmm, format = "wide", plot = "correlation")
residuals(eUN.lmm, format = "wide", type = "normalized")
residuals(eUN.lmm, format = "wide", type = "scaled")

## residuals and predicted values
residuals(eUN.lmm, type = "all")
residuals(eUN.lmm, type = "all", keep.data = TRUE)

## partial residuals
residuals(eUN.lmm, type = "partial", var = c("(Intercept)","X6"), plot = "scatterplot")
residuals(eUN.lmm, type = "partial", var = c("X6"), plot = "scatterplot")
}
