% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pmt.R
\name{pmt}
\alias{pmt}
\alias{pmts}
\alias{define_pmt}
\title{Syntactic Sugar for Object Construction}
\usage{
pmt(key, ...)

pmts(
  which = c("all", "onesample", "twosample", "ksample", "multcomp", "paired", "rcbd",
    "association", "table")
)

define_pmt(
  inherit = c("twosample", "ksample", "paired", "rcbd", "association", "table"),
  statistic,
  rejection = c("lr", "l", "r"),
  scoring = "none",
  n_permu = 10000,
  name = "User-Defined Permutation Test",
  alternative = NULL,
  depends = character(),
  plugins = character(),
  includes = character()
)
}
\arguments{
\item{key}{a character string specifying the test. Check \code{pmts()} for valid keys.}

\item{...}{extra parameters passed to the constructor.}

\item{which}{a character string specifying the desired tests.}

\item{inherit}{a character string specifying the type of permutation test.}

\item{statistic}{definition of the test statistic. See Details.}

\item{rejection}{a character string specifying where the rejection region is.}

\item{scoring}{one of:
- a character string in \code{c("none", "rank", "vw", "expon")} specifying the scoring system
- a function that takes a numeric vector and returns an equal-length score vector}

\item{n_permu}{an integer indicating number of permutations for the permutation distribution. If set to \code{0}, all permutations will be used.}

\item{name}{a character string specifying the name of the test.}

\item{alternative}{a character string describing the alternative hypothesis.}

\item{depends, plugins, includes}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}
}
\value{
a test object corresponding to the specified key.

a data frame containing keys and corresponding tests implemented in this package.

a test object based on the specified statistic.
}
\description{
Construct test objects in a unified way.
}
\details{
The test statistic can be defined using either \code{R} or \code{Rcpp}, with the \code{statistic} parameter specified as:
\itemize{
\item \code{R}: a function returning a closure that returns a double.
\item \code{Rcpp}: a character string defining a captureless lambda (since C++11) returning another lambda that captures by value, accepts parameters of the same type as const references, and returns a double.
}

When using \code{Rcpp}, the parameters for different \code{inherit} are listed as follows. Note that the parameter names are for illustration only.
\itemize{
\item \code{"twosample"}: \verb{(Rcpp::NumericVector sample_1, Rcpp::NumericVector sample_2)}
\item \code{"ksample"}: \verb{(Rcpp::NumericVector combined_sample, Rcpp::IntegerVector one_based_group_index)}
\item \code{"paired"}: \verb{(Rcpp::NumericVector sample_1, Rcpp::NumericVector sample_2)}
\item \code{"rcbd"}: \verb{(Rcpp::NumericMatrix block_as_column_data)}
\item \code{"association"}: \verb{(Rcpp::NumericVector sample_1, Rcpp::NumericVector sample_2)}
\item \code{"table"}: \verb{(Rcpp::IntegerMatrix contingency_table)}
}

Defining the test statistic using \code{R} follows a similar approach. The purpose of this design is to pre-calculate certain constants that remain invariant during permutation.
}
\examples{
pmt("twosample.wilcoxon")

pmts("ksample")

x <- rnorm(5)
y <- rnorm(5, 1)

t <- define_pmt(
    inherit = "twosample",
    scoring = base::rank, # equivalent to "rank"
    statistic = function(...) function(x, y) sum(x)
)$test(x, y)$print()

t$scoring <- function(x) qnorm(rank(x) / (length(x) + 1)) # equivalent to "vw"
t$print()

t$n_permu <- 0
t$print()

\donttest{
r <- define_pmt(
    inherit = "twosample",
    n_permu = 1e5,
    statistic = function(x, y) {
        m <- length(x)
        n <- length(y)
        function(x, y) sum(x) / m - sum(y) / n
    }
)

rcpp <- define_pmt(
    inherit = "twosample",
    n_permu = 1e5,
    statistic = "[](NumericVector x, NumericVector y) {
        R_len_t m = x.size();
        R_len_t n = y.size();
        return [=](const NumericVector& x, const NumericVector& y) -> double {
            return sum(x) / m - sum(y) / n;
        };
    }"
)

options(LearnNonparam.pmt_progress = FALSE)
system.time(r$test(x, y))
system.time(rcpp$test(x, y))
}

}
