######################################
#### Functions for class("mbnma") ####
######################################

## quiets concerns of R CMD check re: the .'s that appear in pipelines
if(getRversion() >= "2.15.1")  utils::globalVariables(c(".", "studyID", "agent", "dose", "Var1", "value",
                                                        "Parameter", "fupdose", "groupvar", "y",
                                                        "network", "a", "param", "med", "l95", "u95", "value",
                                                        "Estimate", "2.5%", "50%", "97.5%", "treatment"))

#' Forest plot for results from dose-response MBNMA models
#'
#' Generates a forest plot for dose-response parameters.
#'
#' @param x An S3 object of class `"mbnma"` generated by running
#'   a dose-response MBNMA model
#' @param params A character vector of dose-response parameters to plot.
#' Parameters must be given the same name as monitored nodes in `mbnma` and must be
#' modelled as relative effects (`"rel"`). Can be set to
#' `NULL` to include all available dose-response parameters estimated by `mbnma`.
#' @param agent.labs A character vector of agent labels (including `"Placebo"` if it
#' has been included in the original network). If left as `NULL` (the default) then
#' labels will be used as defined in the data.
#' @param class.labs A character vector of class labels if `mbnma` was modelled using class effects
#' (including `"Placebo"` if it
#' has been included in the original network). If left as `NULL`
#' (the default) then labels will be used as defined in the data.
#' @param ... Arguments to be passed to methods, such as graphical parameters
#'
#' @return A forest plot of class `c("gg", "ggplot")` that has separate panels for
#' different dose-response parameters. Results are plotted on the link scale.
#'
#' @examples
#' \donttest{
#' # Using the triptans data
#' network <- mbnma.network(HF2PPITT)
#'
#' # Run an exponential dose-response MBNMA and generate the forest plot
#' exponential <- mbnma.run(network, fun="exponential")
#' plot(exponential)
#'
#' # Plot only Emax parameters from an Emax dose-response MBNMA
#' emax <- mbnma.emax(network, emax="rel", ed50="rel", method="random")
#' plot(emax, params=c("d.emax"))
#'
#'
#' #### Forest plots including class effects ####
#' # Generate some classes for the data
#' class.df <- HF2PPITT
#' class.df$class <- ifelse(class.df$agent=="placebo", "placebo", "active1")
#' class.df$class <- ifelse(class.df$agent=="eletriptan", "active2", class.df$class)
#' netclass <- mbnma.network(class.df)
#' emax <- mbnma.emax(netclass, emax="rel", ed50="rel", method="random",
#'             class.effect=list("ed50"="common"))
#'
#' # Plot forest plot with different labels for classes
#' plot(emax, class.labs=c("Placebo", "Other Active", "Eletriptan"))
#'
#' # Since "Placebo" is included in the network, it must be included in labels
#' # Failure to do so will cause an error
#' ## ERROR ## plot(emax, class.labs=c("Other Active", "Eletriptan"))
#' }
#'
#' @export
plot.mbnma <- function(x, params=NULL, agent.labs=NULL, class.labs=NULL, ...) {

  # Run checks
  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertClass(x, "mbnma", add=argcheck)
  checkmate::assertCharacter(params, null.ok=TRUE, add=argcheck)
  checkmate::assertCharacter(agent.labs, null.ok=TRUE, add=argcheck)
  checkmate::assertCharacter(class.labs, null.ok=TRUE, add=argcheck)
  checkmate::reportAssertions(argcheck)

  # Check that specified params are monitored in model
  if (!all(params %in% x[["parameters.to.save"]])) {
    stop(paste0("Variable 'params': Must contain elements of set {", paste(x[["parameters.to.save"]], collapse = ", "), "}"))
  }

  # Check that specified params are modelled using relative effects
  for (i in seq_along(params)) {
    if (!(grepl("d\\.", params[i]) | grepl("D\\.", params[i]))) {
      stop(paste0(params[i], " has not been modelled using relative effects and does not vary by agent or class"))
    }
  }

  # Add all available params if is.null(params)
  if (is.null(params)) {
    params <- vector()

    # Add d
    params <- append(params,
                     x[["parameters.to.save"]][grep("^d\\.", x[["parameters.to.save"]])]
    )

    # Add D
    params <- append(params,
                     x[["parameters.to.save"]][grep("^D\\.", x[["parameters.to.save"]])]
    )

    if (length(params)==0) {
      stop("No dose-response relative effects can be identified from the model")
    }
  }


  # Compile parameter data into one data frame
  mbnma.sum <- as.data.frame(x[["BUGSoutput"]][["summary"]])
  plotdata <- mbnma.sum[0,]
  for (i in seq_along(params)) {
    paramdata <- mbnma.sum[grepl(paste0("^", params[i]),rownames(mbnma.sum)),]
    paramdata[["doseparam"]] <- rep(params[i], nrow(paramdata))
    plotdata <- rbind(plotdata, paramdata)
  }

  if (all(grepl("^d\\.1\\[[0-9]+,[0-9]+\\]", rownames(plotdata)))) { # if nonparam function used
    # Remove dose=0 from all agents except placebo
    row <- plotdata[grepl("^d\\.1\\[1,1\\]", rownames(plotdata)),]
    plotdata <- plotdata[!grepl("^d\\.1\\[1,[0-9]+\\]", rownames(plotdata)),]
    plotdata <- rbind(row, plotdata)

    plotdata[["param"]] <- c(1:nrow(plotdata))
  } else {
    plotdata[["param"]] <- as.numeric(gsub("(.+\\[)([0-9]+)(\\])", "\\2", rownames(plotdata)))
  }

  # plotdata[["param"]] <- as.numeric(gsub("(.+\\[)([0-9]+)(\\])", "\\2", rownames(plotdata)))
  # if (any(is.na(plotdata[["param"]]))) {
  #   plotdata[["param"]] <- c(1:nrow(plotdata))
  # }

  # Change param labels for agents
  agentdat <- plotdata[grepl("^d\\.", rownames(plotdata)),]
  if (!is.null(agent.labs)) {
    agentcodes <- as.numeric(gsub("(^.+\\[)([0-9]+)(\\])", "\\2", rownames(agentdat)))
    if (length(agent.labs)!=max(agentcodes)) {
      stop("`agent.labs` length does not equal number of agents within the model")
    } else {
      a.labs <- agent.labs[sort(unique(agentcodes))]
    }
  } else if ("agents" %in% names(x$network)) {
    if (any(x$model.arg$fun %in% c("nonparam.up", "nonparam.down"))) {
      a.labs <- x$network[["treatments"]]
    } else {
      a.labs <- x$network[["agents"]][x$network[["agents"]]!="Placebo"]
    }
  } else {
    a.labs <- sort(unique(agentdat$param))
  }

  # Change param labels for classes
  classdat <- plotdata[grepl("^D\\.", rownames(plotdata)),]
  c.labs <- vector()
  if (nrow(classdat)!=0) {
    if (!is.null(class.labs)) {
      classcodes <- as.numeric(gsub("(^.+\\[)([0-9]+)(\\])", "\\2", rownames(classdat)))
      c.labs <- class.labs[classcodes]
    } else if ("classes" %in% names(x)) {
      c.labs <- x[["classes"]][x[["classes"]]!="Placebo"]
    } else {
      c.labs <- sort(unique(classdat$param))
    }
  }

  # Increase param number for classes
  nagent <- ifelse(nrow(agentdat)>0, max(agentdat$param), 0)
  plotdata$param[grepl("^D\\.", rownames(plotdata))] <-
    plotdata$param[grepl("^D\\.", rownames(plotdata))] + nagent

  # Attach labels
  if (nrow(agentdat)>0) {
    all.labs <- c(a.labs, c.labs)
  } else {all.labs <- c.labs}
  plotdata$param <- factor(plotdata$param, labels=all.labs)

  if (any(is.na(levels(plotdata$param)))) {
    stop("`agent.labs` or `class.labs` have not been specified correctly. Perhaps include `Placebo` in labels")
  }

  g <- ggplot2::ggplot(plotdata, ggplot2::aes(y=`50%`, x=param)) +
    ggplot2::geom_point() +
    ggplot2::geom_errorbar(ggplot2::aes(ymin=`2.5%`, ymax=`97.5%`)) +
    ggplot2::coord_flip()

  g <- g + ggplot2::facet_wrap(~doseparam, scales="free")

  # Axis labels
  g <- g + ggplot2::xlab("Agent / Class") +
    ggplot2::ylab("Effect size") +
    ggplot2::theme_bw()

  return(g)
}





#' Rank parameter estimates
#'
#' Only parameters that vary by agent/class can be ranked.
#'
#' @param direction Indicates whether negative responses are better (taking the
#'   value `-1`) or positive responses are better (taking the value `1`)
#' @param to.rank A numeric vector containing the codes for the agents/classes you wish to rank.
#' If left `NULL` then all agents/classes (depending on the value assigned to `level`) in
#' the model will be ranked. Included codes must be greater than
#' `2` if placebo has been modelled, since placebo cannot be included in the ranking
#' @param level Can be set to `"agent"` to rank across different agents or `"class"` to rank
#' across different classes.
#' @param params A character vector of named parameters in the model that vary by either agent
#' or class (depending on the value assigned to `level`). If left as `NULL` (the default), then
#' ranking will be calculated for all available parameters that vary by agent/class.
#' @param ... Arguments to be passed to methods
#' @inheritParams predict.mbnma
#' @inheritParams rank
#'
#' @details Ranking cannot currently be performed on non-parametric dose-response MBNMA
#'
#' @return An object of `class("mbnma.rank")` which is a list containing a summary data
#' frame, a matrix of rankings for each MCMC iteration, a matrix of probabilities
#' that each agent has a particular rank, and a matrix of cumulative ranking probabilities
#' for each agent, for each parameter that has been ranked.
#'
#' @examples
#' \donttest{
#' # Using the triptans data
#' network <- mbnma.network(HF2PPITT)
#'
#' # Rank selected agents from a linear dose-response MBNMA
#' linear <- mbnma.run(network, fun="linear")
#' ranks <- rank(linear, to.rank=c("zolmitriptan", "eletriptan", "sumatriptan"))
#' summary(ranks)
#'
#' # Rank only ED50 parameters from an Emax dose-response MBNMA
#' emax <- mbnma.emax(network, emax="rel", ed50="rel", method="random")
#' ranks <- rank(emax, params="d.ed50")
#' summary(ranks)
#'
#'
#' #### Ranking by class ####
#' # Generate some classes for the data
#' class.df <- HF2PPITT
#' class.df$class <- ifelse(class.df$agent=="placebo", "placebo", "active1")
#' class.df$class <- ifelse(class.df$agent=="eletriptan", "active2", class.df$class)
#' netclass <- mbnma.network(class.df)
#' emax <- mbnma.emax(netclass, emax="rel", ed50="rel", method="random",
#'             class.effect=list("ed50"="common"))
#'
#' # Rank by class, with negative responses being "better"
#' ranks <- rank(emax, level="class", direction=-1)
#' print(ranks)
#'
#' # Print and generate summary data frame for `mbnma.rank` object
#' summary(ranks)
#' print(ranks)
#'
#' # Plot `mbnma.rank` object
#' plot(ranks)
#' }
#'
#' @export
rank.mbnma <- function(x, params=NULL, direction=1, level="agent", to.rank=NULL, ...) {

  # Checks
  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertClass(x, classes="mbnma", add=argcheck)
  checkmate::assertCharacter(params, null.ok=TRUE, add=argcheck)
  checkmate::assertChoice(direction, choices = c(-1,1), add=argcheck)
  #checkmate::assertNumeric(to.rank, lower = 2, null.ok=TRUE, add=argcheck)
  checkmate::assertChoice(level, choices = c("agent","class"), add=argcheck)
  checkmate::reportAssertions(argcheck)

  if (any(x$model.arg$fun %in% c("nonparam.up", "nonparam.down"))) {
    stop("Ranking cannot currently be performed for non-parametric models")
  }
  if (length(x$model.arg$fun)>1) {
    stop("Ranking cannot currently be performed for models with multiple dose-response functions")
  }

  # Change agent/class to agents/classes
  levels <- ifelse(level=="agent", "agents", "classes")

  if (level=="class") {
    if (is.null(x[["model"]][["data"]]()[["class"]])) {
      stop("`level` has been set to `class` but classes have not been used in the model")
    }
    if (!is.null(to.rank)) {
      warning("Codes in `to.rank` correspond to class codes rather than treatment codes")
    }
  }

  # If treats have not been specified then select all of them - WONT WORK IF PLACEBO NOT INCLUDED
  starttrt <- ifelse(x$network$agents[1]=="Placebo", 2, 1)
  codes.mod <- c(starttrt:max(x[["model"]][["data"]]()[[level]], na.rm=TRUE))
  if (is.null(to.rank)) {
    to.rank <- codes.mod
  } else if (is.numeric(to.rank)) {
    if (!all(to.rank %in% seq(1:max(x[["model"]][["data"]]()[[level]], na.rm=TRUE)))) {
      stop("`to.rank` codes must match those in the dataset for either `agent` or `class`")
    }
  } else if (is.character(to.rank)) {
    if (!all(to.rank %in% x$network[[levels]])) {
      stop("`to.rank` agent/class names must match those in the network for either `agent` or `class`")
    }
    to.rank <- as.numeric(factor(to.rank, levels=x$network[[levels]]))
  }

  if (x$network$agents[1]=="Placebo") {
    to.rank <- to.rank-1
    if (any(to.rank==0)) {
      warning("Placebo (d[1] or D[1]) cannot be included in the ranking for relative effects and will therefore be excluded")
      to.rank <- to.rank[to.rank!=0]
    }
    agents <- x$network[[levels]][to.rank+1]
  } else {
    agents <- x$network[[levels]][to.rank]
  }


  if (direction==-1) {
    decreasing <- FALSE
  } else if (direction==1) {
    decreasing <- TRUE
  } else {stop("`direction` must be either -1 or 1 for ranking")}

  if (is.null(params)) {
    for (i in seq_along(x$BUGSoutput$root.short)) {
      if (length(x$BUGSoutput$long.short[i][[1]])==length(codes.mod)) {
        params <- append(params, x$BUGSoutput$root.short[i])
      }
    }
  } else {
    for (i in seq_along(params)) {
      if (!(params[i] %in% x[["parameters.to.save"]])) {
        stop(paste0(params[i], " has not been monitored by the model. `params` can only include model parameters that have been monitored and vary by agent/class"))
      }
    }
  }

  rank.result <- list()
  for (i in seq_along(params)) {
    if (params[i] %in% x[["parameters.to.save"]]) {
      param.mod <- x[["BUGSoutput"]][["sims.list"]][[params[i]]]

      # Check that selected parameter is different over multiple treatments
      if (!is.matrix(param.mod) | ncol(param.mod)!=length(codes.mod)) {
        msg <- paste0(params[i], " does not vary by ", level, " and therefore cannot be ranked")
        stop(msg)
      }

      param.mod <- param.mod[,to.rank]
      rank.mat <- t(apply(param.mod, MARGIN=1, FUN=function(x) {
        order(order(x, decreasing = decreasing), decreasing=FALSE)
      }))
      #colnames(rank.mat) <- to.rank
      colnames(rank.mat) <- agents

      # Calculate ranking probabilities
      prob.mat <- calcprob(rank.mat, treats=agents)

      # Calculate cumulative ranking probabilities
      cum.mat <- apply(prob.mat, MARGIN=2,
                       FUN=function(col) {cumsum(col)})

      rank.result[[params[i]]] <-
        list("summary"=sumrank(rank.mat),
             #"prob.matrix"=calcprob(rank.mat, treats=to.rank),
             "prob.matrix"=prob.mat,
             "rank.matrix"=rank.mat,
             "cum.matrix"=cum.mat,
             "direction"=direction)

    }
  }
  class(rank.result) <- "mbnma.rank"

  if (length(rank.result)==0) {
    stop(paste0("There are no parameters saved in the model that vary by ", level))
  }

  return(rank.result)
}




#' Predict responses for different doses of agents in a given population based on MBNMA
#' dose-response models
#'
#' Used to predict responses for different doses of agents or to predict
#' the results of a new study. This is calculated by combining
#' relative treatment effects with a given reference treatment response
#' (specific to the population of interest).
#'
#' @param object An S3 object of class `"mbnma"` generated by running
#'   a dose-response MBNMA model
#'
#' @param max.doses A list of numbers. Each named element in the list corresponds to an
#'   agent (either named similarly to agent names given in the data, or named
#'   correspondingly to the codes for agents given in `mbnma`)
#'   and each number for that element corresponds to the maximum dose of the given agent, below which several
#'   predictions will be calculated at different doses (the number of these is determined by `n.doses`).
#'   Can only take positive values. If left as `NULL` (the default) results will be predicted based on the
#'   maximum dose of each agent given in the data.
#' @param n.doses A number indicating the number of doses at which to make predictions
#'   within each agent. The default is `15`.
#' @param exact.doses A list of numeric vectors. Each named element in the list correponds to an
#'   agent (either named similarly to agent names given in the data, or named
#'   correspondingly to the codes for agents given in `mbnma`) and each number within the vector
#'   for that element corresponds to a dose of the agent for which to predict responses.
#'   Doses can only take positive values.
#' @param E0 An object to indicate the value(s) to use for the response at dose = 0 (i.e.
#'   placebo) in the prediction. This can take a number of different formats depending
#'   on how it will be used/calculated. The default is `0` but this will typically lead
#'   to non-sensical predictions.
#'   * `numeric()` A single numeric value representing the deterministic response at dose = 0,
#'   given on the natural scale - so for binomial data, proportions should be given and
#'   for Poisson data, a rate should be given.
#'   * `character()` A single string representing a stochastic distribution for the response
#'   at dose = 0, given on the natural scale - so for binomial data, proportions should be given and
#'   for Poisson data, a rate should be given. This is specified as a random number generator
#'   (RNG) given as a string, and can take any RNG distribution for which a function exists
#'   in R. For example: `"rnorm(n, 7, 0.5)"`.
#'   * `data.frame()` A data frame containing data in the long format (one row per study arm) to be meta-analysed
#'   to estimate the dose = 0 (placebo) response. This could be a set of observational
#'   studies that are specific to the population on which to make
#'   predictions, or it can be a subset of the study arms within the MBNMA dataset
#'   that investigate placebo. See [ref.synth()]
#' @param synth A character object that can take the value `"fixed"` or `"random"` to
#'   specify the the type of pooling to use for synthesis of `E0` if a data frame
#'   has been provided for it. Using `"random"` rather
#'   than `"fixed"` for `synth` will result in wider 95\\% CrI for predictions.
#' @param ... Arguments to be sent to [R2jags::jags()] for synthesis of the network
#'   reference treatment effect (using [ref.synth()])
#'
#'
#' @return An S3 object of class `mbnma.predict` that contains the following
#'   elements:
#'
#' * `summary` A named list of data frames. Each data frame contains
#'   a summary of predicted responses at follow-up times specified in `times`
#'   for each treatment specified in `treats`
#'
#' * `pred.mat` A named list of
#'   matrices. Each matrix contains the MCMC results of predicted responses at
#'   follow-up times specified in `times` for each treatment specified in
#'   `treats`
#'
#' @details
#' The range of doses on which to make predictions can be specified in one of two ways:
#'
#' 1. Use `max.dose` and `n.doses` to specify the maximum dose for each agent and the
#' number of doses within that agent for which to predict responses. Doses will be chosen
#' that are equally spaced from zero to the maximum dose for each agent. This is useful
#' for generating plots of predicted responses (using `[plot-mbnma.predict]`) as it will
#' lead to fitting a smooth dose-response curve (provided `n.doses` is sufficiently high).
#'
#' 2. Use `exact.doses` to specify the exact doses for which to predict responses for each
#' agent. This may be more useful when ranking different predicted responses using
#' `[rank-mbnma.predict]`
#'
#' @examples
#' \donttest{
#' # Using the triptans data
#' network <- mbnma.network(HF2PPITT)
#'
#' # Run an Emax dose-response MBNMA
#' emax <- mbnma.emax(network, emax="rel", ed50="rel", method="random")
#'
#'
#' ###########################
#' ###### Specifying E0 ######
#' ###########################
#'
#' #### Predict responses using deterministic value for E0 ####
#' # Data is binomial so we specify E0 on the natural scale as a probability
#' pred <- predict(emax, E0 = 0.2)
#'
#' # Specifying non-sensical values will return an error
#' #pred <- predict(emax, E0 = -10)
#' ### ERROR ###
#'
#' #### Predict responses using stochastic value for E0 ####
#' # Data is binomial so we might want to draw from a beta distribution
#' pred <- predict(emax, E0 = "rbeta(n, shape1=1, shape2=5)")
#'
#' # Misspecifying the RNG string will return an error
#' #pred <- predict(emax, E0 = "rbeta(shape1=1, shape2=5)")
#' ### ERROR ###
#'
#'
#' #### Predict responses using meta-analysis of dose = 0 studies ####
#'
#' # E0 is assigned a data frame of studies to synthesis
#' # Can be taken from placebo arms in triptans dataset
#' ref.df <- network$data.ab[network$data.ab$agent==1,]
#'
#' # Synthesis can be fixed/random effects
#' pred <- predict(emax, E0 = ref.df, synth="random")
#'
#'
#'
#' ######################################################################
#' #### Specifying which doses/agents for which to predict responses ####
#' ######################################################################
#'
#' # Change the number of predictions for each agent
#' pred <- predict(emax, E0 = 0.2, n.doses=20)
#' pred <- predict(emax, E0 = 0.2, n.doses=3)
#'
#' # Change the range of predicted doses to be the same for all agents
#' # But only predict responses for a subset of agents
#' pred <- predict(emax, E0 = 0.2,
#'             max.doses=list("Placebo"=0, "eletriptan"=5, "sumatriptan"=5))
#' plot(pred) # Plot predictions
#'
#' # Specify several exact combinations of doses and agents to predict
#' pred <- predict(emax, E0 = 0.2,
#'             exact.doses=list("eletriptan"=c(0:5), "sumatriptan"=c(1,3,5)))
#' plot(pred) # Plot predictions
#'
#' # Print and summarise `mbnma.predict` object
#' print(pred)
#' summary(pred)
#'
#' # Plot `mbnma.predict` object
#' plot(pred)
#' }
#'
#' @export
predict.mbnma <- function(object, n.doses=15, max.doses=NULL, exact.doses=NULL,
                          E0=0, synth="fixed",
                          ...) {
  ######## CHECKS ########

  # Run checks
  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertClass(object, "mbnma", add=argcheck)
  checkmate::assertList(max.doses, types="numeric", null.ok=TRUE, add=argcheck)
  checkmate::assertList(exact.doses, types="numeric", null.ok=TRUE, add=argcheck)
  checkmate::assertInt(n.doses, lower=1, add=argcheck)
  #checkmate::assertDataFrame(E0.estimate, null.ok=TRUE, add=argcheck)
  checkmate::assertChoice(synth, choices=c("random", "fixed"), add=argcheck)
  checkmate::reportAssertions(argcheck)

  agents <- object$model$data()$agent
  mbnma.agents <- object$network[["agents"]]

  # Checks for doses
  doses <- NULL
  if (!is.null(exact.doses) & !is.null(max.doses)) {
    warning("`exact.dose` and `max.doses` have both been specified in the arguments, yet only one of these can be used. Deferring to using argument given for `exact.doses`")
    max.doses <- NULL
  }
  if (!is.null(exact.doses)) {
    doses <- exact.doses
  } else if (!is.null(max.doses)) {
    doses <- max.doses

    for (i in seq_along(doses)) {
      doses[[i]] <- signif(seq(0,
                               max(doses[[i]]),
                               length.out=n.doses), 2)
    }

  }

  if (!is.null(doses)) {
    if (length(doses)>max(agents, na.rm=TRUE)) {
      stop("A greater number of agents have been supplied in either `max.doses` or `exact.doses` than are present in the model")
    }

    # If named elements of list are not numeric, check if they match agents in mbnma
    match.pass <- TRUE
    if (is.null(names(doses))) {
      if (length(doses)!=length(mbnma.agents)) {
        stop("If elements in `max.doses` or `exact.doses` are not named then there must be the same number of elements as there are agents in the model, so that they correspond to agent codes")
      }
      names(doses) <- mbnma.agents
      agent.num <- 1:length(mbnma.agents)
    } else if (!is.null(names(doses))) {
      if (any(is.na(suppressWarnings(as.numeric(names(doses)))))) {
        if (!all(names(doses) %in% mbnma.agents)) {
          match.pass <- FALSE
        }
        agent.num <- match(names(doses), mbnma.agents)
      } else {
        if (!all(names(doses) %in% c(1:max(agents, na.rm=TRUE)))) {
          match.pass <- FALSE
        }
        agent.num <- as.numeric(names(doses)) # Add an agent numerical identifier for included agents
      }
      if (match.pass==FALSE) {
        stop("Element names in `doses` must correspond either to agent names in data or agent codes in `object`")
      }
    }



    for (i in seq_along(doses)) {
      if (!all(doses[[i]]>=0)) {
        stop(paste0("Doses given in `doses` must be positive. Agent ", names(doses)[i], " contains negative values."))
      }
    }
  } else {
    # Automatically generate doses list for treatments included in data
    if ("rcs" %in% object$model.arg$fun) {
      dose <- as.vector(object$model$data()$spline[,,1])
    } else {
      dose <- as.vector(object$model$data()$dose)
    }

    agent <- as.vector(agents)

    df <- data.frame("agent"=agent, "dose"=dose)
    df <- unique(df[stats::complete.cases(df),])
    df <- dplyr::arrange(df, agent, dose)
    df$agent <- factor(df$agent, labels=mbnma.agents)

    doses <- list()
    for (i in seq_along(mbnma.agents)) {
      doses[[mbnma.agents[i]]] <- signif(seq(0,
                                             max(df$dose[df$agent==mbnma.agents[i]]),
                                             length.out=n.doses), 2)
    }
    agent.num <- c(1:length(mbnma.agents))
  }



  # Set model arguments
  if (length(object$model.arg$class.effect)>0) {
    stop("`predict() currently does not work with models that use class effects")
  }
  if (object$model.arg$fun[1] %in% c("nonparam.up", "nonparam.down")) {
    stop("`predict() does not work with non-parametric dose-response functions")
  }
  if (length(object$model.arg$fun)>1) {
    #stop("`predict() currently does not work with models that use multiple dose-response functions")

    funs <- c(NA, 1,1,2,3, rep(object$model.arg$knots-1, 3))
    names(funs) <- c("user", "linear", "exponential", "emax", "emax.hill", "rcs", "bs", "ns")
    funs <- funs[names(funs) %in% object$model.arg$fun]

    # Want to find the location of the agents within the vector of agent names in network
    #funi <- which(names(doses) %in% object$network$agents)
    funi <- which(object$network$agents %in% names(doses))
    X <- sapply(object$model.arg$fun[funi], function(x) which(x==names(funs)))
  }

  link <- object$model.arg$link

  n <- object$BUGSoutput$n.keep * object$BUGSoutput$n.chains

  DR <- suppressMessages(
    write.dose.fun(fun=object$model.arg$fun, user.fun=object$model.arg$user.fun,
                   effect="abs"
    )[[1]])
  DR <- gsub("(^.+<-)(.+)", "\\2", DR)

  betaparams <- get.model.vals(object)
  betas <- assignfuns(object$model.arg$fun, object$network$agents, object$model.arg$user.fun,
                      ifelse(is.null(object$model.arg$arg.fun), FALSE, TRUE))


  # Identify E0
  if (is.null(E0)) {
    stop("`E0` has not been given a value. Responses cannot be predicted without a value(s) for dose = 0 (placebo)")
  }
  if (!any(class(E0) %in% c("numeric", "character", "data.frame"))) {
    stop("`E0` can only be of type `numeric()`, `character()` or `data.frame()`")
  }

  if ((is.numeric(E0) | is.character(E0)) & length(E0)!=1) {
    stop("`E0` must take a single numeric (deterministic E0) or character (stochastic E0) value, or be provided with a data frame so that it may be estimated from the data")
  }
  if (is.character(E0)) {
    if (!grepl("^r[a-z]+\\(n,.+\\)", E0)) {
      stop("Distribution for `E0` does not match stochastic random number generator format.\nFormat must take any R random number generator function")
    }
    E0 <- eval(parse(text=E0))
  }

  if ((is.numeric(E0) | is.character(E0))) {
    E0 <- rescale.link(E0, direction="link", link=link)
  } else if (is.data.frame(E0)) {
    E0 <- ref.synth(data.ab=E0, mbnma=object, synth=synth, ...)

    if (!("sd.mu" %in% names(E0))) {
      E0 <- E0$m.mu
    } else {
      E0 <- stats::dnorm(E0$m.mu, E0$sd.mu)
    }
  }


  predict.result <- list()

  if ("rcs" %in% object$model.arg$fun) {
    splinedoses <- doses
    for (i in seq_along(doses)) {
      splinedoses[[i]] <- t(genspline(doses[[i]], knots=object$model.arg$knots,
                              max.dose=max(object$network$data.ab$dose[object$network$data.ab$agent==agent.num[i]])))
    }

  }

  for (i in seq_along(doses)) {
    predict.result[[names(doses)[i]]] <- list()
    for (k in seq_along(doses[[i]])) {
      if (names(doses)[i] %in% c("1", "Placebo") | doses[[i]][k]==0) {
        # Ensures reference agent (placebo) takes E0
        # THIS NEEDS TO BE CHANGED IF INTERCEPT IS RELAXED! sHOULD ONLY BE USED IF INTERCEPT=FALSE
        pred <- E0

      } else {
        tempDR <- gsub("\\[agent\\[i,k\\]\\]", "", DR)
        tempDR <- gsub("\\[i,k\\]", "", tempDR)
        tempDR <- gsub("(\\[i,k,)([0-9\\])", "[\\2", tempDR) # For splines

        # For multiple DR functions
        tempDR <- gsub("X==", "X[i]==", tempDR)
        #tempDR <- gsub("s\\.beta\\.", "beta\\.", tempDR)

        # Need to enclose ifelse() matrices in list() to allow ifelse to return something in matrix form
        if (length(object$model.arg$fun)>1) {
          tempDR <- gsub("(ifelse)(.*?,)(.*?,)", "\\1\\2list(\\3!!!),", tempDR)
          tempDR <- gsub(",\\!\\!\\!", "", tempDR)
          tempDR <- gsub("(.+,)(.*?)$", "\\1list(\\2)", tempDR)
        }



        dose <- doses[[i]][k]
        if ("rcs" %in% object$model.arg$fun) {
          spline <- splinedoses[[i]][,k]
        }

        for (param in seq_along(betaparams)) {
          #print(param)
          if (is.vector(betaparams[[param]]$result)) {
            assign(paste0("s.", names(betaparams)[param]),
                   betaparams[[param]]$result)
          } else if (is.matrix(betaparams[[param]]$result)) {

            # Look for correct column index for each beta param
            colnum <- which(grepl(paste0("\\[", agent.num[i], "\\]"),
                                  colnames(betaparams[[param]]$result)
            ))

            assign(paste0("s.", names(betaparams)[param]),
                   betaparams[[param]]$result[,colnum]
            )
          }
        }

        chunk <- eval(parse(text=tempDR))
        if (is.list(chunk)) {
          chunk <- chunk[[1]]
        }
        pred <- E0 + chunk
        if (length(pred)<=1) {stop()}
      }

      # Convert to natural scale using link function
      pred <- rescale.link(pred, direction="natural", link=link)

      predict.result[[names(doses)[i]]][[as.character(doses[[i]][k])]] <-
        pred
    }
  }

  output <- list("predicts"=predict.result,
                 "likelihood"=object$model.arg$likelihood, "link"=object$model.arg$link,
                 "network"=object$network)

  class(output) <- "mbnma.predict"

  return(output)
}




#' Print summary of MBNMA results to the console
#' @param object An S3 object of class `"mbnma"` generated by running
#'   a dose-response MBNMA model
#' @param ... additional arguments affecting the summary produced
#'
#' @export
summary.mbnma <- function(object, ...) {
  checkmate::assertClass(object, "mbnma")

  # State that function does not work if "parameters.to.save" has been specified
  if (!is.null(object$model.arg$parameters.to.save)) {
    stop("Cannot use `summary()` method if `parameters.to.save` have been assigned. Use `print()` instead.")
  }
  if (any(object$model.arg$fun %in% c("nonparam.up", "nonparam.down"))) {
    stop("Cannot use `summary()` method for non-parametric dose-response functions. Use `print()` instead.")
  }

  # Check for rhat < 1.02
  rhat.warning(object)

  ##### Overall section #####

  # Print title
  cat(crayon::bold("========================================\nDose-response MBNMA\n========================================\n\n"))

  # Print DR function
  if (length(object$model.arg$fun)==1) {
    cat(paste("Dose-response function:", object$model.arg$fun, sep=" "))
  } else if (length(object$model.arg$fun)>1) {
    drtab <- matrix(object$model.arg$fun, ncol=1)
    colnames(drtab) <- "Function"
    rownames(drtab) <- object$network$agents

    cat("Dose-response functions:\n\n")
    print(drtab)
  }

  if (any(object$model.arg$fun == "user")) {
    cat("\nuser.fun:", as.character(object$model.arg$user.fun))
  }



  #overall.sect <- paste(title, overall.sect, sep="\n")

  # Print method section
  #method.sect <- print.method.sect(object)
  cat(print.method.sect(object))

  # Print treatment-level section
  #treat.sect <- print.treat.str(object)
  print.treat.str(object)

  # Class-effect section
  print.class.str(object)

  # Model fit statistics section
  modfit.sect <- print.modfit.str(object)

  #output <- paste(overall.sect, treat.sect, method.sect, "\n", class.sect, "\n\n", modfit.sect, sep="")
  output <- paste("\n\n", modfit.sect, sep="")
  cat(output, ...)
}

