\name{prepareMplusData}
\alias{prepareMplusData}
\title{Create tab-delimited file and Mplus input syntax from R data.frame}
\usage{
  prepareMplusData(df, filename, keepCols, dropCols,
    inpfile = FALSE, interactive = TRUE, overwrite = TRUE)
}
\arguments{
  \item{df}{The R data.frame to be prepared for Mplus}

  \item{filename}{The path and filename for the
  tab-delimited data file for use with Mplus. Example:
  "C:/Mplusdata/data1.dat"}

  \item{keepCols}{A character vector specifying the
  variable names within \code{df} to be output to
  \code{filename} or a numeric vector of the column indices
  to be output or a logical vector corresponding to the
  same.}

  \item{dropCols}{A character vector specifying the
  variable names within \code{df} to be omitted from the
  data output to \code{filename} or a numeric vector of the
  column indices not to be output or a logical vector
  corresponding to the same.}

  \item{inpfile}{Logical value whether the Mplus syntax
  should be written to the console or to an input file.
  Defaults to \code{FALSE}. If \code{TRUE}, the file name
  will be the same as \code{filename} with the extension
  changed to .inp.  Alternately, this can be a character
  string giving the file name to write the Mplus syntax
  to.}

  \item{interactive}{Logical value indicating whether file
  names should be selected interactively. If
  \code{filename} is missing and \code{interative=TRUE},
  then a dialogue box will pop up to select a file or a
  console prompt if in a non interactive context. Defaults
  to \code{TRUE}.}

  \item{overwrite}{Logical value indicating whether data
  and input (if present) files should be overwritten.
  Defaults to \code{TRUE} to be consistent with prior
  behavior.  If \code{FALSE} and the file to write the data
  to already exists, it will throw an error.}
}
\value{
  Invisibly returns a character vector of the Mplus input
  syntax. Primarily called for its side effect of creating
  Mplus data files and optionally input files.
}
\description{
  The \code{prepareMplusData} function converts an R
  data.frame object into a tab-delimited file (without
  header) to be used in an Mplus input file. The
  corresponding Mplus syntax, including the data file
  definition and variable names, is printed to the console
  or optionally to an input file.
}
\examples{
\dontrun{
library(foreign)

study5 <- read.spss("reanalysis-study-5-mt-fall-08.sav", to.data.frame=TRUE)
ASData5 <- subset(study5, select=c("ppnum", paste("as", 1:33, sep="")))

prepareMplusData(ASData5, "study5.dat")


# basic example
test01 <- prepareMplusData(mtcars, "test01.dat")

# see that syntax was stored
test01

# tests for keeping and dropping variables
prepareMplusData(mtcars, "test02.dat", keepCols = c("mpg", "hp"))
prepareMplusData(mtcars, "test03.dat", keepCols = c(1, 2))
prepareMplusData(mtcars, "test04.dat",
  keepCols = c(TRUE, FALSE, FALSE, TRUE, FALSE,
  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE))

prepareMplusData(mtcars, "test05.dat", dropCols = c("mpg", "hp"))
prepareMplusData(mtcars, "test06.dat", dropCols = c(1, 2))
prepareMplusData(mtcars, "test07.dat",
  dropCols = c(TRUE, FALSE, FALSE, TRUE, FALSE,
  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE))


# interactive (test08.dat)
prepareMplusData(mtcars, interactive=TRUE)

# write syntax to input file, not stdout
prepareMplusData(mtcars, "test09.dat", inpfile=TRUE)

# write syntax to alternate input file, not stdout
prepareMplusData(mtcars, "test10.dat", inpfile="test10alt.inp")

# should be error, no file
prepareMplusData(mtcars, interactive=FALSE)

# new warnings if it is going to overwrite files
# (the default to be consistent with prior behavior)
prepareMplusData(mtcars, "test10.dat")

# new warnings if it is going to overwrite files
# (the default to be consistent with prior behavior)
prepareMplusData(mtcars, "test11.dat", inpfile="test10alt.inp")

# new errors if files exist and overwrite=FALSE
prepareMplusData(mtcars, "test10.dat",
  inpfile="test10alt.inp", overwrite=FALSE)
}
}
\author{
  Michael Hallquist
}
\keyword{interface}

