% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/StoSOO.R
\name{StoSOO}
\alias{StoSOO}
\title{StoSOO and SOO algorithms}
\usage{
StoSOO(
  par,
  fn,
  ...,
  lower = rep(0, length(par)),
  upper = rep(1, length(par)),
  nb_iter,
  control = list(verbose = 0, type = "sto", max = FALSE, light = TRUE)
)
}
\arguments{
\item{par}{vector with length defining the dimensionality of the optimization problem.
Providing actual values of \code{par} is not necessary (\code{NA}s are just fine).
Included primarily for compatibility with \code{\link[stats]{optim}}.}

\item{fn}{scalar function to be minimized, with first argument to be optimized over.}

\item{...}{optional additional arguments to \code{fn}.}

\item{lower, upper}{vectors of bounds on the variables.}

\item{nb_iter}{number of function evaluations allocated to optimization.}

\item{control}{list of control parameters:
\itemize{
\item verbose: verbosity level, either \code{0} (default), \code{1} or greater than \code{1},
\item type: either '\code{det}' for optimizing a deterministic function or '\code{sto}' for a stochastic one,
\item k_max: maximum number of evaluations per leaf (default: from analysis),
\item h_max: maximum depth of the tree (default: from analysis),
\item delta: confidence (default: \code{1/sqrt(nb_iter)} - from analysis),
\item light: set to \code{FALSE} to return the search tree,
\item max: if \code{TRUE}, performs maximization.
}}
}
\value{
list with components:
\itemize{
\item \code{par} best set of parameters (for a stochastic function, it corresponds to the minimum reached over the deepest unexpanded node),
\item \code{value} value of \code{fn} at \code{par},
\item \code{tree} search tree built during the execution, not returned unless \code{control$light == TRUE}.
}
}
\description{
Global optimization of a blackbox function given a finite budget of noisy evaluations,
via the Stochastic-Simultaneous Optimistic Optimisation algorithm.
The deterministic-SOO method is available for noiseless observations.
The knowledge of the function's smoothness is not required.
}
\examples{
#------------------------------------------------------------
# Example 1 : Deterministic optimization with SOO
#------------------------------------------------------------
## Define objective
fun1 <- function(x) return(-guirland(x))

## Optimization
Sol1 <- StoSOO(par = NA, fn = fun1, nb_iter = 1000, control = list(type = "det", verbose = 1))

## Display objective function and solution found
curve(fun1, n = 1001)
abline(v = Sol1$par, col = 'red')

#------------------------------------------------------------
# Example 2 : Stochastic optimization with StoSOO
#------------------------------------------------------------
set.seed(42)

## Same objective function with uniform noise
fun2 <- function(x){return(fun1(x) + runif(1, min = -0.1, max = 0.1))}

## Optimization
Sol2 <- StoSOO(par = NA, fn = fun2, nb_iter = 1000, control = list(type = "sto", verbose = 1))

## Display solution
abline(v = Sol2$par, col = 'blue')

#------------------------------------------------------------
# Example 3 : Stochastic optimization with StoSOO, 2-dimensional function
#------------------------------------------------------------

set.seed(42)

## 2-dimensional noisy objective function, defined on [0, pi/4]^2
fun3 <- function(x){return(-sin1(x[1]) * sin1(1 - x[2]) + runif(1, min = -0.05, max = 0.05))}

## Optimizing
Sol3 <- StoSOO(par = rep(NA, 2), fn = fun3, upper = rep(pi/4, 2), nb_iter = 1000)

## Display solution
xgrid <- seq(0, pi/4, length.out = 101)
Xgrid <- expand.grid(xgrid, xgrid)
ref <- apply(Xgrid, 1, function(x){(-sin1(x[1]) * sin1(1 - x[2]))})
filled.contour(xgrid, xgrid, matrix(ref, 101), color.palette  = terrain.colors,
plot.axes = {axis(1); axis(2); points(Xgrid[which.min(ref),, drop = FALSE], pch = 21);
             points(Sol3$par[1],Sol3$par[2], pch = 13)})


}
\references{
R. Munos (2011), Optimistic optimization of deterministic functions without the knowledge of its smoothness,
\emph{NIPS}, 783-791. \cr \cr
M. Valko, A. Carpentier and R. Munos (2013), Stochastic Simultaneous Optimistic Optimization,
\emph{ICML}, 19-27 \url{http://hal.inria.fr/hal-00789606}. Matlab code: \url{https://team.inria.fr/sequel/software/StoSOO}. \cr \cr
P. Preux, R. Munos, M. Valko (2014), Bandits attack function optimization, \emph{IEEE Congress on Evolutionary Computation (CEC)}, 2245-2252.
}
\author{
M. Binois (translation in R code), M. Valko, A. Carpentier, R. Munos (Matlab code)
}
