###########################################################################/**
# @set "class=GenericDataFileSet"
# @RdocMethod getChecksumObjects
#
# @title "Retrieves objects that uniquely identifying the data set"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Not used.}
#   \item{verbose}{A @logical or @see "R.utils::Verbose".}
# }
#
# \value{
#   Returns a named @list of objects that can be used to uniquely 
#   identifying the GenericDataFileSet.
# }
#
# \details{
#   By default, a GenericDataFileSet is assumed to be uniquely
#   identifiable by its files and their fullnames
#   (possibly translated).
# }
#
# @author
#
# \seealso{
#   @seemethod "getChecksum"
#
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("getChecksumObjects", "GenericDataFileSet", function(this, ..., verbose=FALSE) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Argument 'verbose':
  verbose <- Arguments$getVerbose(verbose);
  if (verbose) {
    pushState(verbose);
    on.exit(popState(verbose));
  }


  verbose && enter(verbose, "Retrieving checksum object for identifying this ", class(this)[1]);

  n <- length(this);
  verbose && cat(verbose, "Number of files: ", n);


  verbose && enter(verbose, "Retrieveing checksums for all files");
  fullnames <- getFullNames(this);

  checksums <- character(length(n));
  for (ii in seq_len(n)) {
    df <- getFile(this, ii);
    checksums[ii] <- getChecksum(df);
    verbose && printf(verbose, "File #%d ('%s') checksum: %s\n", ii, fullnames[ii], checksums[ii]);
  } # for (ii ...)
  verbose && exit(verbose);


  files <- data.frame(fullname=fullnames, checksum=checksums, stringsAsFactors=FALSE);

  verbose && cat(verbose, "Checksum data for identifying this ", class(this)[1], ":");
  res <- list(
    fullname = getFullName(this),
    files = files
  );
  verbose && str(verbose, res);


  verbose && exit(verbose);

  res;    
}, protected=TRUE) # getChecksumObjects()




###########################################################################/**
# @RdocMethod getChecksum
#
# @title "Generates a deterministic checksum for an R object"
#
# \description{
#  @get "title" that can be used to identify the data set.
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Arguments passed to @seemethod "getChecksumObjects".}
# }
#
# \value{
#   Returns the checksum as a @character string.
# }
#
# @author
#
# \seealso{
#   Internally, the checksum is calculated on the objects returned by
#   @seemethod "getChecksumObjects".
#
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("getChecksum", "GenericDataFileSet", function(this, ...) {
  # Retrieve data/objects unique for this data set
  data <- getChecksumObjects(this, ...);

  # Calculate the checksum based on this
  getChecksum(data);
})


###########################################################################
# HISTORY:
# 2011-04-06
# o Added getChecksum() to GenericDataFileSet, which calculates the 
#   checksum of the object returned by the protected getChecksumData().
# o Created.
###########################################################################
