#' @title Standard Error Estimate for Lower Partial Moment (LPM) of Returns
#'
#' @description \code{LPM.SE} computes the standard error of the LPM of the returns.
#'
#' @param data Data of returns for one or multiple assets or portfolios.
#' @param k Range parameter for the shape of the IF (the SD gets multiplied k times).
#' @param const Constant threshold.
#' @param se.method A character string indicating which method should be used to compute
#' the standard error of the estimated standard deviation. One or a combination of:
#' \code{"IFiid"} (default), \code{"IFcor"} (default), \code{"IFcorPW"}, \code{"IFcorAdapt"},
#' \code{"BOOTiid"} or \code{"BOOTcor"}.
#' @param cleanOutliers Boolean variable to indicate whether the pre-whitenning of the influence functions TS should be done through a robust filter.
#' @param fitting.method Distribution used in the standard errors computation. Should be one of "Exponential" (default) or "Gamma".
#' @param ... Additional parameters.
#'
#' @return A vector or a list depending on \code{se.method}.
#'
#' @export
#'
#' @author Xin Chen, \email{chenx26@uw.edu}
#' @author Anthony-Alexander Christidis, \email{anthony.christidis@stat.ubc.ca}
#'
#' @examples
#' # Loading data from PerformanceAnalytics
#' data(edhec, package = "PerformanceAnalytics")
#' class(edhec)
#' # Changing the data colnames
#' names(edhec) = c("CA", "CTA", "DIS", "EM", "EMN",
#'                  "ED", "FIA", "GM", "LS", "MA",
#'                  "RV", "SS", "FOF")
#' # Computing the standard errors for
#' # the two influence functions based approaches
#' LPM.SE(edhec, se.method=c("IFiid","IFcor"),
#'        cleanOutliers=FALSE,
#'        fitting.method=c("Exponential", "Gamma")[1])
#'
LPM.SE = function(data, const = 0, k = 1,
                  se.method=c("IFiid","IFcor","IFcorAdapt","IFcorPW","BOOTiid","BOOTcor")[1:2],
                  cleanOutliers=FALSE, fitting.method=c("Exponential", "Gamma")[1],
                  ...){
  data = checkData(data)
  myLPM = t(apply(data, 2, LPM, const = const, k = k, ...))
  rownames(myLPM) = "LPM"
  if(is.null(se.method)){
    return(myLPM)
  } else {
    res=list(LPM=myLPM)
    # for each of the method specified in se.method, compute the standard error
    for(mymethod in se.method){
      res[[mymethod]]=EstimatorSE(data, estimator.fun = "LPM", const = const, k = k,
                                  se.method = mymethod,
                                  cleanOutliers=cleanOutliers,
                                  fitting.method=fitting.method,
                                  ...)
    }
    return(res)
  }
}
