% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rvn_rvh_read.R
\name{rvn_rvh_read}
\alias{rvn_rvh_read}
\title{Read Raven .rvh (watershed discretization) file}
\usage{
rvn_rvh_read(ff)
}
\arguments{
\item{ff}{the filepath of the .rvh file (with .rvh extension included).}
}
\value{
Returns a list including:
\item{SBtable}{a data table of Subbasin characteristics indexed by Subbasin ID (SBID). Includes
the following data columns from the .rvh file : SBID, Name, Downstream_ID, Profile, ReachLength,
Gauged. The rvn_rvh_read() functions supplements this with additional columns: Area, Elevation, AvgLatit,
AvgLongit, AvgSlope, AvgAspect, DomLU, DomLUArea, DomLUFrac, DomVeg, DomVegArea, DomVegFrac.
Elevation, AvgLatit, AvgLongit, AvgSlope, and AvgAspect are the area-weighted averages from all
constituent HRUs. DomLU is the dominant land use name, DomLUArea is the area (in km2) of the
dominant land use and DomLUArea is the percentage of the basin covered with DomLU; same applies to DomVeg.}

\item{HRUtable}{a data table of HRU characteristics, with land use and vegetation classes as factors.
Contains identical information as found in the :HRUs-:EndHRUs block of the .rvh file: ID, Area,
Elevation, Latitude, Longitude, SBID, LandUse,Vegetation, SoilProfile, Terrain, Aquifer, Slope,
and Aspect.}

\item{SBnetwork}{an igraph network graph network describing subbasin stream network connectivity,
with nodes indexed by SBID.}
}
\description{
This routine reads in a valid Raven watershed discretization (.rvh) file and returns the
information about HRUs and Subbasins as data tables. It also returns a subbasin igraph
network object which describes stream network connectivity and adds additional HRU-derived
subbasin characteristics such as total upstream area and dominant land/vegetation classes.
}
\details{
This function does not like comma-delimited tables with a trailing comma. The function also
does not like tabs in the rvh file, the file should be untabified first.
This function uses the igraph library t0o build the networks and compute the total upstream area.
The .rvh file can have arbitrary contents outside of the :HRUs-:EndHRUs and :SubBasins-:EndSubBasins
command blocks.

The ff argument can be a relative path name or absolute one.

The TotalUpstreamArea is the total drainage area upstream of the given subbasin outlet. With this calculation,
headerwater subbasins will have a total upstream area equal to their own subbasin area.
}
\examples{
  # load example rvh file
  nith <- system.file("extdata","Nith.rvh",package = "RavenR")
  rvh <- rvn_rvh_read(nith)

  # number of HRUs
  nrow(rvh$HRUtable)

  # total watershed area
  sum(rvh$HRUtable$Area)

  # sub-table of headwater basins (upstream area = subbasin area)
  rvh$SBtable$SBID[rvh$SBtable$Area == rvh$SBtable$TotalUpstreamArea]

  # sub-table of Urban HRUs
  subset(rvh$HRUtable, LandUse == "URBAN")

  # get total area upstream of subbasin containing outlet
  upstr <- cumsum(rvh$SBtable$Area)
  upstr[rvh$SBtable$Downstream_ID == -1]

  # show upstream areas for each subbasin
  rvh$SBtable[,c("SBID","TotalUpstreamArea")]

  # plot network diagram using igraph library
  igraph::plot.igraph(rvh$SBnetwork)

}
\seealso{
\code{\link{rvn_rvh_write}} to write contents of the generated (and usually modified HRU and SubBasin tables)
\code{\link{rvn_subbasin_network_plot}} to plot the subbasin network
}
\author{
James R. Craig, University of Waterloo
}
