% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregate_multiple_fun.R
\name{aggregate_multiple_fun}
\alias{aggregate_multiple_fun}
\title{Wrapper to \code{aggregate}}
\usage{
aggregate_multiple_fun(
  data,
  by,
  vars,
  fun = NULL,
  ind = NULL,
  ...,
  name_sep = "_",
  seve_sep = ":",
  multi_sep = ",",
  forward_dots = FALSE,
  dots2dots = FALSE
)
}
\arguments{
\item{data}{A data frame containing data to be aggregated}

\item{by}{A data frame defining grouping}

\item{vars}{A named vector or list of variable names in \code{data}. The elements are named by the names of \code{fun}.
All the pairs of variable names and function names thus define all the result variables to be generated.
Parameter \code{vars} will converted to an internal standard by the function \code{\link{fix_vars_amf}}.
Thus, function names and also output variable names can be coded in different ways.
Multiple output variable names can be coded using \code{multi_sep}.
See examples and examples in \code{\link{fix_vars_amf}}.
Indices instead of variable names are allowed.}

\item{fun}{A named list of functions. These names will be used as suffixes in output variable names. Name can be omitted for one function.
A vector of function as strings is also possible. When unnamed, these function names will be used directly.
See the examples of \code{\link{fix_fun_amf}}, which is the function used to convert \code{fun}.
Without specifying \code{fun}, the functions, as strings, are taken from the function names coded in \code{vars}.}

\item{ind}{When non-NULL, a data frame of indices.
When NULL, this variable will be generated internally as \code{data.frame(ind = seq_len(nrow(data)))}.
The parameter is useful for advanced use involving model/dummy matrices.}

\item{...}{Further arguments passed to \code{aggregate} and,
depending on \code{forward_dots}/\code{dots2dots}, forwarded to the functions in \code{fun} (see details).}

\item{name_sep}{A character string used when output variable names are generated.}

\item{seve_sep}{A character string used when output variable names are generated from functions of several variables.}

\item{multi_sep}{A character string used when multiple output variable names are sent as input.}

\item{forward_dots}{Logical vector (possibly recycled) for each element of \code{fun} that determines whether \code{...} should be forwarded (see details).}

\item{dots2dots}{Logical vector (possibly recycled) specifying the behavior when \code{forward_dots = TRUE} (see details).}
}
\value{
A data frame
}
\description{
Wrapper to \code{\link{aggregate}} that allows multiple functions and functions of several variables
}
\details{
One intention of \code{aggregate_multiple_fun} is to be a true generalization of \code{aggregate}.
However, when many functions are involved, passing extra parameters can easily lead to errors.
Therefore \code{forward_dots} and \code{dots2dots} are set to \code{FALSE} by default.
When \code{forward_dots = TRUE} and \code{dots2dots = FALSE}, parameters will be forwarded,
but only parameters that are explicitly defined in the specific \code{fun} function.
For the \code{sum} function, this means that a possible \code{na.rm} parameter is forwarded but not others.
When \code{forward_dots = TRUE} and \code{dots2dots = TRUE}, other parameters will also be forwarded to \code{fun} functions where \code{...} is included.
For the \code{sum} function, this means that such extra parameters will, probably erroneously, be included in the summation (see examples).
}
\examples{
z2 <- SSBtoolsData("z2")
set.seed(12)
z2$y <- round(rnorm(nrow(z2)), 2)
z <- z2[sample.int(nrow(z2), size = 20), ]
aggregate_multiple_fun(
   data = z, 
   by = z[c("kostragr", "hovedint")], 
   vars = c("ant", "y", median = "ant", median = "y", d1 = "ant"),
   fun = c(sum, median = median, d1 = function(x) x[1])  
)

# With functions as named strings 
aggregate_multiple_fun(
   data = z, 
   by = z[c("kostragr", "hovedint")], 
   vars = c(sum = "y", med = "ant", med = "y"),
   fun = c(sum = "sum", med = "median")
)

# Without specifying functions 
# - equivalent to `fun = c("sum", "median")` 
aggregate_multiple_fun(
   data = z, 
   by = z[c("kostragr", "hovedint")], 
   vars = c(sum = "y", median = "ant", median = "y")
)

# with multiple outputs (function my_range)
# and with function of two variables (weighted.mean(y, ant))
my_range <- function(x) c(min = min(x), max = max(x))
aggregate_multiple_fun(
   data = z, 
   by = z[c("kostragr", "hovedint")], 
   vars = list("ant", "y", ra = "ant", wmean  = c("y", "ant")),
   fun = c(sum, ra = my_range, wmean = weighted.mean)
)

# with specified output variable names
my_range <- function(x) c(min = min(x), max = max(x))
aggregate_multiple_fun(
   data = z, 
   by = z[c("kostragr", "hovedint")], 
   vars = list("ant", "y", 
               `antmin,antmax` = list(ra = "ant"), 
                yWmean  = list(wmean  = c("y", "ant"))),
   fun = c(sum, ra = my_range, wmean = weighted.mean)
)


# To illustrate forward_dots and dots2dots
q <- z[1, ]
q$w <- 100 * rnorm(1)
for (dots2dots in c(FALSE, TRUE)) for (forward_dots in c(FALSE, TRUE)) {
  cat("\n=======================================\n")
  cat("forward_dots =", forward_dots, ", dots2dots =", dots2dots)
  out <- aggregate_multiple_fun(
    data = q, by = q["kostragr"], 
    vars = c(sum = "ant", round = "w"), fun = c("sum", "round"),  
    digits = 3, forward_dots = forward_dots, dots2dots = dots2dots)
  cat("\n")
  print(out)
}
# In last case digits forwarded to sum (as ...) 
# and wrongly included in the summation
 
}
