% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computePL}
\alias{computePL}
\title{A Vector Summary of the Persistence Landscape Function}

\description{
Vectorizes the persistence landscape (PL) function constructed from a given persistence diagram. The \eqn{k}th order landscape function of a persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} is defined as 
\deqn{\lambda_k(t) = k\hbox{max}_{1\leq i \leq N} \Lambda_i(t), \quad k\in N,}
where \eqn{k\hbox{max}} returns the \eqn{k}th largest value and 
\deqn{\Lambda_i(t) = \left\{
        \begin{array}{ll}
            t-b_i & \quad t\in [b_i,\frac{b_i+d_i}{2}] \\
            d_i-t & \quad t\in (\frac{b_i+d_i}{2},d_i]\\
            0 & \quad \hbox{otherwise}
        \end{array}
    \right.}
}

\usage{
computePL(D, homDim, scaleSeq, k)
}
\arguments{
\item{D}{matrix with three columns containing the dimension, birth and death values respectively}

\item{homDim}{homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.)}

\item{scaleSeq}{numeric vector of increasing scale values used for vectorization}

\item{k}{order of landscape function. By default, \code{k=1}}
}


\value{
A numeric vector whose elements are the values of the \eqn{k}th order landscape function evaluated at each point of \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}}:
\deqn{(\lambda_k(t_1),\lambda_k(t_2),\ldots,\lambda_k(t_n))}
}

\author{Umar Islambekov}

\references{
1. Bubenik, P. (2015). Statistical topological data analysis using persistence landscapes. Journal of Machine Learning Research, 16(1), 77-102. 

2. Chazal, F., Fasy, B. T., Lecci, F., Rinaldo, A., & Wasserman, L. (2014, June). Stochastic convergence of persistence landscapes and silhouettes. In Proceedings of the thirtieth annual symposium on Computational geometry (pp. 474-483).
}

\examples{
N <- 100 
set.seed(123)
# sample N points uniformly from unit circle and add Gaussian noise
X <- TDA::circleUnif(N,r=1) + rnorm(2*N,mean = 0,sd = 0.2)

# compute a persistence diagram using the Rips filtration built on top of X
D <- TDA::ripsDiag(X,maxdimension = 1,maxscale = 2)$diagram 

scaleSeq = seq(0,2,length.out=11) # sequence of scale values

# compute persistence landscape (PL) for homological dimension H_0 with order of landscape k=1
computePL(D,homDim=0,scaleSeq,k=1)

# compute persistence landscape (PL) for homological dimension H_1 with order of landscape k=1
computePL(D,homDim=1,scaleSeq,k=1)
}
