% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adalara.R
\name{adalara}
\alias{adalara}
\title{Multivariate parallel archetypoid algorithm for large applications (ADALARA)}
\usage{
adalara(data, N, m, numArchoid, numRep, huge, prob, type_alg = "ada", 
        compare = FALSE, vect_tol = c(0.95, 0.9, 0.85), alpha = 0.05, 
        outl_degree = c("outl_strong", "outl_semi_strong", "outl_moderate"), 
        method = "adjbox", aaframe, lass)
}
\arguments{
\item{data}{Data matrix. Each row corresponds to an observation and each column 
corresponds to a variable. All variables are numeric. The data must have row names
so that the algorithm can identify the archetypoids in every sample.}

\item{N}{Number of samples.}

\item{m}{Sample size of each sample.}

\item{numArchoid}{Number of archetypes/archetypoids.}

\item{numRep}{For each \code{numArchoid}, run the archetype algorithm \code{numRep} 
times.}

\item{huge}{Penalization added to solve the convex least squares problems.}

\item{prob}{Probability with values in [0,1].}

\item{type_alg}{String. Options are 'ada' for the non-robust adalara algorithm and 
'ada_rob' for the robust adalara algorithm.}

\item{compare}{Boolean argument to compute the robust residual sum of squares 
if \code{type_alg = "ada"} and the non-robust if \code{type_alg = "ada_rob"}.}

\item{vect_tol}{Vector with the tolerance values. Default c(0.95, 0.9, 0.85).
Needed if \code{method='toler'}.}

\item{alpha}{Significance level. Default 0.05. Needed if \code{method='toler'}.}

\item{outl_degree}{Type of outlier to identify the degree of outlierness.
Default c("outl_strong", "outl_semi_strong", "outl_moderate"). 
Needed if \code{method='toler'}.}

\item{method}{Method to compute the outliers. Options allowed are 'adjbox' for
using adjusted boxplots for skewed distributions, and 'toler' for using
tolerance intervals.}

\item{aaframe}{Boolean value to indicate whether the frame-based (TRUE) 
(Mair et al., 2017) or the classical (FALSE) (Eugster et al., 2009) archetypes 
will be used. The frame-based archetypes are computed with an ancillary python
code available at \url{https://www.uv.es/vivigui/software}.}

\item{lass}{Frame-based archetypes matrix. Only needed if \code{aaframe = TRUE}.}
}
\value{
A list with the following elements:
\itemize{
\item cases Optimal vector of archetypoids.
\item rss Optimal residual sum of squares.
\item outliers: Outliers.
}
}
\description{
The ADALARA algorithm is based on the CLARA clustering algorithm. This is the 
parallel version of the algorithm to try to get faster results. It allows to
detect anomalies (outliers). There are two different methods to detect them:
the adjusted boxplot (default and most reliable option) and tolerance intervals.
If needed, tolerance intervals allow to define a degree of outlierness.
}
\examples{
\dontrun{
library(Anthropometry)
library(doParallel)

# Prepare parallelization (including the seed for reproducibility):
no_cores <- detectCores() - 1
cl <- makeCluster(no_cores)
registerDoParallel(cl)
clusterSetRNGStream(cl, iseed = 1)

# Load data:
data(mtcars)
data <- mtcars
n <- nrow(data)

# Arguments for the archetype/archetypoid algorithm:
# Number of archetypoids:
k <- 3 
numRep <- 2
huge <- 200

# Size of the random sample of observations:
m <- 10
# Number of samples:
N <- floor(1 + (n - m)/(m - k))
N
           
prob <- 0.75            

# ADALARA algorithm:
preproc <- preprocessing(data, stand = TRUE, percAccomm = 1)
data1 <- as.data.frame(preproc$data)

adalara_aux <- adalara(data1, N, m, k, numRep, huge, prob, 
                       "ada_rob", FALSE, method = "adjbox", aaframe = FALSE, lass = NULL)

#adalara_aux <- adalara(data1, N, m, k, numRep, huge, prob, 
#                       "ada_rob", FALSE, vect_tol = c(0.95, 0.9, 0.85), alpha = 0.05, 
#                       outl_degree = c("outl_strong", "outl_semi_strong", "outl_moderate"),
#                       method = "toler", aaframe = FALSE, lass = NULL)

# Take the minimum RSS, which is in the second position of every sublist:
adalara <- adalara_aux[which.min(unlist(sapply(adalara_aux, function(x) x[2])))][[1]]
adalara

# End parallelization:
stopCluster(cl)
}

}
\references{
Eugster, M.J.A. and Leisch, F., From Spider-Man to Hero - Archetypal Analysis in 
R, 2009. Journal of Statistical Software 30(8), 1-23.

Mair, S., Boubekki, A. and Brefeld, U., Frame-based Data Factorizations, 2017.
Proceedings of the 34th International Conference on Machine Learning, 
Sydney, Australia, 1-9.

Vinue, G., (2017). Anthropometry: An R Package for Analysis of Anthropometric Data,
\emph{Journal of Statistical Software} \bold{77(6)}, 1--39
}
\seealso{
\code{\link{do_ada}}, \code{\link{do_ada_robust}}, \code{\link{adalara_no_paral}}
}
\author{
Guillermo Vinue
}
