% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_single_dose_dataset.R
\name{create_single_dose_dataset}
\alias{create_single_dose_dataset}
\title{Create dataset of single doses}
\usage{
create_single_dose_dataset(
  dataset,
  dose_freq = EXDOSFRQ,
  start_date = ASTDT,
  start_datetime = NULL,
  end_date = AENDT,
  end_datetime = NULL,
  lookup_table = dose_freq_lookup,
  lookup_column = CDISC_VALUE,
  nominal_time = NULL,
  keep_source_vars = expr_c(exprs(USUBJID), dose_freq, start_date, start_datetime,
    end_date, end_datetime)
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{dose_freq}, \code{start_date} and \code{end_date} arguments are expected to be in the dataset.}

\item{dose_freq}{The dose frequency

The aggregate dosing frequency used for multiple doses in a row.

Permitted Values: defined by lookup table.}

\item{start_date}{The start date

A date object is expected. This object cannot contain \code{NA} values.

Refer to \code{derive_vars_dt()} to impute and derive a date from a date
character vector to a date object.}

\item{start_datetime}{The start date-time

A date-time object is expected. This object cannot contain \code{NA} values.

Refer to \code{derive_vars_dtm()} to impute and derive a date-time from a date
character vector to a date object.

If the input dataset contains frequencies which refer to \code{DOSE_WINDOW}
equals \code{"HOUR"} or \code{"MINUTE"}, the parameter must be specified.}

\item{end_date}{The end date

A date or date-time object is expected. This object cannot contain \code{NA} values.

Refer to \code{derive_vars_dt()} to impute and derive a date from a date
character vector to a date object.}

\item{end_datetime}{The end date-time

A date-time object is expected. This object cannot contain \code{NA} values.

Refer to \code{derive_vars_dtm()} to impute and derive a date-time from a date
character vector to a date object.

If the input dataset contains frequencies which refer to \code{DOSE_WINDOW}
equals \code{"HOUR"} or \code{"MINUTE"}, the parameter must be specified.}

\item{lookup_table}{The dose frequency value lookup table

The table used to look up \code{dose_freq} values and determine the appropriate
multiplier to be used for row generation. If a lookup table other than the
default is used, it must have columns \code{DOSE_WINDOW}, \code{DOSE_COUNT}, and
\code{CONVERSION_FACTOR}. The default table \code{dose_freq_lookup} is described in
detail \link[=dose_freq_lookup]{here}.

Permitted Values for \code{DOSE_WINDOW}: \code{"MINUTE"}, \code{"HOUR"}, \code{"DAY"},
\code{"WEEK"}, \code{"MONTH"}, \code{"YEAR"}}

\item{lookup_column}{The dose frequency value column in the lookup table

The column of \code{lookup_table}.}

\item{nominal_time}{The nominal relative time from first dose (\code{NFRLT})

Used for PK analysis, this will be in hours and should be 0 for
the first dose.  It can be derived as \code{(VISITDY - 1) * 24} for example.
This will be expanded as the single dose dataset is created.  For example
an \code{EXDOFRQ} of \code{"QD"} will result in the nominal_time being incremented by
24 hours for each expanded record.

The value can be NULL if not needed.}

\item{keep_source_vars}{List of variables to be retained from source dataset

This parameter can be specified if additional information is required in
the output dataset. For example \code{EXTRT} for studies with more than one
drug.}
}
\value{
The input dataset with a single dose per row.
}
\description{
Derives dataset of single dose from aggregate dose information. This may be
necessary when e.g. calculating last dose before an adverse event in \code{ADAE}
or deriving a total dose parameter in \code{ADEX} when \code{EXDOSFRQ != ONCE}.
}
\details{
Each aggregate dose row is split into multiple rows which each
represent a single dose.The number of completed dose periods between
\code{start_date} or \code{start_datetime} and \code{end_date} or \code{end_datetime} is
calculated with \code{compute_duration} and multiplied by \code{DOSE_COUNT}.
For \code{DOSE_WINDOW} values of \code{"WEEK"}, \code{"MONTH"}, and \code{"YEAR"},
\code{CONVERSION_FACTOR} is used to convert into days the time object
to be added to \code{start_date}.

Observations with dose frequency \code{"ONCE"} are copied to the output dataset
unchanged.
}
\examples{
# Example with default lookup

library(lubridate)
library(stringr)
library(tibble)
library(dplyr)

data <- tribble(
  ~USUBJID, ~EXDOSFRQ, ~ASTDT, ~ASTDTM, ~AENDT, ~AENDTM,
  "P01", "Q2D", ymd("2021-01-01"), ymd_hms("2021-01-01 10:30:00"),
  ymd("2021-01-07"), ymd_hms("2021-01-07 11:30:00"),
  "P01", "Q3D", ymd("2021-01-08"), ymd_hms("2021-01-08 12:00:00"),
  ymd("2021-01-14"), ymd_hms("2021-01-14 14:00:00"),
  "P01", "EVERY 2 WEEKS", ymd("2021-01-15"), ymd_hms("2021-01-15 09:57:00"),
  ymd("2021-01-29"), ymd_hms("2021-01-29 10:57:00")
)

create_single_dose_dataset(data)

# Example with custom lookup

custom_lookup <- tribble(
  ~Value,   ~DOSE_COUNT, ~DOSE_WINDOW, ~CONVERSION_FACTOR,
  "Q30MIN", (1 / 30),    "MINUTE",                      1,
  "Q90MIN", (1 / 90),    "MINUTE",                      1
)

data <- tribble(
  ~USUBJID, ~EXDOSFRQ, ~ASTDT, ~ASTDTM, ~AENDT, ~AENDTM,
  "P01", "Q30MIN", ymd("2021-01-01"), ymd_hms("2021-01-01T06:00:00"),
  ymd("2021-01-01"), ymd_hms("2021-01-01T07:00:00"),
  "P02", "Q90MIN", ymd("2021-01-01"), ymd_hms("2021-01-01T06:00:00"),
  ymd("2021-01-01"), ymd_hms("2021-01-01T09:00:00")
)

create_single_dose_dataset(data,
  lookup_table = custom_lookup,
  lookup_column = Value,
  start_datetime = ASTDTM,
  end_datetime = AENDTM
)
# Example with nominal time

data <- tribble(
  ~USUBJID, ~EXDOSFRQ, ~NFRLT, ~ASTDT, ~ASTDTM, ~AENDT, ~AENDTM,
  "P01", "BID", 0, ymd("2021-01-01"), ymd_hms("2021-01-01 08:00:00"),
  ymd("2021-01-07"), ymd_hms("2021-01-07 20:00:00"),
  "P01", "BID", 168, ymd("2021-01-08"), ymd_hms("2021-01-08 08:00:00"),
  ymd("2021-01-14"), ymd_hms("2021-01-14 20:00:00"),
  "P01", "BID", 336, ymd("2021-01-15"), ymd_hms("2021-01-15 08:00:00"),
  ymd("2021-01-29"), ymd_hms("2021-01-29 20:00:00")
)

create_single_dose_dataset(data,
  dose_freq = EXDOSFRQ,
  start_date = ASTDT,
  start_datetime = ASTDTM,
  end_date = AENDT,
  end_datetime = AENDTM,
  lookup_table = dose_freq_lookup,
  lookup_column = CDISC_VALUE,
  nominal_time = NFRLT,
  keep_source_vars = exprs(
    USUBJID, EXDOSFRQ, ASTDT, ASTDTM, AENDT, AENDTM, NFRLT
  )
)

# Example - derive a single dose dataset with imputations

# For either single drug administration records, or multiple drug administration
# records covering a range of dates, fill-in of missing treatment end datetime
# `EXENDTC` by substitution with an acceptable alternate, for example date of
# death, date of datacut may be required. This example shows the
# maximum possible number of single dose records to be derived. The example
# requires the date of datacut `DCUTDT` to be specified correctly, or
# if not appropriate to use `DCUTDT` as missing treatment end data and missing
# treatment end datetime could set equal to treatment start date and treatment
# start datetime. ADSL variables `DTHDT` and `DCUTDT` are preferred for
# imputation use.
#
# All available trial treatments are included, allowing multiple different
# last dose variables to be created in for example `use_ad_template("ADAE")`
# if required.

adsl <- tribble(
  ~STUDYID, ~USUBJID, ~DTHDT,
  "01", "1211", ymd("2013-01-14"),
  "01", "1083", ymd("2013-08-02"),
  "01", "1445", ymd("2014-11-01"),
  "01", "1015", NA,
  "01", "1023", NA
)

ex <- tribble(
  ~STUDYID, ~USUBJID, ~EXSEQ, ~EXTRT, ~EXDOSE, ~EXDOSU, ~EXDOSFRQ, ~EXSTDTC, ~EXENDTC,
  "01", "1015", 1, "PLAC", 0, "mg", "QD", "2014-01-02", "2014-01-16",
  "01", "1015", 2, "PLAC", 0, "mg", "QD", "2014-06-17", "2014-06-18",
  "01", "1015", 3, "PLAC", 0, "mg", "QD", "2014-06-19", NA_character_,
  "01", "1023", 1, "PLAC", 0, "mg", "QD", "2012-08-05", "2012-08-27",
  "01", "1023", 2, "PLAC", 0, "mg", "QD", "2012-08-28", "2012-09-01",
  "01", "1211", 1, "XANO", 54, "mg", "QD", "2012-11-15", "2012-11-28",
  "01", "1211", 2, "XANO", 54, "mg", "QD", "2012-11-29", NA_character_,
  "01", "1445", 1, "PLAC", 0, "mg", "QD", "2014-05-11", "2014-05-25",
  "01", "1445", 2, "PLAC", 0, "mg", "QD", "2014-05-26", "2014-11-01",
  "01", "1083", 1, "PLAC", 0, "mg", "QD", "2013-07-22", "2013-08-01"
)

adsl_death <- adsl \%>\%
  mutate(
    DTHDTM = convert_date_to_dtm(DTHDT),
    # Remove `DCUT` setup line below if ADSL `DCUTDT` is populated.
    DCUTDT = convert_dtc_to_dt("2015-03-06"), # Example only, enter date.
    DCUTDTM = convert_date_to_dtm(DCUTDT)
  )

# Select valid dose records, non-missing `EXSTDTC` and `EXDOSE`.
ex_mod <- ex \%>\%
  filter(!is.na(EXSTDTC) & !is.na(EXDOSE)) \%>\%
  derive_vars_merged(adsl_death, by_vars = exprs(STUDYID, USUBJID)) \%>\%
  # Example, set up missing `EXDOSFRQ` as QD daily dosing regime.
  # Replace with study dosing regime per trial treatment.
  mutate(EXDOSFRQ = if_else(is.na(EXDOSFRQ), "QD", EXDOSFRQ)) \%>\%
  # Create EXxxDTM variables and replace missing `EXENDTM`.
  derive_vars_dtm(
    dtc = EXSTDTC,
    new_vars_prefix = "EXST",
    date_imputation = "first",
    time_imputation = "first",
    flag_imputation = "none",
  ) \%>\%
  derive_vars_dtm_to_dt(exprs(EXSTDTM)) \%>\%
  derive_vars_dtm(
    dtc = EXENDTC,
    new_vars_prefix = "EXEN",
    # Maximum imputed treatment end date must not be not greater than
    # date of death or after the datacut date.
    max_dates = exprs(DTHDTM, DCUTDTM),
    date_imputation = "last",
    time_imputation = "last",
    flag_imputation = "none",
    highest_imputation = "Y",
  ) \%>\%
  derive_vars_dtm_to_dt(exprs(EXENDTM)) \%>\%
  # Select only unique values.
  # Removes duplicated records before final step.
  distinct(
    STUDYID, USUBJID, EXTRT, EXDOSE, EXDOSFRQ, DCUTDT, DTHDT, EXSTDT,
    EXSTDTM, EXENDT, EXENDTM, EXSTDTC, EXENDTC
  )

create_single_dose_dataset(
  ex_mod,
  start_date = EXSTDT,
  start_datetime = EXSTDTM,
  end_date = EXENDT,
  end_datetime = EXENDTM,
  keep_source_vars = exprs(
    STUDYID, USUBJID, EXTRT, EXDOSE, EXDOSFRQ,
    DCUTDT, EXSTDT, EXSTDTM, EXENDT, EXENDTM, EXSTDTC, EXENDTC
  )
)
}
\seealso{
Creating auxiliary datasets: 
\code{\link{consolidate_metadata}()},
\code{\link{create_period_dataset}()},
\code{\link{create_query_data}()}
}
\concept{create_aux}
\keyword{create_aux}
