% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ruleInduction.R
\name{ruleInduction}
\alias{ruleInduction}
\alias{ruleInduction,itemsets-method}
\title{Association Rule Induction from Itemsets}
\usage{
ruleInduction(x, ...)

\S4method{ruleInduction}{itemsets}(
  x,
  transactions = NULL,
  confidence = 0.8,
  method = c("ptree", "apriori"),
  reduce = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{the set of \link{itemsets} from which rules will be induced.}

\item{...}{further arguments.}

\item{transactions}{the \link{transactions} used to mine the itemsets. Can
be omitted for method \code{"ptree"},
if \code{x} contains a (complete set) of frequent itemsets
together with their support counts.}

\item{confidence}{a numeric value between 0 and 1 giving the minimum confidence
threshold for the rules.}

\item{method}{\code{"ptree"} or \code{"apriori"}}

\item{reduce}{remove unused items to speed up the counting process?}

\item{verbose}{report progress?}
}
\value{
An object of class \link{rules}.
}
\description{
Provides the generic function \code{ruleInduction()} and the method to induce all association
\link{rules} which can be generated by the given set of \link{itemsets} from a \link{transactions}
dataset.
}
\details{
All rules that can be created using the supplied itemsets and that surpass the
specified minimum confidence threshold are returned.
\code{ruleInduction()} can be used to produce
closed association rules defined by Pei
et al. (2000) as rules \verb{X => Y} where both \code{X} and \code{Y} are
closed frequent itemsets. See the code example in the Example section.

Rule induction implements several induction methods. The default method is \code{"ptree"}
\itemize{
\item \code{"ptree"} \strong{method without transactions:}
No transactions are need to be specified if
\code{x} contains a complete set of frequent or
itemsets. The itemsets' support counts are stored in a ptree and then retrieved to
create rules and calculate rules confidence. This is very fast,
but fails because of missing
support values if \code{x} is not a complete set of frequent itemsets.
\item \code{"ptree"} \strong{method with transactions:}
If transactions are specified then all transactions are counted into a prefix
tree and
later retrieved to create rules from the itemsets and calculate confidence values.
This is slower, but necessary if \code{x} is not a complete set of frequent itemsets.
To improve speed, unused items are removed from the transaction
data before creating the prefix tree (this behavior can be changed using the
argument \code{reduce}). This might be slower for large transaction
data sets. However, this is highly recommended as the items are also
reordered to reduce the counting time.
\item \code{"apriori"} \strong{method (always needs transactions):}
All association rules are mined from the transactions data set using \code{\link[=apriori]{apriori()}}
with the
smallest support found in the itemsets. In a second step, all rules which cannot
be generated from one of the itemsets are removed. This procedure is very slow,
especially for itemsets with many elements or very low support.
}
}
\examples{
data("Adult")

## find all closed frequent itemsets
closed_is <- apriori(Adult, target = "closed frequent itemsets", support = 0.4)
closed_is

## use rule induction to produce all closed association rules
closed_rules <- ruleInduction(closed_is, transactions = Adult, verbose = TRUE)

## inspect the resulting closed rules
summary(closed_rules)
inspect(head(closed_rules, by = "lift"))

## get rules from frequent itemsets. Here, transactions does not need to be
## specified for rule induction.
frequent_is  <- eclat(Adult, support = 0.4)
assoc_rules <- ruleInduction(frequent_is)
assoc_rules
inspect(head(assoc_rules))

## for itemsets that are not a complete set of frequent itemsets,
## transactions need to be specified.
some_is <- sample(frequent_is, 10)
some_rules <- ruleInduction(some_is, transactions = Adult)
some_rules
}
\references{
Michael Hahsler, Christian Buchta, and Kurt Hornik. Selective
association rule generation. \emph{Computational Statistics,} 23(2):303-315,
April 2008.

Jian Pei, Jiawei Han, Runying Mao. CLOSET: An Efficient Algorithm for Mining
Frequent Closed Itemsets. \emph{ACM SIGMOD Workshop on Research Issues in Data
Mining and Knowledge Discovery (DMKD 2000).}
}
\seealso{
Other mining algorithms: 
\code{\link{APappearance-class}},
\code{\link{AScontrol-classes}},
\code{\link{ASparameter-classes}},
\code{\link{apriori}()},
\code{\link{eclat}()},
\code{\link{fim4r}()},
\code{\link{weclat}()}
}
\author{
Christian Buchta and Michael Hahsler
}
\concept{mining algorithms}
\concept{postporocessing}
\keyword{models}
