#' @title Half-Normal Score
#'
#' @description Score is approximately:
#' \eqn{ \sum{\#[res_i \leq simres_{i,j}] - n } }
#' with the distinction that each element of sum is also scaled to take values from [0,1].
#'
#' \eqn{res_i} is a residual for i-th observation, \eqn{simres_{i,j}} is the residual of j-th simulation for i-th observation, and \eqn{n} is the number of simulations for each observation.
#' Scores are calculated on the basis of simulated data, so they may differ between function calls.
#'
#' @param object An object of class 'explainer' created with function \code{\link[DALEX]{explain}} from the DALEX package.
#' @param ... Extra arguments passed to \link[hnp]{hnp}.
#'
#'
#' @examples
#' dragons <- DALEX::dragons[1:100, ]
#'
#' # fit a model
#' model_lm <- lm(life_length ~ ., data = dragons)
#'
#' # create an explainer
#' exp_lm <- DALEX::explain(model_lm, data = dragons, y = dragons$life_length)
#'
#' # calculate score
#' score_halfnormal(exp_lm)
#'
#' @return An object of class 'score_audit'.
#'
#' @importFrom hnp hnp
#'
#' @export

score_halfnormal <- function(object, ...){
  if(!("explainer" %in% class(object))) stop("The function requires an object created with explain() function from the DALEX package.")
  object <- model_halfnormal(object)

  result <- list(
    name = "halfnormal",
    score = calculate_score_pdf(object)
  )

  class(result) <- "auditor_score"
  return(result)
}


# Calculating Likelihood for each residual
calculate_kde <- function(res, simres){
  simres <- as.numeric(simres)
  (abs(sum(res<=simres) - length(simres)/2))/(length(simres)/2)
}


# Calculating PDF score
calculate_score_pdf <- function(hnpObject){
  res <- hnpObject$`_residuals_`
  simres <- as.data.frame(t(hnpObject[,6:ncol(hnpObject)]))
  n <- length(res)
  PDFs <- mapply(calculate_kde, res, simres)
  return(sum(PDFs))
}

#' @rdname score_halfnormal
#' @export
scoreHalfNormal <- function(object, ...){
  message("Please note that 'scoreHalfNormal()' is now deprecated, it is better to use 'score_halfnormal()' instead.")
  score_halfnormal(object, ...)
}
