#' The fixed degree sequence model (fdsm) for backbone probabilities
#'
#' `fdsm` computes the proportion of generated edges above
#'     or below the observed value using the fixed degree sequence model.
#'     Once computed, use \code{\link{backbone.extract}} to
#'     return the backbone matrix for a given alpha value.
#'
#' @param B graph: An unweighted bipartite graph object of class matrix, sparse matrix, igraph, edgelist, or network object.
#'     Any rows and columns of the associated bipartite matrix that contain only zeros are automatically removed before computations.
#' @param trials numeric: If trials > 1, this represents the number of bipartite graphs generated from the sampled distribution. If 0<trials<1, the number of graphs generated will be \eqn{(R^2-R)/trials + 1} where `R` is the number of rows.
#'     This is the minimum number of trials required to achieve a 'trials' familywise error rate when extracting the backbone using \link{backbone.extract}

#' @param dyad vector length 2: two row entries i,j. Saves each value of the i-th row and j-th column in each projected B* matrix. This is useful for visualizing an example of the empirical null edge weight distribution generated by the model. These correspond to the row and column indices of a cell in the projected matrix, and can be written as their string row names or as numeric values.
#' @param progress Boolean: If \link[utils]{txtProgressBar} should be used to measure progress
#' @param ... optional arguments
#'
#' @details During each iteration, fdsm computes a new B* matrix using the \link{curveball} algorithm. This is a random bipartite matrix with the same row and column sums as the original matrix B.
#'     If a value is supplied for the dyad parameter, when the B* matrix is projected (multiplied by its transpose), the value in the corresponding row and column will be saved.
#'     This allows the user to see the distribution of the edge weights for desired row and column.
#' @details The "backbone" S3 class object returned is composed of two matrices, a summary dataframe and (if specified) a 'dyad_values' vector.
#' @return backbone, a list(positive, negative, dyad_values, summary). Here
#'     `positive` is a matrix of proportion of times each entry of the projected matrix B is above the corresponding entry in the generated projection,
#'     `negative` is a matrix of proportion of times each entry of the projected matrix B is below the corresponding entry in the generated projection,
#'     `dyad_values` is a list of edge weight for i,j in each generated projection, and
#'     `summary` is a data frame summary of the inputted matrix and the model used including: model name, number of rows, skew of row sums, number of columns, skew of column sums, and running time.
#'
#' @references fixed degree sequence model: {Zweig, Katharina Anna, and Michael Kaufmann. 2011. “A Systematic Approach to the One-Mode Projection of Bipartite Graphs.” Social Network Analysis and Mining 1 (3): 187–218. \doi{10.1007/s13278-011-0021-0}}
#' @references curveball algorithm: {Strona, Giovanni, Domenico Nappo, Francesco Boccacci, Simone Fattorini, and Jesus San-Miguel-Ayanz. 2014. “A Fast and Unbiased Procedure to Randomize Ecological Binary Matrices with Fixed Row and Column Totals.” Nature Communications 5 (June). Nature Publishing Group: 4114. \doi{10.1038/ncomms5114}}
#'
#' @export
#'
#' @examples
#' fdsm_props <- fdsm(davis, trials = 100, dyad=c(3,6))

fdsm <- function(B,
                 trials = 1000,
                 dyad = NULL,
                 progress = FALSE,
                 ...){

  #### Argument Checks ####
  if (trials < 0) {stop("trials must be a positive integer")}
  if ((trials > 1) & (trials%%1!=0)) {stop("trials must be decimal < 1, or a positive integer")}
  ### Run Time ###
  run.time.start <- Sys.time()

  #### Class Conversion ####
  convert <- tomatrix(B)
  class <- convert$summary[[1]]
  B <- convert$G
  if (convert$summary[[4]]==TRUE){stop("Graph must be unweighted.")}
  if (convert$summary[[2]]==FALSE){warning("This object is being treated as a bipartite network.")}


  #### Set Trials if Decimal ####
  if ((trials > 0)&(trials<1)){
    trials <- (((dim(B)[1])**2-(dim(B)[1]))/trials) + 1
  }

  #### Bipartite Projection ####
  P <- tcrossprod(B)

  ### Create Positive and Negative Matrices to hold backbone ###
  Positive <- matrix(0, nrow(P), ncol(P))
  Negative <- matrix(0, nrow(P), ncol(P))

  ### Dyad save ###
  edge_weights <- numeric(trials)
  if (length(dyad) > 0){
    if (class(dyad[1]) != "numeric"){
      vec <- match(c(dyad[1], dyad[2]), rownames(B))
    } else {
      vec <- dyad
    }
  }

  #### Build Null Models ####
  for (i in 1:trials){

    #Algorithm credit to: Strona, G., Nappo, D., Boccacci, F., Fattorini, S., San-Miguel-Ayanz, J. (2014). A fast and unbiased procedure to randomize ecological binary matrices with fixed row and column totals. Nature Communications, 5, 4114
    ### Use curveball to create an FDSM Bstar ###
    Bstar <- curveball(B)

    ### Construct Pstar from Bstar ###
    Pstar <- tcrossprod(Bstar)

    ### Start estimation timer; print message ###
    if (i == 1) {
      start.time <- Sys.time()
    }

    ### Check whether Pstar edge is larger/smaller than P edge ###
    Positive <- Positive + (Pstar >= P)+0
    Negative <- Negative + (Pstar <= P)+0

    ### Save Dyad of P ###
    if (length(dyad) > 0){
      edge_weights[i] <- Pstar[vec[1], vec[2]]
    }

    ### Report estimated running time, update progress bar ###
    if (i==10){
      end.time <- Sys.time()
      est = (round(difftime(end.time, start.time, units = "auto"), 2) * (trials/10))
      message("Estimated time to complete is ", est, " " , units(est), " for ", trials, " trials")
      if (progress == "TRUE"){
        pb <- utils::txtProgressBar(min = 0, max = trials, style = 3)
      }
    }
    if ((progress == "TRUE") & (i>=10)) {utils::setTxtProgressBar(pb, i)}
  } #end for loop
  if (progress == "TRUE"){close(pb)}

  #### Find Proportions ####
  ### Proporition of greater than expected and less than expected ###
  Positive <- (Positive/trials)
  Negative <- (Negative/trials)
  rownames(Positive) <- rownames(B)
  colnames(Positive) <- rownames(B)
  rownames(Negative) <- rownames(B)
  colnames(Negative) <- rownames(B)

  ### Run Time ###
  run.time.end <- Sys.time()
  total.time = (round(difftime(run.time.end, run.time.start, units = "secs"), 2))

  #### Compile Summary ####
  r <- rowSums(B)
  c <- colSums(B)

  a <- c("Input Class", "Model", "Number of Rows", "Mean of Row Sums", "SD of Row Sums", "Skew of Row Sums", "Number of Columns", "Mean of Column Sums", "SD of Column Sums", "Skew of Column Sums", "Running Time (secs)")
  b <- c(class[1], "Fixed Degree Sequence Model", dim(B)[1], round(mean(r),5), round(stats::sd(r),5), round((sum((r-mean(r))**3))/((length(r))*((stats::sd(r))**3)), 5), dim(B)[2], round(mean(c),5), round(stats::sd(c),5), round((sum((c-mean(c))**3))/((length(c))*((stats::sd(c))**3)), 5), as.numeric(total.time))
  model.summary <- data.frame(a,b, row.names = 1)
  colnames(model.summary)<-"Model Summary"

  #### Return Backbone Object ####
  if (length(dyad) > 0){
    bb <- list(positive = Positive, negative = Negative, dyad_values = edge_weights, summary = model.summary)
    class(bb) <- "backbone"
    return(bb)
  } else {
    bb <- list(positive = Positive, negative = Negative, summary = model.summary)
    class(bb) <- "backbone"
    return(bb)
  }

} #end fdsm function


#' curveball algorithm
#'
#' @param M matrix
#'
#' @return rm, a matrix with same row sums and column sums as M, but randomized 0/1 entries.
#' @export
#'
#' @references Algorithm and R implementation: \href{https://www.nature.com/articles/ncomms5114}{Strona, Giovanni, Domenico Nappo, Francesco Boccacci, Simone Fattorini, and Jesus San-Miguel-Ayanz. 2014. “A Fast and Unbiased Procedure to Randomize Ecological Binary Matrices with Fixed Row and Column Totals.” Nature Communications 5 (June). Nature Publishing Group: 4114. DOI:10.1038/ncomms5114.}
#' @examples
#' curveball(davis)
curveball<-function(M){
  #### Define Variables ####
  RC=dim(M)
  R=RC[1]
  C=RC[2]
  hp=list()

  #### Mark Locations of One's ####
  for (row in 1:dim(M)[1]) {hp[[row]]=(which(M[row,]==1))}
  l_hp=length(hp)

  #### Curveball Swaps ####
  for (rep in 1:(5*l_hp)){
    AB=sample(1:l_hp,2)
    a=hp[[AB[1]]]
    b=hp[[AB[2]]]
    ab=intersect(a,b)
    l_ab=length(ab)
    l_a=length(a)
    l_b=length(b)
    if ((l_ab %in% c(l_a,l_b))==F){
      tot=setdiff(c(a,b),ab)
      l_tot=length(tot)
      tot=sample(tot, l_tot, replace = FALSE, prob = NULL)
      L=l_a-l_ab
      hp[[AB[1]]] = c(ab,tot[1:L])
      hp[[AB[2]]] = c(ab,tot[(L+1):l_tot])}
  }

  #### Define and Return Random Matrix ####
  rm=matrix(0,R,C)
  for (row in 1:R){rm[row,hp[[row]]]=1}
  rm
}
