% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fdsm.R
\name{fdsm}
\alias{fdsm}
\title{Extract backbone using the Fixed Degree Sequence Model}
\usage{
fdsm(
  B,
  alpha = 0.05,
  trials = NULL,
  missing.as.zero = FALSE,
  signed = FALSE,
  mtc = "none",
  class = "original",
  narrative = FALSE,
  progress = TRUE,
  ...
)
}
\arguments{
\item{B}{An unweighted bipartite graph, as: (1) an incidence matrix in the form of a matrix or sparse \code{\link{Matrix}}; (2) an edgelist in the form of a two-column dataframe; (3) an \code{\link{igraph}} object.}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{trials}{numeric: the number of bipartite graphs generated to approximate the edge weight distribution. If NULL, the number of trials is selected based on \code{alpha} (see details)}

\item{missing.as.zero}{boolean: should missing edges be treated as edges with zero weight and tested for significance}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{mtc}{string: type of Multiple Test Correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "Matrix", "igraph", "edgelist").
If "original", the backbone graph returned is of the same class as \code{B}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}

\item{progress}{boolean: TRUE if the progress of Monte Carlo trials should be displayed.}

\item{...}{optional arguments}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing (1) the weighted graph as a matrix, (2) upper-tail p-values as a
matrix, (3, if \code{signed = TRUE}) lower-tail p-values as a matrix, (4, if present) node attributes as a dataframe, and
(5) several properties of the original graph and backbone model, from which a backbone can subsequently be extracted
using \code{\link[=backbone.extract]{backbone.extract()}}.
}
\description{
\code{fdsm} extracts the backbone of a bipartite projection using the Fixed Degree Sequence Model.
}
\details{
The \code{fdsm} function compares an edge's observed weight in the projection \code{B*t(B)} to the distribution of weights
expected in a projection obtained from a random bipartite network where both the row vertex degrees and column
vertex degrees are \emph{exactly} fixed at their values in \code{B}. It uses the \code{\link[=fastball]{fastball()}} algorithm to generate random
bipartite matrices with give row and column vertex degrees.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger?) is performed for each edge. The resulting backbone
contains edges whose weights are significantly \emph{stronger} than expected in the null model. When \code{signed = TRUE}, a
two-tailed test (is the weight stronger or weaker?) is performed for each edge. The resulting backbone contains
positive edges for those whose weights are significantly \emph{stronger}, and negative edges for those whose weights are
significantly \emph{weaker}, than expected in the null model.

The p-values used to evaluate the statistical significance of each edge are computed using Monte Carlo methods. The number of
\code{trials} performed affects the precision of these p-values. This precision impacts the confidence that a given edge's p-value
is less than the desired \code{alpha} level, and therefore represents a statistically significant edge that should be retained in
the backbone. When \code{trials = NULL}, \code{\link[=trials.needed]{trials.needed()}} is used to estimate the required number of trials to evaluate the
statistical significance of an edges' p-values.
}
\examples{
#A binary bipartite network of 30 agents & 75 artifacts; agents form three communities
B <- rbind(cbind(matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10)))

P <- B\%*\%t(B) #An ordinary weighted projection...
plot(igraph::graph_from_adjacency_matrix(P, mode = "undirected",
                                         weighted = TRUE, diag = FALSE)) #...is a dense hairball

bb <- fdsm(B, alpha = 0.05, trials = 1000, narrative = TRUE, class = "igraph") #An FDSM backbone...
plot(bb) #...is sparse with clear communities
}
\references{
package: {Neal, Z. P. (2022). backbone: An R Package to Extract Network Backbones. \emph{PLOS ONE, 17}, e0269137. \doi{10.1371/journal.pone.0269137}}

fdsm: {Neal, Z. P., Domagalski, R., and Sagan, B. (2021). Comparing Alternatives to the Fixed Degree Sequence Model for Extracting the Backbone of Bipartite Projections. \emph{Scientific Reports}. \doi{10.1038/s41598-021-03238-3}}

fastball: {Godard, Karl and Neal, Zachary P. 2022. fastball: A fast algorithm to sample bipartite graphs with fixed degree sequences. \emph{Journal of Complex Networks} \doi{10.1093/comnet/cnac049}}
}
