% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcapar.R
\name{bcapar}
\alias{bcapar}
\title{Compute parametric bootstrap confidence intervals}
\usage{
bcapar(t0, tt, bb, alpha = c(0.025, 0.05, 0.1, 0.16), J = 10, K = 6,
  trun = 0.001, pct = 0.333, cd = 0, func)
}
\arguments{
\item{t0}{Observed estimate of theta, usually by maximum
likelihood.}

\item{tt}{A vector of parametric bootstrap replications of theta of
length \code{B}, usually large, say \code{B = 2000}}

\item{bb}{A \code{B} by \code{p} matrix of natural sufficient vectors, where
\code{p} is the dimension of the exponential family.}

\item{alpha}{percentiles desired for the bca confidence limits. One
only needs to provide \code{alpha} values below 0.5; the upper
limits are automatically computed}

\item{J, K}{Parameters controlling the jackknife estimates of Monte
Carlo error: \code{J} jackknife folds, with the jackknife standard
errors averaged over \code{K} random divisions of \code{bb}}

\item{trun}{Truncation parameter used in the calculation of the
acceleration \code{a}.}

\item{pct}{Proportion of "nearby" b vectors used in the calculation
of \code{t.}, the gradient vector of theta.}

\item{cd}{If cd is 1 the bca confidence density is also returned;
see Section 11.6 in reference Efron and Hastie (2016) below}

\item{func}{Function \eqn{\hat{\theta} = func(b)}. If this is not missing then
output includes \emph{abc} estimates; see reference DiCiccio and Efron (1992) below}
}
\value{
a named list of several items:
\itemize{
\item \strong{lims} : Bca confidence limits (first column) and the standard
limits (fourth column). Also the abc limits (fifth column) if
\code{func} is provided. The second column, \code{jacksd}, are the
jackknife estimates of Monte Carlo error; \code{pct}, the third
column are the proportion of the replicates \code{tt} less than each
\code{bcalim} value
\item \strong{stats} : Estimates and their jackknife Monte Carlo errors:
\code{theta} = \eqn{\hat{\theta}}; \code{sd}, the bootstrap standard deviation
for \eqn{\hat{\theta}}; \code{a} the acceleration estimate; \code{az} another
acceleration estimate that depends less on extreme values of \code{tt};
\code{z0} the bias-correction estimate; \code{A} the big-A measure of raw
acceleration; \code{sdd} delta method estimate for standard deviation of
\eqn{\hat{\theta}}; \code{mean} the average of \code{tt}
\item \strong{abcstats} : The abc estimates of \code{a} and \code{z0}, returned if \code{func} was provided
\item \strong{ustats} : The bias-corrected estimator \code{2 * t0 - mean(tt)}. \code{ustats}
gives \code{ustat}, an estimate \code{sdu} of its sampling error, and jackknife
estimates of monte carlo error for both \code{ustat} and \code{sdu}. Also given
is \code{B}, the number of bootstrap replications
\item \strong{seed} : The random number state for reproducibility
}
}
\description{
bcapar computes parametric bootstrap confidence
intervals for a real-valued parameter theta in a p-parameter
exponential family. It is described in Section 4 of the
reference below.
}
\examples{
data(diabetes, package = "bcaboot")
X <- diabetes$x
y <- scale(diabetes$y, center = TRUE, scale = FALSE)
lm.model <- lm(y ~ X - 1)
mu.hat <- lm.model$fitted.values
sigma.hat <- stats::sd(lm.model$residuals)
t0 <- summary(lm.model)$adj.r.squared
y.star <- sapply(mu.hat, rnorm, n = 1000, sd = sigma.hat)
tt <- apply(y.star, 1, function(y) summary(lm(y ~ X - 1))$adj.r.squared)
b.star <- y.star \%*\% X
set.seed(1234)
bcapar(t0 = t0, tt = tt, bb = b.star)
}
\references{
DiCiccio T and Efron B (1996). Bootstrap confidence
intervals. Statistical Science 11, 189-228

T. DiCiccio and B. Efron. More accurate confidence intervals in exponential families.
Biometrika (1992) p231-245.

Efron B (1987). Better bootstrap confidence intervals. JASA 82, 171-200

B. Efron and T. Hastie. Computer Age Statistical Inference. Cambridge University Press, 2016.

B. Efron and B. Narasimhan. Automatic Construction of Bootstrap Confidence Intervals, 2018.
}
