% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spread.R
\name{spread}
\alias{spread}
\title{Estimate Bid-Ask Spreads from OHLC Prices}
\usage{
spread(
  x,
  width = nrow(x),
  method = "EDGE",
  probs = c(0.025, 0.975),
  na.rm = FALSE,
  trim = 0
)
}
\arguments{
\item{x}{\code{xts} object with columns \code{Open}, \code{High}, \code{Low}, \code{Close}, representing OHLC prices.}

\item{width}{integer width of the rolling window to use, or vector of endpoints defining the intervals to use. By default, the whole time series is used to compute a single spread estimate.}

\item{method}{the estimator(s) to use. Choose one or more of: \code{EDGE}, \code{AR}, \code{AR2}, \code{CS}, \code{CS2}, \code{Roll}, \code{O}, \code{OC}, \code{OHL}, \code{OHLC}, \code{C}, \code{CO}, \code{CHL}, \code{CHLO}, or any combination of the OHLC methods, e.g. \code{OHLC.CHLO}. See details.}

\item{probs}{vector of probabilities to compute the critical values when the method \code{EDGE} is selected. By default, the critical values at 2.5\% and 97.5\% are computed.}

\item{na.rm}{a \code{logical} value indicating whether \code{NA} values should be stripped before the computation proceeds.}

\item{trim}{the fraction (0 to 0.5) of observations to be trimmed from each end before the spread is computed. Values of trim outside that range are taken as the nearest endpoint.}
}
\value{
Time series of spread estimates.
}
\description{
This function estimates bid-ask spreads from open, high, low, and close prices with several methods.
}
\details{
The method \code{EDGE} implements the Efficient Discrete Generalized Estimator proposed in Ardia-Guidotti-Kroencke (2021).
This is an optimal combination of the OHLC methods when full OHLC price data are available.
The OHLC methods implement the generalized estimators proposed in Ardia-Guidotti-Kroencke (2021).
These estimators can be combined by concatenating their identifiers, e.g., \code{OHLC.CHLO} uses an average of the \code{OHLC} and \code{CHLO} estimators.

The method \code{AR} implements the estimator proposed in Abdi & Ranaldo (2017). \code{AR2} implements the 2-period adjusted version.

The method \code{CS} implements the estimator proposed in Corwin & Schultz (2012). \code{CS2} implements the 2-period adjusted version. Both versions are adjusted for overnight returns as described in the original paper.

The method \code{Roll} implements the estimator proposed in Roll (1984).
}
\examples{
# simulate a price process with spread 1\%
x <- sim(spread = 0.01)

# estimate the spread
spread(x)

# use a rolling window of 21 periods
spread(x, width = 21)

# compute the spread for each month
ep <- endpoints(x, on = "months")
spread(x, width = ep)

# compute the critical values at 5\% and 95\%
spread(x, probs = c(0.05, 0.95))

# use multiple estimators
spread(x, method = c("EDGE", "AR", "CS", "Roll", "OHLC", "OHL.CHL"))

}
\references{
Roll, R. (1984). A simple implicit measure of the effective bid-ask spread in an efficient market. The Journal of Finance, 39 (4), 1127-1139.
\doi{10.1111/j.1540-6261.1984.tb03897.x}

Corwin, S. A., & Schultz, P. (2012). A simple way to estimate bid-ask spreads from daily high and low prices. The Journal of Finance, 67 (2), 719-760.
\doi{10.1111/j.1540-6261.2012.01729.x}

Abdi, F., & Ranaldo, A. (2017). A simple estimation of bid-ask spreads from daily close, high, and low prices. The Review of Financial Studies, 30 (12), 4437-4480.
\doi{10.1093/rfs/hhx084}

Ardia, D., Guidotti E., & Kroencke T. A. (2021). Efficient Estimation of Bid-Ask Spreads from Open, High, Low, and Close Prices. 
Available at SSRN: \url{https://www.ssrn.com/abstract=3892335}
}
