% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Array.Functions.R
\name{Array.Measures}
\alias{Array.Measures}
\title{Operating characteristics for array testing without master pooling}
\usage{
Array.Measures(p, se, sp)
}
\arguments{
\item{p}{matrix of probabilities corresponding to each individual's risk 
of disease.}

\item{se}{the sensitivity of the diagnostic test.}

\item{sp}{the specificity of the diagnostic test.}
}
\value{
A list containing:
\item{T}{the expected number of tests for the array.}
\item{PSe}{a matrix containing each individual's pooling sensitivity, 
corresponding to the input matrix of individual probabilities.}
\item{PSp}{a matrix containing each individual's pooling specificity, 
corresponding to the input matrix of individual probabilities.}
\item{PPV}{a matrix containing each individual's pooling positive predictive
value, corresponding to the input matrix of individual probabilities.}
\item{NPV}{a matrix containing each individual's pooling negative predictive
value, corresponding to the input matrix of individual probabilities.}
}
\description{
Calculate the expected number of tests and accuracy measures
for each individual using array testing without master pooling
}
\details{
This function calculates the operating characteristics for
array testing without master pooling. Operating characteristics calculated
are expected number of tests, pooling sensitivity, pooling specificity, 
pooling positive predictive value, and pooling negative predictive value 
for each individual.
}
\section{Note}{
 This function returns the pooling positive and negative
predictive values for all individuals in the array even though these
measures are diagnostic specific; i.e., PPV (NPV) should only be considered
for those individuals who have tested positive (negative).
}

\examples{
# Calculate the operating characteristics for 
#   non-informative array testing without master
#   pooling, with a 5x5 array and an overall disease 
#   risk of p = 0.02.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
p.mat <- matrix(data=0.02, ncol=5, nrow=5)
Array.Measures(p=p.mat, se=0.95, sp=0.95)

# Calculate the operating characteristics for 
#   informative array testing without master
#   pooling, with a 3x3 array and an overall disease
#   risk of p = 0.03 and alpha = 2.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
set.seed(8791)
p.vec <- p.vec.func(p=0.03, alpha=2, grp.sz=9)
p.mat <- Informative.array.prob(prob.vec=p.vec, nr=3, 
nc=3, method="gd")
Array.Measures(p=p.mat, se=0.99, sp=0.99)
}
\references{
\insertRef{McMahan2012b}{binGroup}
}
\seealso{
\code{\link{MasterPool.Array.Measures}} for calculating operating 
characteristics under non-informative array testing with master pooling, 
\code{\link{hierarchical.desc2}} for three-stage hierarchical and 
non-informative two-stage hierarchical testing, and 
\code{\link{inf.dorf.measures}} for informative two-stage hierarchical 
testing. See \code{\link{p.vec.func}} for generating a vector of 
individual risk probabilities for informative group testing and 
\code{\link{Informative.array.prob}} for arranging individual risk
probabilities in a matrix for informative array testing.  

\url{http://chrisbilder.com/grouptesting}

Other Operating characteristic functions: \code{\link{MasterPool.Array.Measures}},
  \code{\link{hierarchical.desc2}},
  \code{\link{inf.dorf.measures}}
}
\author{
This function was originally written by Christopher S. McMahan for 
McMahan et al. (2012). The function was obtained from 
\url{http://chrisbilder.com/grouptesting}.
}
