##############
## Latent Variable Ordination and Regression using MCMC 
## Site effects fitted as fixed effects
## NB parameterized as V = mu + phi*mu^2
## Ordinal data handled as propotional odds regression, with same cutoff points for all spp. 
## Multinomial regression is available, but CURRENTLY NOT ALLOWED DUE TO COMPUTATION TIME
## This should make sense if the same ordinal scale is applied to all species

## Changes from v0.4
## 1) site.eff => row.eff
## 2) boral.default: n.burnin = 5000; n.iteration = 35000
## 3) get.enviro.cor: Calculates correlation due to shared responses to model matrix X
## 4) family = exponential, gamma, beta added -- beta won't work if y = 0/1. Ad-hoc fix is to add a little bit, but dodgy. gamma parameterized in terms of shape and rate, and dispersion parameter returns rate
## 5) Stochastic search variable selection now implemented: -1 for no selection; 0 for selection of coefficients separately for a covariate; >= 1 for selection of coefficients collectively on one or more covariates. Prior variance of hyperparams[2]*((1-I)*0.0001+I), with prior probability of I being a Bernoulli prior with prob = 0.5
## 6) hypparams extended to four elements: 1) variance for all intercepts, row effects, ordinal cutoff points, 2) variance for all LV coefs, 3) variance for all X coefs, 4) max limit for dispersion parameters. This applies to both boral models with and without latent variables
## 7) get.hpdintervals includes prob argument
## 8) get.more.measures can now be run inside get.measures. Help files continue to exist for both
## 9) Syntax change to make.jagsboralmodel and make.jagsboralnullmodel to use num.X argument instead of X.eff; done to allow ssvs.index to be replicated

## Changes from v0.5
## 1) Default trial.size = 1 except for create.life
## 2) do.fit argument now available to creat JAGS model file but not actually fit the model
## 3) add n.thin, n.burnin, n.iteration to output from boral; 
## 4) corrected some potential errors which may arise the examples in help files, due to problems with the as.mcmc() function in later versions of the R2jags/coda packages
## 5) corrected estimation of the Compound Laplace-Metropolis estimator in get.measures
## 6) get.residual.cor now calculates the residual correlation based on cov2cor(lvs%*%t(lv.coefs))
## 7) Allows the option of custom naming the model file. 
## 8) Rename theta0 to beta0; 
## 9) redo get.enviro.cor and get.residual.cor to take boral fit directly rather than mcmc
## 10) allow row.eff as random effect, with compatability of old row.eff = TRUE and FALSE
## 11) reparameterize the NB using dgamma(1/phi, 1/phi)

## Changes from v0.6
## 1) correct issues for running JAGS when tweedie is used in combination with other responses, i.e., numfish other tweedie specific nodes goes bonkers when not all of family is tweedie
## 2) Tougher priors on the LV coefficients with variance = 20
## 3) Will try to explicitly capture the problem with negative-binomial when the prior on the overdispersion is too flat, and recommend that a tougher prior be used
## 4) for row.eff = "random" now draws from a normal dist with mean zero.
## 5) get.residual.cor now also returns the mean/median of the trace of the residual covariance matrix

## TODO: 1) correct problems with increasing number of LVs causing increase in covariance and variance i.e., scale problem. See also ms-LVMpaper for using logit regression in case of Bernoulli familly...maybe use a probit function instead?; 2) Find a way to do biplots after controlling for scaling; 3) Calculate proprtion of deviance explained based on marginal or conditional log-likelihood? 5) Allow options for weakly informative priors in terms t/Cauchy and half-t distributions; ; 7) Allow spp-specific coefficients to be regressed against traits? HARD!!! 8) draw coefficients as random effects, or reduce rank them? HARD!!!

##############
#  rm(list = ls())
#  library(R2jags); 
#  library(mvtnorm); 
#  library(mvabund); 
#  library(coda); 
#  library(MASS)
#  library(fishMod)
#  source("auxilaryfunctions.R")
# # # #  
#  data(spider)
#  y <- spider$abun
#  X <- spider$x; family = "negative.binomial"; num.lv = 0; row.eff = "fixed"; n.burnin = 4000; n.iteration = 24000; n.thin = 5; save.model = FALSE; calc.ics = TRUE; trial.size <- NULL; seed <- 123; hypparams = c(100,100,100,100); ssvs.index <- -1; ssvs.index <- c(-1,1,2,1,0,-1)

#library(ade4)
#data(dunedata)
#y = dunedata$veg+1 ## Shift levels up to start at 1
#X = model.matrix(~A1 + factor(use) - 1, data = dunedata$envir)
#family = rep("ordinal",30)
#num.lv = 2; row.eff = "none"; n.burnin = 4000; n.iteration = 24000; n.thin = 4; save.model = TRUE; seed = 7; calc.ics = TRUE; trial.size = NULL; num.ord.levels <- 5; hypparams = c(100,100,100,100); 
#X <- matrix(rnorm(30*4),30,4)
#true.beta <- cbind(matrix(rnorm(length(family)*(ncol(X)+1)),length(family),ncol(X)+1),NA); 
#true.beta[nrow(true.beta),1] <- -sum(true.beta[-nrow(true.beta),1])
#true.ordinal <- seq(-0.5,0.5,length=num.ord.levels-1)
#y <- create.life(lv.coefs = true.beta[,c(1,ncol(true.beta))], X = X, X.coefs = true.beta[,-c(1,ncol(true.beta))], family = family, cutoffs = true.ordinal)

# n = 60; p <- 30
# X <- matrix(rnorm(n*2),n,2); beta <- cbind(matrix(rnorm(p*3),p,3),runif(p,0,5)); true.power <- 1.6
# mu <- exp(cbind(1,X)%*%t(beta[,1:3]))
# y <- matrix(NA,n,p)
# for(j in 1:ncol(y)) { y[,j] <- rTweedie(nrow(y), mu = mu[,j], phi = beta[j,4], p = true.power) }
# family = "tweedie"
# num.lv = 0; row.eff = "none"; n.burnin = 4000; n.iteration = 24000; n.thin = 4; save.model = TRUE; seed = 1; calc.ics = TRUE; trial.size = NULL; hypparams = c(100,100,100,100); 
# 
# library(FD)
# data(tussock)
# y <- tussock$trait[,c("height","LDMC","leafN","leafsize","SLA","seedmass","clonality","resprouting","lifespan")]
# y$LDMC <- y$LDMC/1000 ## change to g/g
# y$leafsize <- y$leafsize/100 ## change to cm^2
# y[,"resprouting"] <- as.numeric(y[,"resprouting"])-1 ## 0 = no; 1 = yes
# levels(y[,"clonality"]) = c(3,2,1)
# y[,"clonality"] <- as.numeric(levels(y[,"clonality"]))[y[,"clonality"]]
# levels(y[,"lifespan"]) = c(0,0,1)
# y[,"lifespan"] <- as.numeric(levels(y[,"lifespan"]))[y[,"lifespan"]]
# y <- y[-which(is.na(rowSums(y))),]
# 
# family = c("lnormal","normal","normal","lnormal","lnormal","lnormal","ordinal","binomial","binomial")
# num.lv = 2; row.eff = "none"; n.burnin = 4000; n.iteration = 24000; n.thin = 4; save.model = TRUE; seed = 123; calc.ics = TRUE; trial.size = 1; hypparams = c(100,100,100,100); X <- NULL

# n = 30; s <- 30; num.multinom.levels <- 4
# X <- matrix(rnorm(n*2),n,2)
# X.coefs <- rbind(matrix(0,s-1,2),c(1,2))
# X.multinom.coefs <- array(NA,dim=c(s-1,2,num.multinom.levels))
# for(k in 1:num.multinom.levels) { X.multinom.coefs[,,k] <- rnorm((s-1)*2) }
# row.coefs <- runif(n)
# lv.coefs <- cbind(matrix(runif(s,-3,-1),s,1),2)
# family = c(rep("multinom",s-1),"normal")
# num.lv = 2; row.eff = "fixed"; n.burnin = 4000; n.iteration = 24000; n.thin = 4; save.model = TRUE; seed = 1; calc.ics = TRUE; trial.size = 1; hypparams = c(100,100,100,100); 
# y <- create.life(lv.coefs = lv.coefs, X = X, X.coefs = X.coefs, X.multinom.coefs = X.multinom.coefs, family = family, row.coefs = row.coefs)

boral <- function(y, ...) UseMethod("boral")

## Model is g(mu_{ij}) = row + beta0 + LV_i*theta_j + X_i*beta_j
boral.default <- function (y, X = NULL, family, trial.size = 1, num.lv = 0, row.eff = "none", n.burnin = 5000, n.iteration = 35000, n.thin = 5, save.model = FALSE, seed = 123, calc.ics = TRUE, hypparams = c(100, 20, 100, 100), ssvs.index = -1, do.fit = TRUE, model.name = NULL, ...) {
	if(is.null(dim(y))) { cat("Converting y into a one column matrix.\n"); y <- matrix(y, ncol = 1) }
	if(!is.null(X) & is.null(dim(X))) { cat("Converting X into a one column matrix\n"); X <- matrix(X, ncol = 1) }
	if(length(hypparams) != 4) { stop("hypparams must be a vector of four elements. Please see boral help file as to what the elements correspond to.\n") }
    
	if(num.lv == 1) warnings("We won't stop you, but one latent variable is unlikely to be successful in capturing between column correlation!")
	if(num.lv > 5) warnings("We won't stop you, but please consider if you really want more than five latent variables in the model!")
	
	if(length(family) != ncol(y) & length(family) != 1) { stop("Number of elements in family must either one or the # of columns in y") }
	if(length(family) == 1) family <- rep(family, ncol(y))
	if(!all(family %in% c("negative.binomial", "poisson", "binomial", "normal", "lnormal", "tweedie", "ordinal", "exponential", "gamma", "beta"))) 
		stop("At least one of the elements in family is not compatible with current version of boral...sorry!")
	if(any(family == "ordinal")) {
		if(sum(y[, family == "ordinal"] == 0) > 0) stop("For ordinal data, please shift minimum level to 1.")
		print("It is assumed all ordinal columns have the same number of levels -- please see help file as to the motivation behind this.")
		print("boral may take a ridiculously long time to fit ordinal data models. Apolgoies in advance!\n") }
	if(any(family == "binomial")) {
		if (!all(unlist(y[, family == "binomial"]) %in% c(0, 1))) stop("Any columns of y specified to be binomial must only contain 0/1 elements") }
	
	if(row.eff == FALSE) row.eff <- "none"; if(row.eff == TRUE) row.eff <- "fixed"
	if(!(row.eff %in% c("none", "fixed", "random"))) stop("row.eff must be one of none/fixed/random.")
	
	if(!is.null(X)) { X.eff <- TRUE } else { X.eff <- FALSE }
	X.eff <- as.numeric(X.eff)
	if(!X.eff) { num.X <- 0 } else { num.X <- ncol(X) }

	if(!(length(ssvs.index) %in% c(1, ncol(X)))) stop("Number of elements in ssvs.index must either be one or the # of columns in X.")
	if(length(ssvs.index) == 1 & X.eff) ssvs.index <- rep(ssvs.index, ncol(X))
	if(any(ssvs.index < -1)) stop("Elements of ssvs.index can only take values in -1, 0, or any positive integer; please see help file for guide.")
	if(any(family == "binomial") & !(length(trial.size) %in% c(1, length(family)))) 
        stop("trial.size needs to be specified if any columns are binomially distributed; can either be a single element or a vector equal to the # of columns in y. The latter will assume the specified trial size for all rows labelled binomial in the family argument.")
	if(any(family == "binomial") & length(trial.size) == 1) {
		complete.trial.size <- rep(0, ncol(y))
		complete.trial.size[which(family == "binomial")] <- trial.size }
	if(any(family == "binomial") & length(trial.size) == ncol(y)) { complete.trial.size <- trial.size }
	if(all(family != "binomial")) { complete.trial.size <- rep(0, ncol(y)) }
	
	if(all(family != "ordinal")) {
        num.ord.levels <- 0; }
	if(any(family == "ordinal")) {
		num.ord.levels <- max(y[, family == "ordinal"]); }
	if(all(family != "multinom")) {
		num.multinom.levels <- 0; index.multinom.cols <- NULL }
	if(any(family == "multinom")) { 
		num.multinom.levels <- apply(y[, family == "multinom"], 2, max)
		index.multinom.cols <- which(family == "multinom") }

	n <- nrow(y); p <- ncol(y)
 	n.chains = 1; ## Run one chain only to avoid arbitrary rotation problems
	if(num.lv > 0) make.jagsboralmodel(family, num.X, row.eff, n, p, hypparams, ssvs.index, model.name)
	if(num.lv == 0)  make.jagsboralnullmodel(family, num.X, row.eff, n, p, hypparams, ssvs.index, model.name)
 	if(!do.fit) { 
		cat("JAGS model file created only. Thank you, come again!\n")
		break; }

	jags.data <- list("y", "n", "p", "num.lv", "num.X", "complete.trial.size", "num.ord.levels", "num.multinom.levels")
	if(X.eff) jags.data <- c("X", jags.data)
	if(any(family == "ordinal")) {
		ones <- matrix(1, n, p); jags.data <- c(jags.data, "ones") }
	
	jags.params <- c("all.params")
	if(num.lv > 0) jags.params <- c(jags.params, "lvs")
	if(row.eff != "none") jags.params <- c(jags.params, "row.params")
	if(row.eff == "random") jags.params <- c(jags.params, "row.ranef.sigma2")
	if(X.eff & any(family != "multinom")) jags.params <- c(jags.params, "X.params")
	if(X.eff & any(family == "multinom")) jags.params <- c(jags.params, "X.multinom.params")
	if(any(family == "tweedie")) jags.params <- c(jags.params, "powerparam")
	if(any(family == "ordinal")) jags.params <- c(jags.params, "alpha")
	if(any(ssvs.index == 0)) jags.params <- c(jags.params, paste("probindX", which(ssvs.index == 0), sep = ""))
	if(any(ssvs.index > 0)) jags.params <- c(jags.params, paste("probGpX", unique(ssvs.index[ssvs.index > 0]), sep = ""))
	
	jags.inits <- NULL
	if(any(family %in% "tweedie")) {
		initial.list <- list(numfish = matrix(1, n, sum(family=="tweedie")))
		if(any(family %in% "ordinal")) initial.list$alpha0 <- seq(-1, 1, length = num.ord.levels - 1)
		jags.inits <- function() { return(initial.list) }
		}
    set.seed(seed)

    actual.filename <- model.name
    if(is.null(actual.filename)) actual.filename <- "jagsboralmodel.txt"

    jagsfit <- try(suppressWarnings(jags(data = jags.data, inits = jags.inits, jags.params, model.file = actual.filename, n.iter = n.iteration, n.burnin = n.burnin, n.thin = n.thin, n.chains = n.chains, DIC = T)),silent=T)
    
    
	if(inherits(jagsfit,"try-error")) {
		lookfornberror <- grep("Slicer stuck at value with infinite density", jagsfit[[1]])
		if(any(family == "negative.binomial") & lookfornberror == TRUE) { 
			cat("MCMC fitting through JAGS failed. This is likely due to the the prior on the overdispersion parameter in the negative binomial been too `flat'. Please consider a tougher prior, e.g. hypparams[4] = 20, or switch to a Poisson family for those response that don't appear to actually be overdispersed. The error below informs you explicity which column of y the MCMC sampling ran into trouble. For instance, all.params[24,4] suggests that column 24 of y is causing issues.\n")
			print(jagsfit) }

		else {
			cat("MCMC fitting through JAGS failed:\n")
			print(jagsfit) }

		cat("boral fit failed...Exiting. Sorry!\n") 
		return()
		}
    
    
	## Format into big matrix; also deals with the conflict of as.mcmc converting to lists or the MCMC samples themselves
	fit.mcmcBase <- jagsfit$BUGSoutput
	fit.mcmc <- mcmc(fit.mcmcBase$sims.matrix, start = 1, thin = n.thin) ## Thanks to Guilliaume Blanchet for this!
	if(n.chains == 1) combined.fit.mcmc <- fit.mcmc
	rm(fit.mcmc)
#	if(length(as.mcmc(jagsfit)) > 1) fit.mcmc <- as.mcmc(jagsfit) 
# 	if(n.chains > 1) {
# 		for(k in 1:n.chains) {
# 			if(k == 1) { combined.fit.mcmc <- fit.mcmc[[1]]; fit.mcmc[[1]] <- NA }
# 			if(k > 1) { combined.fit.mcmc <- rbind(combined.fit.mcmc, fit.mcmc[[k]]); fit.mcmc[[k]] <- NA }
# 			} }
    
    
 	## Flip dispersion parameters for NB returns phi_j, for normal and lognormal return sigma^2_j
	sel.thetas <- grep("all.params", colnames(combined.fit.mcmc))
	sel.thetas2 <- as.numeric(sel.thetas[(length(sel.thetas) - p + 1):length(sel.thetas)])
	combined.fit.mcmc[, sel.thetas2] <- 1/combined.fit.mcmc[, sel.thetas2]
	if(any(family %in% c("tweedie", "beta", "gamma","negative.binomial"))) 
		combined.fit.mcmc[, sel.thetas2[family %in% c("tweedie", "beta", "gamma")]] <- 1/combined.fit.mcmc[, sel.thetas2[family %in% c("tweedie", "beta", "gamma","negative.binomial")]] ## Flip it back for tweedie, beta, gamma (parameterization in terms of shape and rate)
	if(any(family %in% c("poisson", "binomial", "ordinal", "multinom", "exponential"))) 
		combined.fit.mcmc[, sel.thetas2[family %in% c("poisson", "binomial", "ordinal", "multinom", "exponential")]] <- 0

		
  	## For any multinomial columns, set the corresponding rows in X.coefs to zero
	if(any(family == "multinom") & X.eff == 1) {
		for(k in index.multinom.cols) {
			sel.multinom.col <- grep(paste("X.params\\[", k, ",+", sep = ""), colnames(combined.fit.mcmc))
			combined.fit.mcmc[, sel.multinom.col] <- 0 }
		}

		
 	## Make output beautiful
	if(is.null(colnames(y))) colnames(y) <- 1:ncol(y)
	if(is.null(rownames(y))) rownames(y) <- 1:nrow(y)
	if(X.eff) {
		if(is.null(colnames(X))) colnames(X) <- 1:ncol(X)
		if(is.null(rownames(X))) rownames(X) <- 1:nrow(X) }

		
	out.fit <- list(lv.coefs.median = matrix(apply(combined.fit.mcmc[, grep("all.params", colnames(combined.fit.mcmc))], 2, median), nrow = p), lv.coefs.iqr = matrix(apply(combined.fit.mcmc[, grep("all.params", colnames(combined.fit.mcmc))], 2, IQR), nrow = p), lv.coefs.mean = matrix(apply(combined.fit.mcmc[, grep("all.params", colnames(combined.fit.mcmc))], 2, mean), nrow = p), lv.coefs.sd = matrix(apply(combined.fit.mcmc[, grep("all.params", colnames(combined.fit.mcmc))], 2, sd), nrow = p))

	rownames(out.fit$lv.coefs.median) <- rownames(out.fit$lv.coefs.iqr) <- rownames(out.fit$lv.coefs.mean) <- rownames(out.fit$lv.coefs.sd) <- colnames(y)

	
	if(num.lv > 0) { 
		out.fit$lv.median = matrix(apply(combined.fit.mcmc[, grep("lvs", colnames(combined.fit.mcmc))], 2, median), nrow = n)
		out.fit$lv.iqr = matrix(apply(combined.fit.mcmc[, grep("lvs", colnames(combined.fit.mcmc))], 2, IQR), nrow = n)
		out.fit$lv.mean = matrix(apply(combined.fit.mcmc[, grep("lvs", colnames(combined.fit.mcmc))], 2, mean), nrow = n)
		out.fit$lv.sd = matrix(apply(combined.fit.mcmc[, grep("lvs", colnames(combined.fit.mcmc))], 2, sd), nrow = n)
		rownames(out.fit$lv.median) <- rownames(out.fit$lv.iqr) <- rownames(out.fit$lv.mean) <- rownames(out.fit$lv.sd) <- rownames(y)
		colnames(out.fit$lv.median) <- colnames(out.fit$lv.iqr) <- colnames(out.fit$lv.mean) <- colnames(out.fit$lv.sd) <- paste("LV", 1:num.lv, sep = "")
		colnames(out.fit$lv.coefs.median) <- colnames(out.fit$lv.coefs.iqr) <- colnames(out.fit$lv.coefs.mean) <- colnames(out.fit$lv.coefs.sd) <- c("beta0", 
		paste("theta", 1:num.lv, sep = ""), "Dispersion") 
		}
	if(num.lv == 0) {
		colnames(out.fit$lv.coefs.median) <- colnames(out.fit$lv.coefs.iqr) <- colnames(out.fit$lv.coefs.mean) <- colnames(out.fit$lv.coefs.sd) <- c("beta0", "Dispersion") }
	
	
	if(row.eff != "none") {
		out.fit$row.coefs.median <- apply(combined.fit.mcmc[, grep("row.params", colnames(combined.fit.mcmc))], 2, median)
		out.fit$row.coefs.iqr <- apply(combined.fit.mcmc[, grep("row.params", colnames(combined.fit.mcmc))], 2, IQR)
		out.fit$row.coefs.mean <- apply(combined.fit.mcmc[, grep("row.params", colnames(combined.fit.mcmc))], 2, mean)
		out.fit$row.coefs.sd <- apply(combined.fit.mcmc[, grep("row.params", colnames(combined.fit.mcmc))], 2, sd)
		
		names(out.fit$row.coefs.median) <- names(out.fit$row.coefs.iqr) <- names(out.fit$row.coefs.mean) <- names(out.fit$row.coefs.sd) <- rownames(y)
	
		if(row.eff == "random") {
			out.fit$row.ranef.median <- median(combined.fit.mcmc[, grep("row.ranef.sigma2", colnames(combined.fit.mcmc))])
			out.fit$row.ranef.iqr <- IQR(combined.fit.mcmc[, grep("row.ranef.sigma2", colnames(combined.fit.mcmc))])
			out.fit$row.ranef.mean <- mean(combined.fit.mcmc[, grep("row.ranef.sigma2", colnames(combined.fit.mcmc))])
			out.fit$row.ranef.sd <- sd(combined.fit.mcmc[, grep("row.ranef.sigma2", colnames(combined.fit.mcmc))])
            
			names(out.fit$row.ranef.median) <- names(out.fit$row.ranef.iqr) <- names(out.fit$row.ranef.mean) <- names(out.fit$row.ranef.sd) <- c("Row random effects variance") }
		}

		
	if(X.eff) {
		out.fit$X.coefs.median <- matrix(apply(combined.fit.mcmc[, grep("X.params", colnames(combined.fit.mcmc))], 2, median), nrow = p)
		out.fit$X.coefs.iqr <- matrix(apply(combined.fit.mcmc[, grep("X.params", colnames(combined.fit.mcmc))], 2, IQR), nrow = p)
		out.fit$X.coefs.mean <- matrix(apply(combined.fit.mcmc[, grep("X.params", colnames(combined.fit.mcmc))], 2, mean), nrow = p)
		out.fit$X.coefs.sd <- matrix(apply(combined.fit.mcmc[, grep("X.params", colnames(combined.fit.mcmc))], 2, sd), nrow = p)
		rownames(out.fit$X.coefs.median) <- rownames(out.fit$X.coefs.iqr) <- rownames(out.fit$X.coefs.mean) <- rownames(out.fit$X.coefs.sd) <- colnames(y)
		colnames(out.fit$X.coefs.median) <- colnames(out.fit$X.coefs.iqr) <- colnames(out.fit$X.coefs.mean) <- colnames(out.fit$X.coefs.sd) <- colnames(X)
		if(any(ssvs.index == 0)) {
			out.fit$ssvs.indcoefs.mean <- matrix(apply(combined.fit.mcmc[, grep("probindX", colnames(combined.fit.mcmc))], 
                2, mean), nrow = p)
			rownames(out.fit$ssvs.indcoefs.mean) <- colnames(y)
			colnames(out.fit$ssvs.indcoefs.mean) <- colnames(X)[which(ssvs.index == 0)]
			out.fit$ssvs.indcoefs.sd <- matrix(apply(combined.fit.mcmc[, grep("probindX", colnames(combined.fit.mcmc))], 2, sd), nrow = p)
			rownames(out.fit$ssvs.indcoefs.sd) <- colnames(y)
			colnames(out.fit$ssvs.indcoefs.sd) <- colnames(X)[which(ssvs.index == 0)]
			}
		if(any(ssvs.index > 0)) {
			out.fit$ssvs.gpcoefs.mean <- apply(as.matrix(combined.fit.mcmc[, grep("probGpX", colnames(combined.fit.mcmc))]), 2, mean)
			names(out.fit$ssvs.gpcoefs.mean) <- paste("Gp", unique(ssvs.index[ssvs.index > 0]), sep = "")
			out.fit$ssvs.gpcoefs.sd <- apply(as.matrix(combined.fit.mcmc[, grep("probGpX", colnames(combined.fit.mcmc))]), 2, sd)
			names(out.fit$ssvs.gpcoefs.sd) <- paste("Gp", unique(ssvs.index[ssvs.index > 0]), sep = "")
			}
		}
		
#   	if(X.eff & any(family == "multinom")) {
#   		out.fit$X.multinom.coefs.median <- array(apply(combined.fit.mcmc[,grep("X.multinom.params", colnames(combined.fit.mcmc))],2,median),dim=c(length(index.multinom.cols),num.X,num.multinom.levels))
#   		out.fit$X.multinom.coefs.iqr <- array(apply(combined.fit.mcmc[,grep("X.multinom.params", colnames(combined.fit.mcmc))],2,IQR),dim=c(length(index.multinom.cols),num.X,num.multinom.levels))
#   		out.fit$X.multinom.coefs.mean <- array(apply(combined.fit.mcmc[,grep("X.multinom.params", colnames(combined.fit.mcmc))],2,mean),dim=c(length(index.multinom.cols),num.X,num.multinom.levels))
#   		out.fit$X.multinom.coefs.sd <- array(apply(combined.fit.mcmc[,grep("X.multinom.params", colnames(combined.fit.mcmc))],2,sd),dim=c(length(index.multinom.cols),num.X,num.multinom.levels))
#   
#   		dimnames(out.fit$X.multinom.coefs.median) <- dimnames(out.fit$X.multinom.coefs.iqr) <- dimnames(out.fit$X.multinom.coefs.mean) <- dimnames(out.fit$X.multinom.coefs.sd) <- list("1" = index.multinom.cols, "2" = colnames(X), "level" = 1:num.multinom.levels)
#   		}

	if(any(family == "ordinal")) {
		out.fit$cutoffs.median <- apply(combined.fit.mcmc[, grep("alpha", colnames(combined.fit.mcmc))], 2, median)
		out.fit$cutoffs.iqr <- apply(combined.fit.mcmc[, grep("alpha", colnames(combined.fit.mcmc))], 2, IQR)
		out.fit$cutoffs.mean <- apply(combined.fit.mcmc[, grep("alpha", colnames(combined.fit.mcmc))], 2, mean)
		out.fit$cutoffs.sd <- apply(combined.fit.mcmc[, grep("alpha", colnames(combined.fit.mcmc))], 2, sd)
		names(out.fit$cutoffs.median) <- names(out.fit$cutoffs.iqr) <- names(out.fit$cutoffs.mean) <- names(out.fit$cutoffs.sd) <- paste(1:(num.ord.levels - 1), "|", 2:num.ord.levels, sep = "") }

	if(any(family == "tweedie")) {
		out.fit$powerparam.median <- median(combined.fit.mcmc[, grep("powerparam", colnames(combined.fit.mcmc))])
		out.fit$powerparam.iqr <- IQR(combined.fit.mcmc[, grep("powerparam", colnames(combined.fit.mcmc))])
		out.fit$powerparam.mean <- mean(combined.fit.mcmc[, grep("powerparam", colnames(combined.fit.mcmc))])
		out.fit$powerparam.sd <- sd(combined.fit.mcmc[, grep("powerparam", colnames(combined.fit.mcmc))])
		names(out.fit$powerparam.median) <- names(out.fit$powerparam.iqr) <- names(out.fit$powerparam.mean) <- names(out.fit$powerparam.sd) <- "Common power parameter"
		}

	get.hpds <- get.hpdintervals(y, X, combined.fit.mcmc, num.lv)
	out.fit$hpdintervals <- get.hpds
	if(calc.ics) {
		cat("Calculating Information criteria\n")
		get.ics <- get.measures(y, X, family, complete.trial.size, row.eff, num.lv, combined.fit.mcmc, more.measures = FALSE)
		ics <- c(get.dic(jagsfit), get.ics$waic, get.ics$eaic, get.ics$ebic, get.ics$aic.median, get.ics$bic.median, get.ics$comp.lm)
		names(ics) <- c("Conditional DIC", "WAIC", "EAIC", "EBIC", "AIC at post. median", "BIC at post. median", "Compound L-M at post. median")
		out.fit$ics <- ics
		}
			
	if(save.model) { out.fit$jags.model <- jagsfit }

	out.fit$call <- match.call()
	out.fit$n <- n
	out.fit$p <- p
	if(!is.null(X)) out.fit$X <- X
	if(is.null(X)) out.fit$X <- NULL
	out.fit$y <- y
	out.fit$family <- family
	out.fit$num.lv <- num.lv
	out.fit$num.X <- num.X
	out.fit$row.eff <- row.eff
	out.fit$calc.ics <- calc.ics
	out.fit$trial.size <- complete.trial.size
	out.fit$hypparams <- hypparams
	out.fit$ssvs.index <- ssvs.index
	out.fit$num.ord.levels <- num.ord.levels
	out.fit$n.burnin <- n.burnin
	out.fit$n.thin <- n.thin
	out.fit$n.iteration <- n.iteration
	
	class(out.fit) <- "boral"
	if(!save.model) { if(file.exists(actual.filename)) file.remove(actual.filename) }

	return(out.fit) }
 	

 	
################	
lvsplot <- function(x, jitter = FALSE, a = 1, newplot = TRUE, biplot = TRUE, ind.spp = NULL, ...) {
 	if(x$num.lv > 2) stop("Manual plotting required for plotting beyond 2 latent variables")
 	if(x$num.lv == 0) stop("No latent variables to plot.")
 
 	n <- nrow(x$lv.median); p <- nrow(x$lv.coefs.median)
 	if(!is.null(ind.spp)) { if(ind.spp > p) { ind.spp <- p } }
	if(biplot == TRUE & !is.null(ind.spp)) { cat("Only the first", ind.spp, "`most important' latent variable coefficients included in biplot\n") }
	if(biplot == TRUE & is.null(ind.spp)) { ind.spp <- p; cat("All latent variable coefficients included in biplot\n") }

 	if(newplot == TRUE)
 		par(cex = a, cex.axis = a, cex.lab = a+0.5, mar = c(5,5,3,1), mfrow = c(1,2), cex.main = a+0.5, ...) 
 	if(newplot == FALSE)
 		par(cex = a, cex.axis = a, cex.lab = a+0.5, mar = c(5,5,3,1), cex.main = a+0.5, ...) 
 
 	if(x$num.lv == 1) {
 		plot(1:n, x$lv.median, xlab = "Row index", ylab = "Latent variable 1", main = "Plot of the latent variable posterior medians", cex = 1.2*a, type = "n", ...)
 		text(x = 1:n, y = x$lv.median, label = 1:n, cex = 1.2*a)

 		plot(1:n, x$lv.mean, xlab = "Row index", ylab = "Latent variable 1", main = "Plot of the latent variable posterior means", cex = 1.2*a, type = "n", ...)
 		text(x = 1:n, y = x$lv.mean, label = 1:n, cex = 1.2*a) 
 		}	


 	if(x$num.lv == 2) {
		## Scale by L2norms
		x$lv.median2 <- scale(x$lv.median,center=T,scale=F)#*matrix(sqrt(colSums(x$lv.coefs.median[,2:3]^2))/sqrt(colSums(x$lv.median^2)),n,2,byrow=T) 
		x$lv.coefs.median2 <- scale(x$lv.coefs.median[,2:3]*matrix(sqrt(colSums(x$lv.median2^2))/sqrt(colSums(x$lv.coefs.median[,2:3]^2)),p,2,byrow=T),center=T,scale=F) 

		if(!biplot) {
			plot(x$lv.median, xlab = "Latent variable 1", ylab = "Latent variable 2", main = "Plot of the latent variable posterior medians", cex = 1.2*a, type = "n", ...)
			if(!jitter) text(x$lv.median, label = 1:n, cex = 1.2*a)
			if(jitter) text(jitter(x$lv.median[,1]), jitter(x$lv.median[,2]), label = 1:n, cex = 1.2*a)
			}

		if(biplot) {
			largest.lnorms <- order(rowSums(x$lv.coefs.median2^2),decreasing=T)[1:ind.spp]
			plot(rbind(x$lv.median2,x$lv.coefs.median2), xlab = "Latent variable 1", ylab = "Latent variable 2", main = "Biplot of the latent variable and loadings posterior medians", cex = a, type = "n")
			if(!jitter) text(x$lv.median2, label = 1:n, cex = 1.2*a)
			if(jitter) text(jitter(x$lv.median2[,1]), jitter(x$lv.median2[,2]), label = 1:n, cex = 1.2*a)
			text(x$lv.coefs.median2[largest.lnorms,], label = rownames(x$lv.coefs.mean[largest.lnorms,]), col = "red", cex = 0.9*a)			
			}


		## Scale by L2norms
		x$lv.mean2 <- scale(x$lv.mean,center=T,scale=F) #*matrix(sqrt(colSums(x$lv.coefs.mean[,2:3]^2))/sqrt(colSums(x$lv.mean^2)),n,2,byrow=T) 
		x$lv.coefs.mean2 <- scale(x$lv.coefs.mean[,2:3]*matrix(sqrt(colSums(x$lv.mean2^2))/sqrt(colSums(x$lv.coefs.mean[,2:3]^2)),p,2,byrow=T),center=T,scale=F) 

		if(!biplot) {
			plot(x$lv.mean, xlab = "Latent variable 1", ylab = "Latent variable 2", main = "Biplot of the latent variable posterior means", cex = 1.2*a, type = "n", ...)
			if(!jitter) text(x$lv.mean, label = 1:n, cex = 1.2*a)
			if(jitter) text(jitter(x$lv.mean[,1]), jitter(x$lv.mean[,2]), label = 1:n, cex = 1.2*a)
			}

		if(biplot) {
			largest.lnorms <- order(rowSums(x$lv.coefs.mean2^2),decreasing=T)[1:ind.spp]
			plot(rbind(x$lv.mean2,x$lv.coefs.mean2), xlab = "Latent variable 1", ylab = "Latent variable 2", main = "Plot of the latent variable posterior means", cex = 1.2*a, type = "n")
			if(!jitter) text(x$lv.mean2, label = 1:n, cex = 1.2*a)
			if(jitter) text(jitter(x$lv.mean2[,1]), jitter(x$lv.mean2[,2]), label = 1:n, cex = 1.2*a)
			text(x$lv.coefs.mean2[largest.lnorms,], label = rownames(x$lv.coefs.mean[largest.lnorms,]), col = "red", cex = 0.9*a)
			}
 		}	

 	}


print.boral <- function(x, ...) {
 	cat("Call:\n")
 	print(x$call)
 	cat("\n")
 	cat("Response matrix attributes\n \t# of rows:", x$n, "\n\t# of columns:", x$p, "\n") 
 	cat("Model attributes\n \tColumn families:", x$family, "\n\t# of latent variables:", x$num.lv, "\n\tRow effect included (none/fixed/random)?", x$row.eff, "\n") 
 	if(any(x$family == "binomial")) cat("Trial sizes used (columns with binomial families):", x$trial.size,"\n")
 	if(any(x$family == "ordinal")) cat("Number of levels for ordinal data:", x$num.ord.levels,"\n")
 	#cat("Hyperparameters (variance in normal priors of coefficients, maximum in uniform prior for dispersion):", x$hypparams, "\n") 
 	if(x$num.X > 0) cat("Model matrix with", x$num.X, "covariates also fitted\n\n")
 	if(any(x$ssvs.index > -1)) cat("SSVS performed on covariates with indices", x$ssvs.index, "\n\n")
# 	cat("Output attributes\n")
# 	print(attributes(x))
 	}
	
print.summary.boral <- function(x, ...) {
 	cat("Call:\n")
 	print(x$call)
 	cat("\n")
 	
 	if(x$est == "median") { cat("Median point estimates\n\n LV coefficients (thetas) and dispersion parameter\n"); print(x$coefficients); cat("\n") }
 	if(x$est == "mean") { cat("Mean point estimates\n\n LV coefficients (thetas) and dispersion parameter\n"); print(x$coefficients); cat("\n") }	
 	
 	if(!is.null(x$row.coefficients)) { cat("Row coefficients\n"); print(x$row.coefficients); cat("\n") }
 	if(!is.null(x$X.coefficients)) { cat("X coefficients (betas)\n"); print(x$X.coefficients); cat("\n") }
 	if(!is.null(x$X.multinom.coefficients)) { cat("There are also coefficients corresponding to multinomial columns which have not been printed"); }
 	
 	if(any(x$family == "ordinal")) { cat("Proportional odds (Cumulative logit) regression intercepts\n"); print(x$cutoffs); cat("\n") }
 	if(any(x$family == "tweedie")) { cat("Tweedie power parameter \n"); print(x$powerparam); cat("\n") }
 
 	if(x$calc.ics) {
 		cat("DIC (pD = var(deviance)/2):", as.vector(unlist(x$ics[1])), "\n")
 		cat("WAIC:", as.vector(unlist(x$ics[2])), "\n")
 		cat("EAIC:", as.vector(unlist(x$ics[3])), "\n")
 		cat("EBIC:", as.vector(unlist(x$ics[4])), "\n")
 		cat("AIC at posterior median:", as.vector(unlist(x$ics[5])), "\n")
 		cat("BIC at posterior median:", as.vector(unlist(x$ics[6])), "\n")
 		cat("Compound Laplace-Metropolis estimator at posterior median:", as.vector(unlist(x$ics[7])), "\n")
		}

	if(!is.null(x$ssvs.indcoefs.prob)) { cat("SSVS probabilities on individual coefficients\n"); print(x$ssvs.indcoefs.prob); cat("\n") }
	if(!is.null(x$ssvs.gpcoefs.prob)) { cat("SSVS probabilities on groups of coefficients\n"); print(x$ssvs.gpcoefs.prob); cat("\n") }			
	}	
		
summary.boral <- function(object, est = "median", ...) {
	if(est == "median") {
 		gather.output <- list(call = object$call, coefficients = round(object$lv.coefs.med,3))
 		if(object$row.eff == TRUE) gather.output$row.coefficients = round(object$row.coefs.med,3)
 		if(object$num.X > 0) gather.output$X.coefficients = round(object$X.coefs.med,3)
 		if(any(object$family == "ordinal")) gather.output$cutoffs = round(object$cutoffs.median,3)
 		if(any(object$family == "tweedie")) gather.output$powerparam = round(object$powerparam.median,3)
 		if(!is.null(object$X.multinom.coefs.median)) gather.output$X.multinom.coefficients = round(object$X.multinom.coefs.median,3) }
 
 	if(est == "mean") {
 		gather.output <- list(call = object$call, coefficients = round(object$lv.coefs.mean,3))
 		if(object$row.eff == TRUE) gather.output$row.coefficients = round(object$row.coefs.mean,3)
 		if(object$num.X > 0) gather.output$X.coefficients = round(object$X.coefs.mean,3)
 		if(any(object$family == "ordinal")) gather.output$cutoffs = round(object$cutoffs.mean,3)
 		if(any(object$family == "tweedie")) gather.output$powerparam = round(object$powerparam.mean,3)
 		if(!is.null(object$X.multinom.coefs.mean)) gather.output$X.multinom.coefficients = round(object$X.multinom.coefs.mean,3) }
 
 	gather.output$est <- est
 	gather.output$calc.ics <- object$calc.ics
	gather.output$trial.size <- object$trial.size
 	gather.output$num.ord.levels <- object$num.ord.levels
 	gather.output$ssvs.index <- object$ssvs.index
 	
	if(any(object$ssvs.index == 0)) gather.output$ssvs.indcoefs.prob = round(object$ssvs.indcoefs.mean,3)
	if(any(object$ssvs.index > 0)) gather.output$ssvs.gpcoefs.prob = round(object$ssvs.gpcoefs.mean,3) 

	if(object$calc.ics) gather.output$ics = object$ics
 	class(gather.output) <- "summary.boral"
 	gather.output 
 	}
 			
 			
plot.boral <- function(x, est = "median", jitter = FALSE, a = 1, ...) {
 	if(all(x$family %in% c("ordinal","multinom"))) 
 		stop("Residuals are not defined, and therefore residual analysis cannot be performed, if all columns of y are ordinal")
 	get.mus <- fitted.boral(x, est = est)$out
 	get.etas <- get.mus
 	get.ds.res <- ds.residuals(object = x, est = est)$residuals
 	for(j in 1:ncol(x$y)) {
 		if(x$family[j] %in% c("binomial","beta")) get.etas[,j] <- log((get.mus[,j]+1e-5)/(1-get.mus[,j]+1e-5))
 		if(x$family[j] %in% c("poisson","lnormal","negative.binomial","tweedie","exponential")) get.etas[,j] <- log(get.mus[,j]+1e-5)
 		if(x$family[j] == "normal") get.etas[,j] <- (get.mus[,j]) }
 
 	par(ask = T, cex = a, mar = c(5,5,1,1), cex.main = a) 
 	palette(rainbow(ncol(get.etas)))
 
 	matplot(get.etas, get.ds.res, ylab = "Dunn-Smyth Residuals", xlab = "Linear Predictors", type="n")
 	for(i in 1:ncol(get.etas)) { points(get.etas[,i], get.ds.res[,i], col=palette()[i]) }
 	abline(h=0, lty = 2, lwd = 2)
	
# 	matplot(get.mus, get.ds.res, ylab = "Dunn-Smyth Residuals", xlab = "Fitted Values", type="n")
# 	for(i in 1:ncol(get.mus)) { points(get.mus[,i], get.ds.res[,i], col=palette()[i]) }
# 	abline(h=0, lty = 2, lwd = 2)father

	matplot(get.ds.res, ylab = "Dunn-Smyth Residuals", xlab = "Row index",type="n", xaxt = "n")
 	axis(side = 1, at = 1:nrow(x$y), labels = rownames(x$lv.mean), cex.axis = 1)
 	for (i in 1:ncol(get.mus)) { points(seq(1,nrow(x$y)),get.ds.res[,i], col=palette()[i]) }
 	abline(0,0,lty=2)
 
 	matplot(t(get.ds.res), ylab = "Dunn-Smyth Residuals", xlab = "Column index", type="n", xaxt = "n")
 	axis(side = 1, at = 1:ncol(x$y), labels = rownames(x$coefs.mean), cex.axis = 1)
 	for(i in 1:ncol(get.mus)) { points(rep(i,nrow(get.etas)), get.ds.res[,i], col=palette()[i]) }
 	abline(h=0, lty = 2, lwd = 2)
 
	get.ds.res2 <- as.vector(unlist(get.ds.res))
 	qqnorm(get.ds.res2[is.finite(get.ds.res2)], main = "Normal Quantile Plot")
 	
 	palette("default")
 	}
 	
