% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brm_archetype_effects.R
\name{brm_archetype_effects}
\alias{brm_archetype_effects}
\title{Treatment effect archetype}
\usage{
brm_archetype_effects(
  data,
  covariates = TRUE,
  prefix_interest = "x_",
  prefix_nuisance = "nuisance_",
  baseline = !is.null(attr(data, "brm_baseline")),
  baseline_subgroup = !is.null(attr(data, "brm_baseline")) && !is.null(attr(data,
    "brm_subgroup")),
  baseline_subgroup_time = !is.null(attr(data, "brm_baseline")) && !is.null(attr(data,
    "brm_subgroup")),
  baseline_time = !is.null(attr(data, "brm_baseline"))
)
}
\arguments{
\item{data}{A classed data frame from \code{\link[=brm_data]{brm_data()}}, or an informative
prior archetype from a function like \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}}.}

\item{covariates}{Logical of length 1.
\code{TRUE} (default) to include any additive covariates declared with
the \code{covariates} argument of \code{\link[=brm_data]{brm_data()}},
\code{FALSE} to omit.
For informative prior archetypes, this option is set in
functions like \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}} rather than in
\code{\link[=brm_formula]{brm_formula()}} in order to make sure columns are appropriately
centered and the underlying model matrix has full rank.}

\item{prefix_interest}{Character string to prepend to the new columns
of generated fixed effects of interest (relating to group, subgroup,
and/or time).
In rare cases, you may need to set a non-default prefix to prevent
name conflicts with existing columns in the data, or rename
the columns in your data.
\code{prefix_interest} must not be the same value as \code{prefix_nuisance}.}

\item{prefix_nuisance}{Same as \code{prefix_interest}, but relating to
generated fixed effects NOT of interest (not relating to group,
subgroup, or time). Must not be the same value as \code{prefix_interest}.}

\item{baseline}{Logical of length 1.
\code{TRUE} to include an additive effect for baseline
response, \code{FALSE} to omit.
Default is \code{TRUE} if \code{\link[=brm_data]{brm_data()}} previously declared a baseline
variable in the dataset.
Ignored for informative prior archetypes.
For informative prior archetypes, this option should be set in
functions like \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}} rather than in
\code{\link[=brm_formula]{brm_formula()}} in order to make sure columns are appropriately
centered and the underlying model matrix has full rank.}

\item{baseline_subgroup}{Logical of length 1.}

\item{baseline_subgroup_time}{Logical of length 1.
\code{TRUE} to include baseline-by-subgroup-by-time interaction,
\code{FALSE} to omit.
Default is \code{TRUE} if \code{\link[=brm_data]{brm_data()}} previously declared baseline
and subgroup variables in the dataset.
Ignored for informative prior archetypes.
For informative prior archetypes, this option should be set in
functions like \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}} rather than in
\code{\link[=brm_formula]{brm_formula()}} in order to make sure columns are appropriately
centered and the underlying model matrix has full rank.}

\item{baseline_time}{Logical of length 1.
\code{TRUE} to include baseline-by-time interaction, \code{FALSE} to omit.
Default is \code{TRUE} if \code{\link[=brm_data]{brm_data()}} previously declared a baseline
variable in the dataset.
Ignored for informative prior archetypes.
For informative prior archetypes, this option should be set in
functions like \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}} rather than in
\code{\link[=brm_formula]{brm_formula()}} in order to make sure columns are appropriately
centered and the underlying model matrix has full rank.}
}
\value{
A special classed \code{tibble} with data tailored to
the successive differences archetype. The dataset is augmented with
extra columns with the \code{"archetype_"} prefix, as well as special
attributes to tell downstream functions like \code{\link[=brm_formula]{brm_formula()}} what to
do with the object.
}
\description{
Create an informative prior archetype for a simple treatment
effect parameterization.
}
\details{
In this archetype, each fixed effect is either a placebo response
or a treatment effect.

To illustrate, suppose the dataset has two treatment groups A and B,
time points 1, 2, and 3, and no other covariates. Assume group A
is the reference group (e.g. placebo).
Let \code{mu_gt} be the marginal mean of the response at group
\code{g} time \code{t} given data and hyperparameters.
The model has fixed effect parameters \code{beta_1}, \code{beta_2}, ..., \code{beta_6}
which express the marginal means \code{mu_gt} as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  `mu_A1 = beta_1`
  `mu_A2 = beta_2`
  `mu_A3 = beta_3`

  `mu_B1 = beta_1 + beta_4`
  `mu_B2 = beta_2 + beta_5`
  `mu_B3 = beta_3 + beta_6`
}\if{html}{\out{</div>}}

Above, \code{beta_2} is the group mean of treatment group A at time 2,
and \code{beta_5} is the treatment effect of B relative to A at time 2.
}
\section{Prior labeling for \code{\link[=brm_archetype_effects]{brm_archetype_effects()}}}{

In the reference group (e.g. placebo) each fixed effect is a cell
mean at a time point. In each non-reference group, each fixed effect
is the treatment effect relative to the reference (at a time point).
The labeling scheme in \code{\link[=brm_prior_label]{brm_prior_label()}} and
\code{\link[=brm_prior_archetype]{brm_prior_archetype()}} translate straightforwardly. For example,
\code{brm_prior_label(code = "normal(1.2, 5)", group = "A", time = "2")}
declares a \code{normal(1.2, 5)} on \code{beta_2} (cell mean of the reference
group at time 2). Similarly,
\code{brm_prior_label(code = "normal(1.3, 6)", group = "B", time = "2")}
declares a \code{normal(1.3, 6)} prior on the treatment effect of group
\code{B} relative to group \code{A} at discrete time point \code{2}.
To confirm that you set the prior correctly, compare the \code{brms} prior
with the output of \code{summary(your_archetype)}.
See the examples for details.
}

\section{Nuisance variables}{

In the presence of covariate adjustment, functions like
\code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}} convert nuisance factors into binary
dummy variables, then center all those dummy variables and any
continuous nuisance variables at their means in the data.
This ensures that the main model coefficients
of interest are not implicitly conditional on a subset of the data.
In other words, preprocessing nuisance variables this way preserves
the interpretations of the fixed effects of interest, and it ensures
informative priors can be specified correctly.
}

\section{Prior labeling}{

Informative prior archetypes use a labeling scheme to assign priors
to fixed effects. How it works:

\if{html}{\out{<div class="sourceCode">}}\preformatted{1. First, assign the prior of each parameter a collection
  of labels from the data. This can be done manually or with
  successive calls to [brm_prior_label()].
2. Supply the labeling scheme to [brm_prior_archetype()].
  [brm_prior_archetype()] uses attributes of the archetype
  to map labeled priors to their rightful parameters in the model.
}\if{html}{\out{</div>}}

For informative prior archetypes, this process is much more convenient
and robust than manually calling \code{\link[brms:set_prior]{brms::set_prior()}}.
However, it requires an understanding of how the labels of the priors
map to parameters in the model. This mapping varies from archetype
to archetype, and it is documented in the help pages of
archetype-specific functions such as \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}}.
}

\examples{
set.seed(0L)
data <- brm_simulate_outline(
  n_group = 2,
  n_patient = 100,
  n_time = 4,
  rate_dropout = 0,
  rate_lapse = 0
) |>
  dplyr::mutate(response = rnorm(n = dplyr::n())) |>
  brm_data_change() |>
  brm_simulate_continuous(names = c("biomarker1", "biomarker2")) |>
  brm_simulate_categorical(
    names = c("status1", "status2"),
    levels = c("present", "absent")
  )
dplyr::select(
  data,
  group,
  time,
  patient,
  starts_with("biomarker"),
  starts_with("status")
)
archetype <- brm_archetype_effects(data)
archetype
summary(archetype)
formula <- brm_formula(archetype)
formula
prior <- brm_prior_label(
  code = "normal(1, 2.2)",
  group = "group_1",
  time = "time_2"
) |>
  brm_prior_label("normal(1, 3.3)", group = "group_1", time = "time_3") |>
  brm_prior_label("normal(1, 4.4)", group = "group_1", time = "time_4") |>
  brm_prior_label("normal(2, 2.2)", group = "group_2", time = "time_2") |>
  brm_prior_label("normal(2, 3.3)", group = "group_2", time = "time_3") |>
  brm_prior_label("normal(2, 4.4)", group = "group_2", time = "time_4") |>
  brm_prior_archetype(archetype)
prior
class(prior)
if (identical(Sys.getenv("BRM_EXAMPLES", unset = ""), "true")) {
tmp <- utils::capture.output(
  suppressMessages(
    suppressWarnings(
      model <- brm_model(
        data = archetype,
        formula = formula,
        prior = prior,
        chains = 1,
        iter = 100,
        refresh = 0
      )
    )
  )
)
suppressWarnings(print(model))
brms::prior_summary(model)
draws <- brm_marginal_draws(
  data = archetype,
  formula = formula,
  model = model
)
summaries_model <- brm_marginal_summaries(draws)
summaries_data <- brm_marginal_data(data)
brm_plot_compare(model = summaries_model, data = summaries_data)
}
}
\seealso{
Other informative prior archetypes: 
\code{\link{brm_archetype_average_cells}()},
\code{\link{brm_archetype_average_effects}()},
\code{\link{brm_archetype_cells}()},
\code{\link{brm_archetype_successive_cells}()},
\code{\link{brm_archetype_successive_effects}()}
}
\concept{informative prior archetypes}
