% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mlp-fit.R
\name{brulee_mlp}
\alias{brulee_mlp}
\alias{brulee_mlp.default}
\alias{brulee_mlp.data.frame}
\alias{brulee_mlp.matrix}
\alias{brulee_mlp.formula}
\alias{brulee_mlp.recipe}
\title{Fit neural networks}
\usage{
brulee_mlp(x, ...)

\method{brulee_mlp}{default}(x, ...)

\method{brulee_mlp}{data.frame}(
  x,
  y,
  epochs = 100L,
  hidden_units = 3L,
  activation = "relu",
  penalty = 0.001,
  mixture = 0,
  dropout = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 0.01,
  rate_schedule = "none",
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_mlp}{matrix}(
  x,
  y,
  epochs = 100L,
  hidden_units = 3L,
  activation = "relu",
  penalty = 0.001,
  mixture = 0,
  dropout = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 0.01,
  rate_schedule = "none",
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_mlp}{formula}(
  formula,
  data,
  epochs = 100L,
  hidden_units = 3L,
  activation = "relu",
  penalty = 0.001,
  mixture = 0,
  dropout = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 0.01,
  rate_schedule = "none",
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_mlp}{recipe}(
  x,
  data,
  epochs = 100L,
  hidden_units = 3L,
  activation = "relu",
  penalty = 0.001,
  mixture = 0,
  dropout = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 0.01,
  rate_schedule = "none",
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{Depending on the context:
\itemize{
\item A \strong{data frame} of predictors.
\item A \strong{matrix} of predictors.
\item A \strong{recipe} specifying a set of preprocessing steps
created from \code{\link[recipes:recipe]{recipes::recipe()}}.
}

The predictor data should be standardized (e.g. centered or scaled).}

\item{...}{Options to pass to the learning rate schedulers via
\code{\link[=set_learn_rate]{set_learn_rate()}}. For example, the \code{reduction} or \code{steps} arguments to
\code{\link[=schedule_step]{schedule_step()}} could be passed here.}

\item{y}{When \code{x} is a \strong{data frame} or \strong{matrix}, \code{y} is the outcome
specified as:
\itemize{
\item A \strong{data frame} with 1 column (numeric or factor).
\item A \strong{matrix} with numeric column  (numeric or factor).
\item A  \strong{vector}  (numeric or factor).
}}

\item{epochs}{An integer for the number of epochs of training.}

\item{hidden_units}{An integer for the number of hidden units, or a vector
of integers. If a vector of integers, the model will have \code{length(hidden_units)}
layers each with \code{hidden_units[i]} hidden units.}

\item{activation}{A character vector for the activation function )such as
"relu", "tanh", "sigmoid", and so on). See \code{\link[=brulee_activations]{brulee_activations()}} for
a list of possible values. If \code{hidden_units} is a vector, \code{activation}
can be a character vector with length equals to \code{length(hidden_units)}
specifying the activation for each hidden layer.}

\item{penalty}{The amount of weight decay (i.e., L2 regularization).}

\item{mixture}{Proportion of Lasso Penalty (type: double, default: 0.0). A
value of mixture = 1 corresponds to a pure lasso model, while mixture = 0
indicates ridge regression (a.k.a weight decay).}

\item{dropout}{The proportion of parameters set to zero.}

\item{validation}{The proportion of the data randomly assigned to a
validation set.}

\item{optimizer}{The method used in the optimization procedure. Possible choices
are 'LBFGS' and 'SGD'. Default is 'LBFGS'.}

\item{learn_rate}{A positive number that controls the initial rapidity that
the model moves along the descent path. Values around 0.1 or less are
typical.}

\item{rate_schedule}{A single character value for how the learning rate
should change as the optimization proceeds. Possible values are
\code{"none"} (the default), \code{"decay_time"}, \code{"decay_expo"}, \code{"cyclic"} and
\code{"step"}. See \code{\link[=schedule_decay_time]{schedule_decay_time()}} for more details.}

\item{momentum}{A positive number usually on \verb{[0.50, 0.99]} for the momentum
parameter in gradient descent.  (\code{optimizer = "SGD"} only)}

\item{batch_size}{An integer for the number of training set points in each
batch. (\code{optimizer = "SGD"} only)}

\item{class_weights}{Numeric class weights (classification only). The value
can be:
\itemize{
\item A named numeric vector (in any order) where the names are the outcome
factor levels.
\item An unnamed numeric vector assumed to be in the same order as the outcome
factor levels.
\item A single numeric value for the least frequent class in the training data
and all other classes receive a weight of one.
}}

\item{stop_iter}{A non-negative integer for how many iterations with no
improvement before stopping.}

\item{verbose}{A logical that prints out the iteration history.}

\item{formula}{A formula specifying the outcome term(s) on the left-hand side,
and the predictor term(s) on the right-hand side.}

\item{data}{When a \strong{recipe} or \strong{formula} is used, \code{data} is specified as:
\itemize{
\item A \strong{data frame} containing both the predictors and the outcome.
}}
}
\value{
A \code{brulee_mlp} object with elements:
\itemize{
\item \code{models_obj}: a serialized raw vector for the torch module.
\item \code{estimates}: a list of matrices with the model parameter estimates per
epoch.
\item \code{best_epoch}: an integer for the epoch with the smallest loss.
\item \code{loss}: A vector of loss values (MSE for regression, negative log-
likelihood for classification) at each epoch.
\item \code{dim}: A list of data dimensions.
\item \code{y_stats}: A list of summary statistics for numeric outcomes.
\item \code{parameters}: A list of some tuning parameter values.
\item \code{blueprint}: The \code{hardhat} blueprint data.
}
}
\description{
\code{brulee_mlp()} fits neural network models using stochastic gradient
descent. Multiple layers can be used.
}
\details{
This function fits feed-forward neural network models for regression (when
the outcome is a number) or classification (a factor). For regression, the
mean squared error is optimized and cross-entropy is the loss function for
classification.

When the outcome is a number, the function internally standardizes the
outcome data to have mean zero and a standard deviation of one. The prediction
function creates predictions on the original scale.

By default, training halts when the validation loss increases for at least
\code{step_iter} iterations. If \code{validation = 0} the training set loss is used.

The \emph{predictors} data should all be numeric and encoded in the same units (e.g.
standardized to the same range or distribution). If there are factor
predictors, use a recipe or formula to create indicator variables (or some
other method) to make them numeric. Predictors should be in the same units
before training.

The model objects are saved for each epoch so that the number of epochs can
be efficiently tuned. Both the \code{\link[=coef]{coef()}} and \code{\link[=predict]{predict()}} methods for this
model have an \code{epoch} argument (which defaults to the epoch with the best
loss value).

The use of the L1 penalty (a.k.a. the lasso penalty) does \emph{not} force
parameters to be strictly zero (as it does in packages such as \pkg{glmnet}).
The zeroing out of parameters is a specific feature the optimization method
used in those packages.
\subsection{Learning Rates}{

The learning rate can be set to constant (the default) or dynamically set
via a learning rate scheduler (via the \code{rate_schedule}). Using
\code{rate_schedule = 'none'} uses the \code{learn_rate} argument. Otherwise, any
arguments to the schedulers can be passed via \code{...}.
}
}
\examples{
\donttest{
if (torch::torch_is_installed()) {

 ## -----------------------------------------------------------------------------
 # regression examples (increase # epochs to get better results)

 data(ames, package = "modeldata")

 ames$Sale_Price <- log10(ames$Sale_Price)

 set.seed(122)
 in_train <- sample(1:nrow(ames), 2000)
 ames_train <- ames[ in_train,]
 ames_test  <- ames[-in_train,]


 # Using matrices
 set.seed(1)
 fit <-
   brulee_mlp(x = as.matrix(ames_train[, c("Longitude", "Latitude")]),
               y = ames_train$Sale_Price, penalty = 0.10)

 # Using recipe
 library(recipes)

 ames_rec <-
  recipe(Sale_Price ~ Bldg_Type + Neighborhood + Year_Built + Gr_Liv_Area +
         Full_Bath + Year_Sold + Lot_Area + Central_Air + Longitude + Latitude,
         data = ames_train) \%>\%
   # Transform some highly skewed predictors
   step_BoxCox(Lot_Area, Gr_Liv_Area) \%>\%
   # Lump some rarely occurring categories into "other"
   step_other(Neighborhood, threshold = 0.05)  \%>\%
   # Encode categorical predictors as binary.
   step_dummy(all_nominal_predictors(), one_hot = TRUE) \%>\%
   # Add an interaction effect:
   step_interact(~ starts_with("Central_Air"):Year_Built) \%>\%
   step_zv(all_predictors()) \%>\%
   step_normalize(all_numeric_predictors())

 set.seed(2)
 fit <- brulee_mlp(ames_rec, data = ames_train, hidden_units = 20,
                    dropout = 0.05, rate_schedule = "cyclic", step_size = 4)
 fit

 autoplot(fit)

 library(ggplot2)

 predict(fit, ames_test) \%>\%
   bind_cols(ames_test) \%>\%
   ggplot(aes(x = .pred, y = Sale_Price)) +
   geom_abline(col = "green") +
   geom_point(alpha = .3) +
   lims(x = c(4, 6), y = c(4, 6)) +
   coord_fixed(ratio = 1)

 library(yardstick)
 predict(fit, ames_test) \%>\%
   bind_cols(ames_test) \%>\%
   rmse(Sale_Price, .pred)


 # ------------------------------------------------------------------------------
 # classification

 library(dplyr)
 library(ggplot2)

 data("parabolic", package = "modeldata")

 set.seed(1)
 in_train <- sample(1:nrow(parabolic), 300)
 parabolic_tr <- parabolic[ in_train,]
 parabolic_te <- parabolic[-in_train,]

 set.seed(2)
 cls_fit <- brulee_mlp(class ~ ., data = parabolic_tr, hidden_units = 2,
                        epochs = 200L, learn_rate = 0.1, activation = "elu",
                        penalty = 0.1, batch_size = 2^8, optimizer = "SGD")
 autoplot(cls_fit)

 grid_points <- seq(-4, 4, length.out = 100)

 grid <- expand.grid(X1 = grid_points, X2 = grid_points)

 predict(cls_fit, grid, type = "prob") \%>\%
  bind_cols(grid) \%>\%
  ggplot(aes(X1, X2)) +
  geom_contour(aes(z = .pred_Class1), breaks = 1/2, col = "black") +
  geom_point(data = parabolic_te, aes(col = class))

 }
}
}
\seealso{
\code{\link[=predict.brulee_mlp]{predict.brulee_mlp()}}, \code{\link[=coef.brulee_mlp]{coef.brulee_mlp()}}, \code{\link[=autoplot.brulee_mlp]{autoplot.brulee_mlp()}}
}
