\name{mi.eval}
\alias{mi.eval}
\title{
Multiple-imputation evaluation
}
\description{
Evaluation of an expression across multiply imputed data sets.
}
\usage{
mi.eval(EXPR, robust, cluster, coef., vcov., df.=NULL, parallel=FALSE, \dots)
}
\arguments{
  \item{EXPR}{
An R expression to evaluate. This expression must contain a \code{data}
argument that specifies a list that contains an object called
\code{imputations} that, in turn, contains a list of data frames containing
different imputations. Multiply imputed data sets generated by
\code{\link[Amelia]{amelia}} or \code{\link[mitools]{imputationList}}
can both be used as the data object.
}
  \item{robust}{
Whether to use Huber-White robust standard errors. The default is
\code{TRUE} if \code{cluster} is specified and \code{FALSE} otherwise.
}
  \item{cluster}{
A vector specifing clusters for the purpose of computing clustered
robust standard errors. This can be a variable inside the
imputed data set. If unspecified, standard errors are not clustered. If
specified, \code{robust} cannot be \code{FALSE}.
}
  \item{coef.}{
The function used to get a numeric vector of coefficient estimates when
evaluated on  an object returned from evaluating \code{EXPR} for each
data set. The default is to use \code{\link[stats]{coef}}.
}
  \item{vcov.}{
The function that returns a numeric matrix giving the variance-covariance
matrix when evaluated on an object returned from evaluating \code{EXPR}
for each data set. The default is to use \code{\link{vcovCR}} if
\code{cluster} is specified, \code{\link[sandwich]{vcovHC}} if
\code{robust=TRUE} and \code{cluster} is not specified, and
\code{\link[stats]{vcov}} otherwise.
}
  \item{df.}{
Either the degrees of freedom for each model or a function that
calculates degrees of freedom on an object returned from evaluating
\code{EXPR} for each data set. The default value of \code{NULL} uses
the minimum result of applying \code{\link[stats]{df.residual}} it
returns a numeric value when applied to the object returned by
\code{EXPR} and this object is not of class \code{glm} and \code{Inf}
otherwise.
}
  \item{parallel}{
A logical indicating whether to evaluate \code{EXPR} across data sets in
parallel using \code{\link[parallel]{mclapply}}. The default is serial
evalation using \code{\link[base]{lapply}}.
}
  \item{\dots}{
Any additional arguments to be passed to \code{\link[base]{lapply}} or
\code{\link[parallel]{mclapply}} when evaluating \code{EXPR} across data sets.
}
}
\details{
This function evaluates a R command for each of several multiply imputed
data sets and combines results across data sets into a single set of
estimates.
}
\value{
An object of class \code{mi.estimates} containing the coefficient
estimates, variance-covariance matrix, and related information.
}

\seealso{
See Also \code{\link{summary.mi.estimates}},
\code{\link[Amelia]{amelia}},
\code{\link[stats]{coef}},
\code{\link[stats]{vcov}}
and
\code{\link[stats]{df.residual}}.
}
\examples{
if (require("Amelia")) {
    ## Load data
    data(africa)
    africa$civlib <- factor(round(africa$civlib*6), ordered=TRUE)

    ## Estimate a linear model using imputed data sets
    model0 <- lm(trade ~ log(gdp_pc), data=africa, subset=year==1973)
    summary(model0)

    ## Impute using Amelia    
    a.out <- amelia(x = africa, cs = "country", ts = "year",
                    logs = "gdp_pc", ord="civlib")

    ## Estimate a linear model using imputed data sets
    model1 <- mi.eval(lm(trade ~ log(gdp_pc), data=a.out, subset=year==1973))

    ## Show estimates
    model1
    coef(model1)

    ## Show summary information
    summary(model1)

    if (require("MASS")) {
        ## Estimate an ordered logit model
        model2 <- mi.eval(polr(civlib ~ log(gdp_pc) + log(population),
                               data=a.out))
        summary(model2)

        ## Also show thresholds by including thresholds with coefficients
        model3 <- mi.eval(polr(civlib ~ log(gdp_pc) + log(population),
                               data=a.out),
                          coef=function(x) c(x$coefficients, x$zeta))
        summary(model2)
    }
}
}
\keyword{ models }
\keyword{ htest }
