# last revision: 13 Jan 2012

#' @include ChemosensorsClass.R
NULL

#----------------------------
# Class defintion
#----------------------------

validSensorNoiseModel <- function(object)
{
  if(sum(ssd(object) < 0)) return("'ssd' is negative")
  if(nrow(object@sndata) != length(object@snf)) return("slots 'sndata' and 'snf' are of different size.")
  
  return(TRUE)
}

#' Class SensorNoiseModel.
#'
#' Class \code{SensorNoiseModel} simulates a degradation in the performance of an individual sensor 
#' by generating noise in the sensitivity coefficients. This class is a super-class of the class \code{Sensor}.
#'
#' Slots of the class:
#' \tabular{rl}{
#'   \code{num} \tab Sensor number (\code{1:17}), which noise profile is used. The default value is \code{1}. \cr
#'   \code{gases} \tab Gas indices. \cr
#'   \code{ngases} \tab The number of gases. \cr
#'   \code{gnames} \tab Names of gases. \cr
#'   \code{concUnits} \tab Concentration units external to the model. \cr
#'   \code{concUnitsInt} \tab Concentration units internal for the model, values used to evaluate the statistics on coefficients. \cr
#'   \code{ssd} \tab Parameter of standard deviation used to generate the noise. The deault value is 0.1. \cr
#'   \code{sntype} \tab Noise type (\code{randomWalk}). \cr
#'   \code{snf} \tab Scaling factor for the amplitude to generate the noise. The default value is \code{c(1, 1, 0.2)}. \cr
#'   \code{sndata} \tab The reference data of standard deviation values from UNIMAN dataset (see \code{\link{UNIMANsnoise}}). \cr
#' }
#'
#' Methods of the class:
#' \tabular{rl}{
#'   \code{predict} \tab Generates noise vectors of a given length. \cr
#'   \code{ssd} \tab Gets the noise level. \cr
#'   \code{ssd<-} \tab Sets the noise level. \cr
#'   \code{ncoef} \tab Gets the number of coefficients. \cr
#' }
#' 
#' The \code{plot} method has three types (parameter \code{y}):
#' \tabular{rl}{
#'   \code{barplot} \tab (default) Shows distribution of sd values stored in slot \code{sndata}. \cr
#'   \code{noise} \tab Depicts the noise vectors generated by the model. \cr
#'   \code{walk} \tab  Shows the random walk on a scatterplot for two given coefficients. \cr
#' }
#'  
#' @name SensorNoiseModel-class
#' @rdname SensorNoiseModel-class
#' @seealso \code{\link{UNIMANsnoise}}, \code{\link{Sensor}}
#' @example inst/examples/SensorNoiseModel-class.R
#' @exportClass SensorNoiseModel
setClass(Class="SensorNoiseModel", 
  representation=representation(
    # class-specific slots
    ssd = "numeric",
    # class-common slots
    idx = "numeric", num = "numeric", gases = "numeric", ngases = "numeric", gnames="character", concUnits="character", concUnitsInt="character",
    # class-specific slots
    sntype = "character", snf = "numeric", sndata = "matrix"),  
  validity=validSensorNoiseModel
)


#----------------------------
# Print/Show Methods
#----------------------------

#' @rdname class-methods
#' @aliases print,SensorNoiseModel-method
setMethod ("print","SensorNoiseModel", function(x, ...)
{
  cat(" Sensor Noise Model\n")
  cat(" - num", numStr(x), "\n")
  cat(" -", ngases(x), "gases", paste(gnames(x), collapse=", "), "\n")
  cat(" - ssd:", paste(ssd(x), collapse=", "), "\n")
  cat(" - noise type:", type(x), "\n")
  cat(" - noise-factor:", paste(noisef(x), collapse=", "), "\n")    
})

setMethod ("show","SensorNoiseModel", function(object)
{
  cat(" Sensor Noise Model (ssd ", 
    paste(ssd(object), collapse=", "), "), ", "noise type '", 
    type(object), "'", "\n", sep='')
})


#----------------------------
# Plot Methods
#----------------------------

#----------------------------
# Get/Set Methods
#----------------------------

#' @rdname noise-methods
#' @aliases ssd,SensorNoiseModel-method
setMethod("ssd", "SensorNoiseModel", function(x) x@ssd)

#' @rdname get-methods
#' @aliases type,SensorNoiseModel-method
setMethod("type", "SensorNoiseModel", function(x) x@sntype)

#' @rdname SensorNoiseModel-class
#' @aliases noisef,SensorNoiseModel-method
setMethod("noisef", "SensorNoiseModel", function(x) x@snf)

#' @name ssd<-
#' @aliases ssd<-,SensorNoiseModel-method
#' @rdname noise-methods
setReplaceMethod("ssd", "SensorNoiseModel", function(object, value) 
{
  object@ssd <- value
  validObject(object)
  return (object)
})

#----------------------------
# Model Methods
#----------------------------

#----------------------------
# Predict methods
#----------------------------

#----------------------------
# Noise Methods
#----------------------------

#' @name nsd<-
#' @aliases nsd<-,SensorNoiseModel-method
#' @rdname noise-methods
setReplaceMethod("nsd", "SensorNoiseModel", function(object, value) 
{
  ssd(object) <- value
  validObject(object)
  return (object)
})

