% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cirls.control.R
\name{cirls.control}
\alias{cirls.control}
\title{Parameters controlling CIRLS fitting}
\usage{
cirls.control(epsilon = 1e-08, maxit = 25, trace = FALSE, Cmat = NULL,
  lb = 0L, ub = Inf, qp_solver = "osqp", qp_pars = list())
}
\arguments{
\item{epsilon}{Positive convergence tolerance. The algorithm converges when the relative change in deviance is smaller than \code{epsilon}.}

\item{maxit}{Integer giving the maximal number of CIRLS iterations.}

\item{trace}{Logical indicating if output should be produced for each iteration.}

\item{Cmat}{Constraint matrix specifying the linear constraints applied to coefficients. Can also be provided as a list of matrices for specific terms.}

\item{lb, ub}{Lower and upper bound vectors for the linear constraints. Identical values in \code{lb} and \code{ub} identify equality constraints. Recycled if length is different than the number of constraints defined by \code{Cmat}.}

\item{qp_solver}{The quadratic programming solver. One of \code{"osqp"}, \code{"quadprog"} or \code{"coneproj"}.}

\item{qp_pars}{List of parameters specific to the quadratic programming solver. See respective packages help.}
}
\value{
A named list containing arguments to be used in \link[=cirls.fit]{cirls.fit}.
}
\description{
Internal function controlling the \link[stats:glm]{glm} fit with linear constraints. Typically only used internally by \link[=cirls.fit]{cirls.fit}, but may be used to construct a control argument.
}
\details{
The \code{control} argument of \link[stats:glm]{glm} is by default passed to the \code{control} argument of \link[=cirls.fit]{cirls.fit}, which uses its elements as arguments for \link[=cirls.control]{cirls.control}: the latter provides defaults and sanity checking. The control parameters can alternatively be passed through the \code{...} argument of \link[stats:glm]{glm}. See \link[stats:glm.control]{glm.control} for details on general GLM fitting control, and \link[=cirls.fit]{cirls.fit} for details on arguments specific to constrained GLMs.
}
\examples{
# Simulate predictors and response with some negative coefficients
set.seed(111)
n <- 100
p <- 10
betas <- rep_len(c(1, -1), p)
x <- matrix(rnorm(n * p), nrow = n)
y <- x \%*\% betas + rnorm(n)

# Define constraint matrix (includes intercept)
# By default, bounds are 0 and +Inf
Cmat <- cbind(0, diag(p))

# Fit GLM by CIRLS
res1 <- glm(y ~ x, method = cirls.fit, Cmat = Cmat)
coef(res1)

# Same as passing Cmat through the control argument
res2 <- glm(y ~ x, method = cirls.fit, control = list(Cmat = Cmat))
identical(coef(res1), coef(res2))
}
\seealso{
the main function \link[=cirls.fit]{cirls.fit}, and \link[stats:glm.control]{glm.control}.
}
