% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/02_color_space_conversion.R
\name{convertColorSpace}
\alias{convertColorSpace}
\title{Convert between color spaces}
\usage{
convertColorSpace(color.coordinate.matrix, from = "sRGB", to = "Lab",
  sample.size = 1e+05, from.ref.white, to.ref.white)
}
\arguments{
\item{color.coordinate.matrix}{A color coordinate matrix with rows as colors
and channels as columns. If a color histogram (e.g. as returned by
\code{\link{getImageHist}}) is passed, the 'Pct' column is ignored.}

\item{from, to}{Input and output color spaces, passed to
\code{\link[grDevices]{convertColor}}. See details.}

\item{sample.size}{Number of pixels to be randomly sampled from filtered pixel
array for conversion. If not numeric or larger than number of colors
provided (i.e. cluster matrix), all colors are converted. See details.}

\item{from.ref.white, to.ref.white}{Reference whites passed to
\code{\link[grDevices]{convertColor}}. Unlike \code{convertColor}, no
default is provided. See details for explanation of different reference
whites.}
}
\value{
A 3- or 4-column matrix depending on whether
  \code{color.coordinate.matrix} included a 'Pct' column (as from
  \code{\link{getImageHist}}), with one column per channel.
}
\description{
Wrapper for \code{\link[grDevices]{convertColor}} that builds in random
sampling, error messages, and removes default illuminant (D65) to enforce
manual specification of a reference white.
}
\details{
Color spaces are all passed to
  \code{\link[grDevices]{convertColor}}, and can be any of: \code{"XYZ"},
  \code{"sRGB"}, \code{"Apple RGB"}, \code{"CIE RGB"}, \code{"Lab"}, or
  \code{"Luv"}.
  
  \code{Lab} and \code{Luv} color spaces are approximately perceptually
  uniform, meaning they usually do the best job of reflecting intuitive color
  distances without the non-linearity problems of more familiar RGB spaces.
  However, because they describe object colors, they require a reference
  'white light' color (dimly and brightly lit photographs of the same object
  will have very different RGB palettes, but similar Lab palettes if
  appropriate white references are used). The idea here is that the apparent
  colors in an image depend not just on the "absolute" color of an object,
  but also on the available light in the scene. There are seven CIE
  standardized illuminants available in \code{colordistance} (A, B, C, E, and
  D50, D55, and D65), but the most common are: \itemize{ \item \code{"A"}:
  Standard incandescent lightbulb \item \code{"D65"}: Average daylight \item
  \code{"D50"}: Direct sunlight}
  
Color conversions will be highly dependent on the reference white used, which
is why no default is provided. Users should look into
\href{https://en.wikipedia.org/wiki/Standard_illuminant}{standard
illuminants} to choose an appropriate reference for a dataset.

The conversion from RGB to a standardized color space (XYZ, Lab, or Luv) is
approximate, non-linear, and relatively time-consuming. Converting a large
number of pixels can be computationally expensive, so
\code{convertColorSpace} will randomly sample a specified number of rows to
reduce the time. The default sample size, 100,000 rows, takes about 5 seconds
convert from sRGB to Lab space on an early 2015 Macbook with 8 GB of RAM.
Time scales about linearly with number of rows converted.
}
\examples{
# Convert a single RGB triplet and then back convert it
rgb_color <- c(0, 1, 0)
lab_color <- colordistance::convertColorSpace(rgb_color,
 from="sRGB", to="Lab", to.ref.white="D65")
rgb_again <- colordistance::convertColorSpace(lab_color,
 from="Lab", to="sRGB", from.ref.white="D65")

# Convert pixels from loadImage() function
img <- colordistance::loadImage(system.file("extdata",
"Heliconius/Heliconius_B/Heliconius_07.jpeg", package="colordistance"))
lab_pixels <- colordistance::convertColorSpace(img$filtered.rgb.2d,
from="sRGB", to="XYZ", sample.size=5000)

# Convert clusters
img <- colordistance::loadImage(system.file("extdata",
"Heliconius/Heliconius_B/Heliconius_07.jpeg", package="colordistance"))
img_hist <- colordistance::getImageHist(img, bins=2, plotting=FALSE)
lab_clusters <- colordistance::convertColorSpace(img_hist, to.ref.white="D55")

}
