% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calibrate.R
\name{calibrate}
\alias{calibrate}
\alias{calibrate,EffectScenario-method}
\alias{calibrate,list-method}
\title{Fit model parameters to experimental data}
\usage{
calibrate(x, ...)

\S4method{calibrate}{EffectScenario}(
  x,
  par,
  data,
  endpoint,
  hessian = TRUE,
  metric_fun = sse,
  as_tibble = FALSE,
  catch_errors = TRUE,
  verbose = FALSE,
  ...
)

\S4method{calibrate}{list}(
  x,
  par,
  endpoint,
  hessian = TRUE,
  metric_fun = sse,
  metric_total = sum,
  as_tibble = FALSE,
  catch_errors = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{either a single \code{EffectScenario} or a list of \code{CalibrationSet} objects to be fitted}

\item{...}{additional parameters passed on to \code{stats::optim()} and \code{simulate()}}

\item{par}{named numeric vector with parameters to fit and their start values}

\item{data}{\code{data.frame} with two or more columns with experimental data,
1st column contains time points, 2nd and following columns contain values
which the scenario is fitted to.}

\item{endpoint}{\code{character}, name of model output column to optimize on}

\item{hessian}{\code{logical}, if \code{TRUE} the Hessian is returned by the optimization routine}

\item{metric_fun}{vectorized function to calculate an error term that is
minimized during optimization, defaults to sum of squared errors}

\item{as_tibble}{\code{logical}, if \code{TRUE} a tibble with scenario and optimization
result is returned, else only the optimization result is returned}

\item{catch_errors}{\code{logical}, if \code{TRUE} simulation errors due to e.g. out of
bounds parameters are caught and optimization continues, else optimization fails}

\item{verbose}{\code{logical}, if \code{TRUE} then debug outputs are displayed during
optimization}

\item{metric_total}{vectorized function to aggregate multiple error terms to a
single value, defaults to \code{sum()}}
}
\value{
By default, a list of fitted parameters (as produced by \code{\link[stats:optim]{stats::optim()}})
is returned. Alternatively, when setting the argument as_tibble = TRUE,
a \code{tibble} is returned with 2 columns, the first column lists the
scenario and the 2nd column the list of fitted parameters.
}
\description{
The function \code{calibrate()}performs the calibration (fitting) of model
parameters to observed data. Two options are available:
\enumerate{
\item Either a single \link{scenario} is fitted to a single dataset.
The dataset must represent a time-series of an output variable of the
model, e.g. observed biomass over time (effect data).
\item Or a scenario is fitted to one or multiple datasets at once. Each
dataset is represented by a \linkS4class{CalibrationSet} which also
contains a scenario which describes the conditions during observation,
such as exposure and environmental properties. Each \code{CalibrationSet}
can also be given a weight to be considered in the fitting. By default,
the total sum of squared errors is used as the target function which is
minimized.
}
}
\details{
Effect data must be supplied as a \code{data.frame} in long format with two
columns: the first column contains timestamps and the second column
the observed property. The effect data may contain replicates as extra rows.
}
\section{Methods (by class)}{
\itemize{
\item \code{calibrate(EffectScenario)}: Fit a single scenario

\item \code{calibrate(list)}: Fit using a \link{CalibrationSet}

}}
\examples{
## 1st option: Fit scenario parameters using a single dataset
metsulfuron

# Observed biomass from Schmitt et al. 2013 experiment
obs32 <- data.frame(
  time=c(0, 3, 5, 7, 7.01, 10, 12, 14),
  obs=c(12, 25, 30, 33, 33, 64, 127, 300)
)

# calibrate with an ExposureScenario and corresponding effect data
metsulfuron \%>\%
  calibrate(
    par=c(k_phot_max=1, k_resp=0.1 ),
    data=obs32,
    endpoint="BM",
  )

# plot trial data
obs32$trial <- "conc=1.0"
plot_sd(treatments=metsulfuron@exposure@series, rs_mean=obs32)

## 2nd option: Fit scenario parameters using CalibrationSets

# Create one CalibrationSet for each observed exposure level and time-series
# from Schmitt et al. (2013) dataset
library(dplyr)
Schmitt2013 \%>\%
  group_by(ID) \%>\%
  group_map(function(data, key) {
    exp <- data \%>\% select(t, conc)
    obs <- data \%>\% select(t, obs)
    sc <- metsulfuron \%>\% set_exposure(exp)
    CalibrationSet(sc, obs)
  }) -> cs
# Fit parameters
calibrate(cs, par=c(k_phot_max=1), endpoint="BM")

}
