% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_eco_variables.R
\name{sdSumry}
\alias{sdSumry}
\title{Calculate basic spatial coverage and diversity metrics}
\usage{
sdSumry(
  dat,
  xy,
  taxVar,
  crs = "epsg:4326",
  collections = NULL,
  quotaQ = NULL,
  quotaN = NULL,
  omitDom = FALSE
)
}
\arguments{
\item{dat}{A \code{data.frame} or \code{matrix} containing taxon names,
coordinates, and any associated variables; or a list of such structures.}

\item{xy}{A vector of two elements, specifying the name or numeric position
of columns in \code{dat} containing coordinates, e.g. longitude and latitude.
Coordinates for any shared sampling sites should be identical, and where sites
are raster cells, coordinates are usually expected to be cell centroids.}

\item{taxVar}{The name or numeric position of the column containing
taxonomic identifications. \code{taxVar} must be of same class as \code{xy}, e.g. a
numeric column position if \code{xy} is given as a vector of numeric positions.}

\item{crs}{Coordinate reference system as a GDAL text string, EPSG code,
or object of class \code{crs}. Default is latitude-longitude (\code{EPSG:4326}).}

\item{collections}{The name or numeric position of the column containing
unique collection IDs, e.g. 'collection_no' in PBDB data downloads.}

\item{quotaQ}{A numeric value for the coverage (quorum) level at which to
perform coverage-based rarefaction (shareholder quorum subsampling).}

\item{quotaN}{A numeric value for the quota of taxon occurrences to subsample
in classical rarefaction.}

\item{omitDom}{If \code{omitDom = TRUE} and \code{quotaQ} or \code{quotaN}
is supplied, remove the most common taxon prior to rarefaction. The \code{nTax}
and \code{evenness} returned are unaffected.}
}
\value{
A \code{matrix} of spatial and optional diversity metrics. If \code{dat} is a
\code{list} of \code{data.frame} objects, output rows correspond to input elements.
}
\description{
Summarise the geographic scope and position of occurrence data, and
optionally estimate diversity and evenness
}
\details{
\code{sdSumry()} compiles metadata about a sample or list of samples,
before or after spatial subsampling. The function counts the number
of collections (if requested), taxon presences (excluding repeat incidences
of a taxon at a given site), and unique spatial sites;
it also calculates site centroid coordinates, latitudinal range (degrees),
great circle distance (km), mean pairwise distance (km), and summed
minimum spanning tree length (km).
Coordinates and their distances are computed with respect to the original
coordinate reference system if supplied, except in calculation of latitudinal
range, for which projected coordinates are transformed to geodetic ones.
If \code{crs} is unspecified, by default points are assumed to be given in
latitude-longitude and distances are calculated with spherical geometry.

The first two diversity variables returned are the raw count of observed taxa
and the Summed Common species/taxon Occurrence Rate (SCOR). SCOR reflects
the degree to which taxa are common/widespread and is decoupled from
richness or abundance (Hannisdal \emph{et al.} 2012). SCOR is calculated as the
sum across taxa of the log probability of incidence, \eqn{\lambda}.
For a given taxon, \eqn{\lambda = -ln(1 - p)},
where \eqn{p} is estimated as the fraction of occupied sites.
Very widespread taxa make a large contribution to an assemblage SCOR,
while rare taxa have relatively little influence.

If \code{quotaQ} is supplied, \code{sdSumry()} rarefies richness at the
given coverage value and returns the point estimate of richness (Hill number 0)
and its 95\% confidence interval, as well as estimates of evenness (Pielou's J)
and frequency-distribution sample coverage (given by \code{iNEXT$DataInfo}).
If \code{quotaN} is supplied, \code{sdSumry()} rarefies richness to the given
number of occurrence counts and returns the point estimate of richness
and its 95\% confidence interval.
Coverage-based and classical rarefaction are both calculated with
\code{\link[iNEXT:estimateD]{iNEXT::estimateD()}} internally. For details, such as how diversity
is extrapolated if sample coverage is insufficient to achieve a specified
rarefaction level, consult Chao and Jost (2012) and Hsieh \emph{et al.} (2016).
}
\examples{
# generate occurrences
set.seed(9)
x  <- sample(rep(1:5, 10))
y  <- sample(rep(1:5, 10))
# make some species 2x or 4x as common
abund <- c(rep(4, 5), rep(2, 5), rep(1, 10))
sp <- sample(letters[1:20], 50, replace = TRUE, prob = abund)
obs <- data.frame(x, y, sp)

# minimum sample data returned
sdSumry(obs, c('x','y'), 'sp')

# also calculate evenness and coverage-based rarefaction diversity estimates
sdSumry(obs, xy = c('x','y'), taxVar = 'sp', quotaQ = 0.7)

}
\references{
\insertRef{Chao2012}{divvy}

\insertRef{Hannisdal2012}{divvy}

\insertRef{Hsieh2016}{divvy}
}
\seealso{
\code{\link[=rangeSize]{rangeSize()}}
}
