#' ds_search
#'
#' Search csv resource file stored in the datastore.
#'
#' @param resource_id character: DKAN resource ID
#' @param fields character vector: fields to be returned by the request
#' @param filters named character vector: filters to be applied when retrieving records
#' @param limit numeric: Number of records to retrieve
#' @param offset numeric: Number of results to skip in the beginning
#' @param sort named character vector: field to be sorted by
#' @param q character: full text query filter
#' @param url character: The DKAN site URL
#' @param credentials Optional list parameter. Default values are Cookie and Token generated by dkan_setup()
#'
#' @return data.frame or character
#' @export
#'
#' @examples
#' \dontrun{
#' ds_search(resource_id = '10c578a6-63c4-40bd-a55d-0c27bf276283',
#'           fields = c('Country','City','Region','Population'),
#'           filters = list('Country'=c('co','my'), 'Region'=c('04','09','22')),
#'           limit = 20,
#'           offset = 0,
#'           sort = c('Country'='asc'),
#'           q = "puertica",
#'           url = get_url(),
#'           as = 'df')
#' }

ds_search <- function(resource_id,
                      fields = NULL,
                      filters = NULL,
                      limit = 100,
                      offset = 0,
                      sort = NULL,
                      q = NULL,
                      url = get_url(),
                      credentials = list(cookie = dkanr::get_cookie(),
                                         token = dkanr::get_token())) {
  # authentication
  cookie <- credentials$cookie
  token <- credentials$token

  # DKAN settings
  path <- "api/action/datastore/search.json"

  # build the url
  query <- build_ds_search_query(resource_id, fields, filters, sort, q, offset, limit)
  url <- httr::modify_url(url, path = path, query = query)
  # execute the query
  res <- httr::GET(
    url = url,
    httr::add_headers(.headers = c(
      "Content-Type" = "application/json",
      "charset" = "utf-8",
      "Cookie" = cookie,
      "X-CSRF-Token" = token
    )),
    encode = "json"
  )


  err_handler(res)

  out <- httr::content(res)$result$records

  # Handle possible null value
  out <- purrr::modify_depth(out, .depth = 2, function(x) {
    if (is.null(x)) {
      NA
    } else {
      x
    }})

  return(out)
}
