% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flows.R
\name{dodgr_flows_aggregate}
\alias{dodgr_flows_aggregate}
\title{dodgr_flows_aggregate}
\usage{
dodgr_flows_aggregate(graph, from, to, flows, wt_profile = "bicycle",
  contract = FALSE, heap = "BHeap", quiet = TRUE)
}
\arguments{
\item{graph}{\code{data.frame} or equivalent object representing the network
graph (see Details)}

\item{from}{Vector or matrix of points \strong{from} which aggregate flows are to
be calculated (see Details)}

\item{to}{Vector or matrix of points \strong{to} which aggregate flows are to be
calculated (see Details)}

\item{flows}{Matrix of flows with \code{nrow(flows)==length(from)} and
\code{ncol(flows)==length(to)}.}

\item{wt_profile}{Name of weighting profile for street networks (one of foot,
horse, wheelchair, bicycle, moped, motorcycle, motorcar, goods, hgv, psv;
only used if \code{graph} is not provided, in which case a street network is
downloaded and correspondingly weighted).}

\item{contract}{If \code{TRUE}, calculate flows on contracted graph before
mapping them back on to the original full graph (recommended as this will
generally be much faster).}

\item{heap}{Type of heap to use in priority queue. Options include
Fibonacci Heap (default; \code{FHeap}), Binary Heap (\code{BHeap}),
\code{Radix}, Trinomial Heap (\code{TriHeap}), Extended Trinomial Heap
(\code{TriHeapExt}, and 2-3 Heap (\code{Heap23}).}

\item{quiet}{If \code{FALSE}, display progress messages on screen.}
}
\value{
Modified version of graph with additonal \code{flow} column added.
}
\description{
Aggregate flows throughout a network based on an input matrix of flows
between all pairs of \code{from} and \code{to} points.
}
\examples{
graph <- weight_streetnet (hampi)
from <- sample (graph$from_id, size = 10)
to <- sample (graph$to_id, size = 5)
to <- to [!to \%in\% from]
flows <- matrix (10 * runif (length (from) * length (to)),
                 nrow = length (from))
graph <- dodgr_flows_aggregate (graph, from = from, to = to, flows = flows)
# graph then has an additonal 'flows` column of aggregate flows along all
# edges. These flows are directed, and can be aggregated to equivalent
# undirected flows on an equivalent undirected graph with:
graph_undir <- merge_directed_flows (graph)
# This graph will only include those edges having non-zero flows, and so:
nrow (graph); nrow (graph_undir) # the latter is much smaller

# The following code can be used to convert the resultant graph to an `sf`
# object suitable for plotting
\dontrun{
geoms <- dodgr_to_sfc (graph_undir)
gc <- dodgr_contract_graph (graph_undir)
gsf <- sf::st_sf (geoms)
gsf$flow <- gc$graph$flow

# example of plotting with the 'mapview' package
library (mapview)
flow <- gsf$flow / max (gsf$flow)
ncols <- 30
cols <- colorRampPalette (c ("lawngreen", "red")) (ncols) [ceiling (ncols * flow)]
mapview (gsf, color = cols, lwd = 10 * flow)
}

# An example of flow aggregation across a generic (non-OSM) highway,
# represented as the `routes_fast` object of the \\pkg{stplanr} package,
# which is a SpatialLinesDataFrame containing commuter densities along
# components of a street network.
\dontrun{
library (stplanr)
# merge all of the 'routes_fast' lines into a single network
r <- overline (routes_fast, attrib = "length", buff_dist = 1)
r <- sf::st_as_sf (r)
# then extract the start and end points of each of the original 'routes_fast'
# lines and use these for routing with `dodgr`
l <- lapply (routes_fast@lines, function (i)
             c (sp::coordinates (i) [[1]] [1, ],
                tail (sp::coordinates (i) [[1]], 1)))
l <- do.call (rbind, l)
xy_start <- l [, 1:2]
xy_end <- l [, 3:4]
# Then just specify a generic OD matrix with uniform values of 1:
flows <- matrix (1, nrow = nrow (l), ncol = nrow (l))
# We need to specify both a `type` and `id` column for the
# \\link{weight_streetnet} function.
r$type <- 1
r$id <- seq (nrow (r))
graph <- weight_streetnet (r, type_col = "type", id_col = "id",
                           wt_profile = 1)
f <- dodgr_flows_aggregate (graph, from = xy_start, to = xy_end, flows = flows)
# Then merge directed flows and convert to \\pkg{sf} for plotting as before:
f <- merge_directed_flows (f)
geoms <- dodgr_to_sfc (f)
gc <- dodgr_contract_graph (f)
gsf <- sf::st_sf (geoms)
gsf$flow <- gc$graph$flow
# sf plot:
plot (gsf ["flow"])
}
}
