#' Normal-halfnormal distribution
#'
#' Probablitiy density function, distribution, quantile function and random number generation for the normal-halfnormal distribution.
#'
#' @aliases pnormhnorm, qnormhnorm, rnormhnorm
#'
#' @param x vector of quantiles.
#' @param mu vector of \eqn{\mu}
#' @param sigma_v vector of \eqn{\sigma_V}. Must be positive.
#' @param sigma_u vector of \eqn{\sigma_U}. Must be positive.
#' @param s \eqn{s=-1} for production and \eqn{s=1} for cost function.
#' @param deriv derivative of order \code{deriv} of the log density. Available are 1,2,3,4.
#' @param xg optional, index arrays for upper triangular matrices, generated by \code{trind.generator(K)} and supplied to \code{chainrule}.
#' @param log.p logical; if TRUE, probabilities p are given as log(p).
#'
#' @return \code{dnormhnorm} gives the density, \code{pnormhnorm} give the distribution function, \code{qnormhnorm} gives the quantile function, and \code{rnormhnorm} generates random numbers, with given parameters. If the derivatives are calculated these are provided as the attributes \code{gradient}, \code{hessian}, \code{l3} and \code{l4} of the output of the density.
#'
#' @details A random variable \eqn{\mathcal{E}} follows a normal-halfnormal distribution if \eqn{\mathcal{E} = V + s \cdot U }, where \eqn{V \sim N(\mu, \sigma_V^2)} and \eqn{U \sim HN(\sigma_U^2)}.
#' The density is given by \deqn{f_\mathcal{E}(\epsilon)=\frac{1}{\sqrt{\sigma_V^2+\sigma_U^2}} \phi(\frac{\epsilon-\mu}{\sqrt{\sigma_V^2+\sigma_U^2}}) \Phi(s \frac{\sigma_U}{\sigma_V} \frac{x-\mu}{\sqrt{\sigma_V^2+\sigma_U^2}}) \qquad,}
#' where \eqn{s=-1} for production and \eqn{s=1} for cost function.
#'
#'
#' @examples
#' pdf <- dnormhnorm(x=seq(-3, 3, by=0.1), mu=1, sigma_v=2, sigma_u=3, s=-1)
#' cdf <- pnormhnorm(q=seq(-3, 3, by=0.1), mu=1, sigma_v=2, sigma_u=3, s=-1)
#' q <- qnormhnorm(p=seq(0.1, 0.9, by=0.1), mu=1, sigma_v=2, sigma_u=3, s=-1)
#' r <- rnormhnorm(n=100, mu=1, sigma_v=2, sigma_u=3, s=-1)
#'
#' @references
#' \itemize{
#' \item \insertRef{aigner1977formulation}{dsfa}
#' \item \insertRef{kumbhakar2015practitioner}{dsfa}
#' \item \insertRef{schmidt2020analytic}{dsfa}
#' \item \insertRef{gradshteyn2014table}{dsfa}
#' \item \insertRef{azzalini2013skew}{dsfa}
#' }
#'
#' @export
#'
#normhnorm
dnormhnorm <- function(x, mu=0, sigma_v=1, sigma_u=1, s=-1, deriv=0, xg=NULL, log.p = FALSE){
  #Density function of the normhnorm distribution

  #Check for correct inputs
  if (any(sigma_v <= 0)){
    stop(paste("sigma_v must be positive", "\n", ""))
  }

  if (any(sigma_u <= 0)){
    stop(paste("sigma_u must be positive", "\n", ""))
  }

  if (any(!s%in%c(-1,1))){
    stop(paste("s must be {-1, 1}", "\n", ""))
  }

  #Transform parameters
  n<-length(x)
  v<-sigma_v
  u<-sigma_u
  a<-s*u/v
  w<-sqrt(v^2+u^2)
  tau<-1/w
  z<-(x-mu)*tau

  zeta0<-sn::zeta(0,a*z)

  #Get loglikelihood
  out  <-  log(1/sqrt(2*pi))+log(tau)-0.5*z^2+zeta0

  # out <- dsn(x=x, xi=para$xi, omega=para$omega, alpha=para$alpha, log=log.p)

  #Exponentialize output
  if (!log.p){
    out <- exp(out)
  }

  if(deriv>0){
    zeta1<-sn::zeta(1,a*z)
    zeta2<-sn::zeta(2,a*z)

    #First derivatives of w wrt v,u
    d1wdv1<-v*tau#v*w^(-1)
    d1wdu1<-u*tau#u*w^(-1)

    #Second derivatives of w wrt v,u
    d2wdv2<-tau+tau*(-1)*d1wdv1^2#tau+v*tau^2*(-1)*d1wdv1#w^(-1)-v^2*w^(-3)
    d2wdv1du1<-tau*(-1)*d1wdv1*d1wdu1#v*tau^2*(-1)*d1wdu1#-v*u*w^(-3)
    d2wdu2<-tau+tau*(-1)*d1wdu1^2#tau+u*tau^2*(-1)*d1wdu1#w^(-1)-u^2*w^(-3)

    #First derivatives of a wrt v, u
    d1adv1<-s*u*v^(-2)*(-1)
    d1adu1<-s*v^(-1)

    #Second derivatives of a wrt v, u
    d2adv2<-s*u*v^(-3)*2
    d2adv1du1<-s*v^(-2)*(-1)
    d2adu2<-0

    #First derivatives of z wrt mu, v, u
    d1zdmu1<--w^(-1)
    d1zdv1<--z*w^(-1)*d1wdv1
    d1zdu1<--z*w^(-1)*d1wdu1

    #Second derivatives of z wrt mu, v, su
    d2zdmu2<-0
    d2zdmu1dv1<-w^(-2)*d1wdv1
    d2zdmu1du1<-w^(-2)*d1wdu1
    d2zdv2<--z*w^(-2)*(-2)*d1wdv1^2-z*w^(-1)*d2wdv2
    d2zdv1du1<--z*w^(-2)*(-2)*d1wdv1*d1wdu1-z*w^(-1)*d2wdv1du1
    d2zdu2<--z*w^(-2)*(-2)*d1wdu1^2-z*w^(-1)*d2wdu2

    #First derivatives of az wrt mu, v, u
    d1azdmu1<-a*d1zdmu1
    d1azdv1<-d1adv1*z+a*d1zdv1
    d1azdu1<-d1adu1*z+a*d1zdu1

    #Second derivatives of az wrt mu, v, u
    d2azdmu2<-0#a*d2zdmu2
    d2azdmu1dv1<-d1adv1*d1zdmu1+a*d2zdmu1dv1
    d2azdmu1du1<-d1adu1*d1zdmu1+a*d2zdmu1du1
    d2azdv2<-d2adv2*z+2*d1adv1*d1zdv1+a*d2zdv2
    d2azdv1du1<-d2adv1du1*z+d1adv1*d1zdu1+d1adu1*d1zdv1+a*d2zdv1du1
    d2azdu2<-2*d1adu1*d1zdu1+a*d2zdu2

    if(deriv>2){
      ## the third derivatives
      zeta3<-sn::zeta(3,a*z)

      #Third derivatives of w wrt v,u
      d3wdv3<-tau^2*(-1)*d1wdv1+tau^2*d1wdv1^3+tau*(-2)*d1wdv1*d2wdv2#tau^2*(-1)*d1wdv1+tau^2*(-1)*d1wdv1+v*tau^3*(2)*d1wdv1^2+v*tau^2*d2wdv2*(-1) #-w^(-3)*v-2*v*w^(-3)+3*v^3*w^(-5)
      d3wdv2du1<-tau^2*1*d1wdv1^2*d1wdu1+tau*(-1)*d2wdv2*d1wdu1+tau*(-1)*d1wdv1*d2wdv1du1#tau^2*(-1)*d1wdu1+v*tau^3*(2)*d1wdu1*d1wdv1+v*tau^2*d2wdv1du1*(-1)#-w^(-3)*u+3*v^2*u*w^(-5)
      d3wdv1du2<-tau^2*1*d1wdv1*d1wdu1^2+tau*(-1)*d2wdu2*d1wdv1+tau*(-1)*d1wdu1*d2wdv1du1#tau^2*(-1)*d1wdv1+u*tau^3*(2)*d1wdv1*d1wdu1+u*tau^2*d2wdv1du1*(-1)#-w^(-3)*v+3*u^2*v*w^(-5)
      d3wdu3<-tau^2*(-1)*d1wdu1+tau^2*d1wdu1^3+tau*(-2)*d1wdu1*d2wdu2#-w^(-3)*u-2*u*w^(-3)+3*u^3*w^(-5)

      #Third derivatives of a wrt v, u
      d3adv3<-s*u*v^(-4)*(-6)
      d3adv2du1<-s*v^(-3)*(2)
      d3adv1du2<-0
      d3adu3<-0

      #Third derivatives of z wrt mu, v, u
      d3zdmu3<-0
      d3zdmu2dv1<-0
      d3zdmu2du1<-0
      d3zdmu1dv2<-w^(-3)*(-2)*d1wdv1^2+w^(-2)*d2wdv2
      d3zdmu1dv1du1<-w^(-3)*(-2)*d1wdu1*d1wdv1+w^(-2)*d2wdv1du1
      d3zdmu1du2<-w^(-3)*(-2)*d1wdu1^2+w^(-2)*d2wdu2
      d3zdv3<--z*w^(-3)*(6)*d1wdv1^3-z*w^(-2)*(-6)*d1wdv1*d2wdv2-z*w^(-1)*d3wdv3#-z*w^(-3)*(6)*d1wdv1^3-z*w^(-2)*(-4)*d1wdv1*d2wdv2-z*w^(-2)*(-2)*d1wdv1*d2wdv2-z*w^(-1)*d3wdv3
      d3zdv2du1<--z*w^(-3)*(6)*d1wdu1*d1wdv1^2-z*w^(-2)*(-4)*d1wdv1*d2wdv1du1-z*w^(-2)*(-2)*d1wdu1*d2wdv2-               z*w^(-1)*d3wdv2du1
      d3zdv1du2<--z*w^(-3)*(6)*d1wdu1^2*d1wdv1-z*w^(-2)*(-4)*d1wdu1*d2wdv1du1-z*w^(-2)*(-2)*d1wdv1*d2wdu2-z*w^(-1)*d3wdv1du2
      d3zdu3<--z*w^(-3)*(6)*d1wdu1^3-z*w^(-2)*(-4)*d1wdu1*d2wdu2-z*w^(-2)*(-2)*d1wdu1*d2wdu2-z*w^(-1)*d3wdu3

      #Third derivatives of az wrt mu, v, u
      d3azdmu3<-0#a*d3zdmu3
      d3azdmu2dv1<-0#d1azdv1*d2zdmu2+a*d3zdmu2dv1
      d3azdmu2du1<-0#d1azdu1*d2zdmu2+a*d3zdmu2du1
      d3azdmu1dv2<-d2adv2*d1zdmu1+2*d1adv1*d2zdmu1dv1+a*d3zdmu1dv2
      d3azdmu1dv1du1<-d2adv1du1*d1zdmu1+d1adv1*d2zdmu1du1+d1adu1*d2zdmu1dv1+a*d3zdmu1dv1du1
      d3azdmu1du2<-2*d1adu1*d2zdmu1du1+a*d3zdmu1du2
      d3azdv3<-d3adv3*z+3*d2adv2*d1zdv1+3*d1adv1*d2zdv2+a*d3zdv3
      d3azdv2du1<-d3adv2du1*z+d2adv2*d1zdu1+2*d2adv1du1*d1zdv1+2*d1adv1*d2zdv1du1+d1adu1*d2zdv2+a*d3zdv2du1
      d3azdv1du2<-2*d2adv1du1*d1zdu1+2*d1adu1*d2zdv1du1+d1adv1*d2zdu2+a*d3zdv1du2
      d3azdu3<-3*d1adu1*d2zdu2+a*d3zdu3

      ## the fourth derivatives
      zeta4<-sn::zeta(4,a*z)

      #Fourth derivatives of az wrt mu, v, u
      d4azdmu4<-0
      d4azdmu3dv1<-0
      d4azdmu3du1<-0
      d4azdmu2dv2<-0
      d4azdmu2dv1du1<-0
      d4azdmu2du2<-0
      d4azdmu1dv1du2<-(3 * s * (u^3 - 4 * u * v^2))/(u^2 + v^2)^(7/2)
      d4azdmu1dv3<-(3 * s * u * (2 * u^6 + 7 * u^4 * v^2 + 8 * u^2 * v^4 + 8 * v^6))/(v^4 * (u^2 + v^2)^(7/2))
      d4azdmu1dv2du1<-(s * (9 * u^2 * v - 6 * v^3))/(u^2 + v^2)^(7/2)
      d4azdmu1du3   <-(3 * s * v * (-4 * u^2 + v^2))/(u^2 + v^2)^(7/2)
      d4azdv4<- -((3 * s * u * (8 * u^8 + 36 * u^6 * v^2 + 63 * u^4 * v^4 + 40 * u^2 * v^6 + 40 * v^8) * (mu -x))/(v^5 * (u^2 + v^2)^(9/2)))
      d4azdv3du1<-(3 * s * (3 * u^4 - 24 * u^2 * v^2 + 8 * v^4) * (mu - x))/(u^2 + v^2)^(9/2)
      d4azdv2du2<-(15 * s * v * (-3 * u^3 + 4 * u * v^2) * (mu - x))/(u^2 + v^2)^(9/2)
      d4azdv1du3<--((3 * s * (4 * u^4 - 27 * u^2 * v^2 + 4 * v^4) * (mu - x))/(u^2 + v^2)^(9/2))
      d4azdu4<-(15 * s * v * (4 * u^3 - 3 * u * v^2) * (mu - x))/(u^2 + v^2)^(9/2)

      g3<-cbind(d3azdmu3,d3azdmu2dv1,d3azdmu2du1,d3azdmu1dv2,d3azdmu1dv1du1,d3azdmu1du2,d3azdv3,d3azdv2du1,d3azdv1du2,d3azdu3)
      g4<-cbind(d4azdmu4,d4azdmu3dv1,d4azdmu3du1,
                d4azdmu2dv2,d4azdmu2dv1du1,d4azdmu2du2,
                d4azdmu1dv3,d4azdmu1dv2du1,d4azdmu1dv1du2,d4azdmu1du3,
                d4azdv4,d4azdv3du1,d4azdv2du2,d4azdv1du3,
                d4azdu4)
    } else {
      g3<-NULL
      g4<-NULL
      zeta3<-NULL
      zeta4<-NULL
    }

    l<-chainrule(f1=zeta1, f2=zeta2, f3=zeta3, f4=zeta4,
                 g1=cbind(d1azdmu1, d1azdv1, d1azdu1),
                 g2=cbind(d2azdmu2, d2azdmu1dv1, d2azdmu1du1, d2azdv2, d2azdv1du1, d2azdu2),
                 g3=g3, g4=g4,
                 deriv=deriv, xg=xg)

    #Calculate l1,l2,l3,l4
    ## the first derivatives
    ## order mu, v, u
    l1 <- l$h1

    l1[,1] <--z*d1zdmu1+l1[,1]
    l1[,2] <--tau*d1wdv1-z*d1zdv1+l1[,2]
    l1[,3] <--tau*d1wdu1-z*d1zdu1+l1[,3]

    ## the second derivatives
    ## order mumu, muv, muu, vv, vu, uu
    l2 <- l$h2

    l2[,1] <--d1zdmu1^2+l2[,1]
    l2[,2] <--d1zdv1*d1zdmu1-z*d2zdmu1dv1+l2[,2]
    l2[,3] <--d1zdu1*d1zdmu1-z*d2zdmu1du1+l2[,3]
    l2[,4] <-tau^2*d1wdv1^2-tau*d2wdv2-d1zdv1^2-z*d2zdv2+l2[,4]
    l2[,5] <-tau^2*d1wdv1*d1wdu1-tau*d2wdv1du1-d1zdv1*d1zdu1-z*d2zdv1du1+l2[,5]
    l2[,6] <-tau^2*d1wdu1^2-tau*d2wdu2-d1zdu1^2-z*d2zdu2+l2[,6]

    #Set gradient and hessian as attributes
    attr(out,"gradient")<-l1
    attr(out,"hessian")<-l2

    if(deriv>2){
      # order mumumu, mumuv, mumuu,
      #       muvv, muvu, muuu
      #       vvv, vvu, vuu
      #       uuu
      l3 <- l$h3

      l3[,1] <- 0+l3[,1]
      l3[,2] <- (2 * v)/(u^2 + v^2)^2+l3[,2]#-2*d1zdmu1*d2zdmu1dv1+l$d3[,2]
      l3[,3] <- (2 * u)/(u^2 + v^2)^2+l3[,3]#-2*d1zdmu1*d2zdmu1du1+l$d3[,3]
      l3[,4] <- (2 * (u^2 - 3 * v^2) * (mu - x))/(u^2 + v^2)^3+l3[,4]#-d2zdv2*d1zdmu1-2*d1zdv1*d2zdmu1dv1-z*d3zdmu1dv2+l$d3[,4]
      l3[,5] <- (8 * u * v *(-mu + x))/(u^2 + v^2)^3+l3[,5]#-d2zdv1du1*d1zdmu1-d1zdv1*d2zdmu1du1-d1zdu1*d2zdmu1dv1-z*d3zdmu1dv1du1+l$d3[,5]
      l3[,6] <--((2 * (3 * u^2 - v^2) * (mu - x))/(u^2 + v^2)^3)+l3[,6]# w^(-3)*(-2)*d1wdv1^3+w^(-2)*2*d1wdv1*d2wdv2+3*w^(-2)*d1wdv1*d2wdv2-tau*d3wdv3-3*d1zdv1*d2wdv2-z*d3zdv3
      l3[,7] <- (2 * v * (-4 * v^2 * (u^2 + v^2) + 3 * (u^2 + v^2)^2 + 12 * v^2 * (mu - x)^2 - 6 * (u^2 + v^2) * (mu - x)^2))/(u^2 + v^2)^4+l3[,7]
      l3[,8] <- (2 * u * (u^4 - 3 * v^4 - 2 * mu^2 * (u^2 - 5 * v^2) + 4 * mu * (u^2 - 5 * v^2) * x + 10 * v^2 * x^2 - 2 * u^2 * (v^2 + x^2)))/(u^2 + v^2)^4+l3[,8]
      l3[,9] <- (2 * v * (-4 * u^2 * (u^2 + v^2) + (u^2 + v^2)^2 + 12 * u^2 * (mu - x)^2 - 2 * (u^2 + v^2) * (mu - x)^2))/(u^2 + v^2)^4+l3[,9]
      l3[,10]<- (2 * u * (-4 * u^2 * (u^2 + v^2) + 3 * (u^2 + v^2)^2 + 12 * u^2 * (mu - x)^2 - 6 * (u^2 + v^2) * (mu - x)^2))/(u^2 + v^2)^4+l3[,10]

      #order mumumumu, mumumuv, mumumuu,
      #      mumuvv, mumuvu, mumuuu
      #      muvvv, muvvu, muvuu
      #      muuuu, vvvv, vvvu,
      #      vvuu, vuuu, uuuu

      l4 <- l$h4

      l4[,1]<-l4[,1]
      l4[,2]<-l4[,2]
      l4[,3]<-l4[,3]
      l4[,4]<-(2 * (u^2 - 3 * v^2))/(u^2 + v^2)^3+ l4[,4]#-2*d2zdmu1dv1^2-2*d1zdmu1*d3zdmu1dv2+l4[,4]
      l4[,5]<--((8 * u * v)/(u^2 + v^2)^3) + l4[,5]
      l4[,6]<-(-6 * u^2 + 2 * v^2)/(u^2 + v^2)^3+ l4[,6]#-2*d2zdmu1du1^2-2*d1zdmu1*d3zdmu1du2+l4[,6]
      l4[,7]<-(24 * (-u^2 * v + v^3) * (mu - x))/(u^2 + v^2)^4+l4[,7]
      l4[,8]<- -((8 * u * (u^2 - 5 * v^2) * (mu - x))/(u^2 + v^2)^4)+l4[,8]
      l4[,9]<-(24 * u * (u - v) * (u + v) * (mu - x))/(u^2 + v^2)^4+l4[,9]
      l4[,10]<- -((8 * u * (u^2 - 5 * v^2) * (mu - x))/(u^2 + v^2)^4)+l4[,10]
      l4[,11]<-(6 * (u^6 - 5 * u^4 * v^2 - 5 * u^2 * v^4 + v^6 - 2 * mu^2 * (u^4 - 10 * u^2 * v^2 + 5 * v^4) + 4 * mu * (u^4 - 10 * u^2 * v^2 + 5 * v^4) * x - 2 * (u^4 - 10 * u^2 * v^2 + 5 * v^4) * x^2))/(u^2 + v^2)^5+l4[,11]
      l4[,12]<-(24 * u * v * (-u^4 + v^4 + mu^2 * (3 * u^2 - 5 * v^2) + 2 * mu * (-3 * u^2 + 5 * v^2) * x + (3 * u^2 - 5 * v^2) * x^2))/(u^2 + v^2)^5+l4[,12]
      l4[,13]<-(4 * mu^2 * (5 * u^4 - 38 * u^2* v^2 + 5 * v^4) - 6 * (u^6 - 5 * u^4 * v^2 - 5 * u^2 * v^4 + v^6) - 8 * mu * (5 * u^4 - 38 * u^2 * v^2 + 5 * v^4) * x + 4 * (5 * u^4 - 38 * u^2 * v^2 + 5 * v^4) * x^2)/(u^2 + v^2)^5+l4[,13]
      l4[,14]<-(24 * u * v * (u^4 - v^4 + mu^2 * (-5 * u^2 + 3 * v^2) + 2 * mu * (5 * u^2 - 3 * v^2) * x - 5 * u^2 * x^2 + 3 * v^2 * x^2))/(u^2 + v^2)^5+l4[,14]
      l4[,15]<-(6 * (u^6 + v^6 - 2 * mu^2 * (5 * u^4 - 10 * u^2 * v^2 + v^4) + 4 * mu * (5 * u^4 - 10 * u^2 * v^2 + v^4) * x - 2 * v^4 * x^2 - 5 * u^4 * (v^2 + 2 * x^2) - 5 * u^2 * (v^4 - 4 * v^2 * x^2)))/(u^2 + v^2)^5+l4[,15]

      #Set third and fourth derivatives as attributes
      attr(out,"l3")<-l3
      attr(out,"l4")<-l4
    }
  }

  #Return ouptut
  return(out)
}

#' @describeIn dnormhnorm distribution function for the normal-halfnormal distribution.
#' @param p vector of probabilities.
#' @param lower.tail logical; if TRUE (default), probabilities are \eqn{P[X \le x]}  otherwise, \eqn{P[X > x]}.
#' @export
pnormhnorm <- function(q, mu=0, sigma_v=1, sigma_u=1, s=-1, deriv=0, xg=NULL, lower.tail = TRUE, log.p = FALSE){
  #Distribution function of the normhnorm distribution

  #Check for correct inputs
  if (any(sigma_v <= 0)){
    stop(paste("sigma_v must be positive", "\n", ""))
  }

  if (any(sigma_u <= 0)){
    stop(paste("sigma_u must be positive", "\n", ""))
  }

  if (any(!s%in%c(-1,1))){
    stop(paste("s must be {-1, 1}", "\n", ""))
  }

  n<-length(q)
  v<-sigma_v
  u<-sigma_u
  a<-s*u/v
  w<-sqrt(v^2+u^2)
  tau<-1/w
  z<-(q-mu)*tau
  #Transform parameters
  out <-stats::pnorm(z)-2*vec_T.Owen(z, a)

  # out <- sn::psn(x=q, xi=para$xi, omega=para$omega, alpha=para$alpha)

  if(deriv>0){
    #First derivatives of w wrt v,u
    d1wdv1<-v*tau#v*w^(-1)
    d1wdu1<-u*tau#u*w^(-1)

    #Second derivatives of w wrt v,u
    d2wdv2<-tau+tau*(-1)*d1wdv1^2#tau+v*tau^2*(-1)*d1wdv1#w^(-1)-v^2*w^(-3)
    d2wdv1du1<-tau*(-1)*d1wdv1*d1wdu1#v*tau^2*(-1)*d1wdu1#-v*u*w^(-3)
    d2wdu2<-tau+tau*(-1)*d1wdu1^2#tau+u*tau^2*(-1)*d1wdu1#w^(-1)-u^2*w^(-3)

    #First derivatives of a wrt v, u
    d1adv1<-s*u*v^(-2)*(-1)
    d1adu1<-s*v^(-1)

    #Second derivatives of a wrt v, u
    d2adv2<-s*u*v^(-3)*2
    d2adv1du1<-s*v^(-2)*(-1)
    d2adu2<-0

    #First derivatives of z wrt mu, v, u
    d1zdmu1<--w^(-1)
    d1zdv1<--z*w^(-1)*d1wdv1
    d1zdu1<--z*w^(-1)*d1wdu1

    #First derivative of pnorm wrt z
    d1pnormdz1<-stats::dnorm(z) #exp(-(z^2/2))/sqrt(2*pi)

    #First derivatives of -2*T.Owen wrt z, a
    d1T.Owendz1<-stats::dnorm(z)*erf(a*z/sqrt(2)) #(exp(-(z^2/2)) * erf((a* z)/sqrt(2)))/sqrt(2 * pi)
    d1T.Owenda1<--(exp(-(1/2)* (1 + a^2) *z^2)/((1 + a^2) * pi))

    #First derivatives of cdf wrt z, a
    # d1cdfdz1<-(exp(-(z^2/2)) * (1 + erf((a * z)/sqrt(2))))/sqrt(2*pi)
    # d1cdfda1<--(exp(-(1/2) * (1 + a^2) * z^2))/((1 + a^2) *pi)

    #Second derivatives of z wrt mu, v, su
    d2zdmu2<-0
    d2zdmu1dv1<-w^(-2)*d1wdv1
    d2zdmu1du1<-w^(-2)*d1wdu1
    d2zdv2<--z*w^(-2)*(-2)*d1wdv1^2-z*w^(-1)*d2wdv2
    d2zdv1du1<--z*w^(-2)*(-2)*d1wdv1*d1wdu1-z*w^(-1)*d2wdv1du1
    d2zdu2<--z*w^(-2)*(-2)*d1wdu1^2-z*w^(-1)*d2wdu2

    #Second derivatives of pnorm wrt z
    d2pnormdz2<--z*stats::dnorm(z)# -((exp(-(z^2/2))* z)/sqrt(2*pi))

    #Second derivatives of -2*T.Owen wrt z, a
    d2T.Owendz2<-(exp(-(z^2/2))* (2 * a *exp(-(1/2) * a^2 * z^2) - sqrt(2*pi) * z * erf((a *z)/sqrt(2))))/(2*pi)
    d2T.Owendz1da1<-(exp(-(1/2) * (1 + a^2) * z^2) * z)/pi
    d2T.Owenda2<-(a *exp(-(1/2) * (1 + a^2) * z^2) * (2 + (1 + a^2) * z^2))/((1 + a^2)^2 *pi)

    #First derivatives of cdf wrt z, a
    # d2cdfdz2<-(exp(-(1/2) * (1 + a^2) * z^2) * (2 * a + exp((a^2 * z^2)/2) *sqrt(2 *pi)*z *(-2 + erfc((a * z)/sqrt(2)))))/(2* pi)
    # d2cdfdz1a1<-(exp(-(1/2) *  (1 + a^2) * z^2) * z)/pi
    # d2cdfda2<-(a * exp(-(1/2) * (1 + a^2) * z^2) * (2 + (1 + a^2) * z^2))/((1 + a^2)^2 *pi)

    g3<-NULL
    g4<-NULL


    dpnorm<-chainrule(f1=d1pnormdz1, f2=d2pnormdz2, f3=NULL, f4=NULL,
                      g1=cbind(d1zdmu1, d1zdv1, d1zdu1),
                      g2=cbind(d2zdmu2, d2zdmu1dv1, d2zdmu1du1, d2zdv2, d2zdv1du1, d2zdu2),
                      g3=NULL, g4=NULL,
                      deriv=deriv, xg=xg)

    #Calculate l1,l2,l3,l4
    ## the first derivatives
    ## order mu, v, u
    l1 <- dpnorm$h1

    l1[,1] <--((exp(-((mu - q)^2/(2 * (u^2 + v^2)))) * erf((s * u * (-mu + q))/(sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2 *pi) * sqrt(u^2 + v^2))) + l1[,1]
    l1[,2] <-(exp(-(((1 + (s^2 * u^2)/v^2) * (mu - q)^2)/(2 * (u^2 + v^2))))* s * u)/(pi * (s^2 * u^2 + v^2)) + (exp(-((mu - q)^2/(2 * (u^2 + v^2)))) *
                                                                                                                   v * (mu - q) * erf((s * u * (-mu + q))/(sqrt(2)* v *sqrt(u^2 + v^2))))/(sqrt(2*pi)* (u^2 + v^2)^(3/2)) +l1[,2]
    l1[,3] <--((exp(-(((1 + (s^2 * u^2)/v^2) * (mu - q)^2)/(2 * (u^2 + v^2))))* s * v)/(pi * (s^2 * u^2 + v^2))) + (exp(-((mu - q)^2/(2 * (u^2 + v^2))))* u * (mu - q) * erf((s * u * (-mu + q))/(sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2*pi)* (u^2 + v^2)^(3/2)) + l1[,3]

    ## the second derivatives
    ## order mumu, muv, muu, vv, vu, uu
    l2 <- dpnorm$h2

    l2[,1] <- (exp(-((mu - q)^2/(2 * (u^2 + v^2)))) * (2 * exp(-((s^2 * u^2 * (mu - q)^2)/(2 * v^2 * (u^2 + v^2))))* s * u * sqrt(u^2 + v^2) +
                                                         sqrt(2*pi)* v * (mu - q) * erf((s * u * (-mu + q))/(sqrt(2) * v * sqrt(u^2 + v^2)))))/(2 * pi * v * (u^2 + v^2)^(3/2)) + l2[,1]
    l2[,2] <- -((exp(-(((s^2 * u^2 + v^2) * (mu - q)^2)/(2 * v^2 * (u^2 + v^2))))* s * u * (u^2 + 2 * v^2) * (mu - q))/(pi* v^2 *(u^2 + v^2)^2)) + (
      exp(-((mu - q)^2/(2 * (u^2 + v^2)))) * v * (u^2 + v^2 - (mu - q)^2) * erf((s * u * (-mu + q))/(
        sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2 *pi)* (u^2 + v^2)^(5/2)) + l2[,2]
    l2[,3] <- (exp(-(((s^2 * u^2 + v^2) * (mu - q)^2)/(2 * v^2 * (u^2 + v^2)))) * s * v * (mu - q))/(pi* (u^2 + v^2)^2) + (exp(-((mu - q)^2/(2 * (u^2 + v^2))))*
                                                                                                                             u * (u^2 + v^2 - (mu - q)^2) * erf((s * u * (-mu + q))/( sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2 * pi)* (u^2 + v^2)^(5/2)) + l2[,3]
    l2[,4] <- (exp(-(((1 + (s^2 * u^2)/v^2) * (mu - q)^2)/(2 * (u^2 + v^2))))* s * u * (-2 * v^4 * (u^2 + v^2)^3 +  mu^2 * (s^2 * u^2 + v^2) * (2 * u^2 * v^4 + 3 * v^6 +
                                                                                                                                                  s^2 * u^2 * (u^2 + 2 * v^2)^2) - 2 * mu * (s^2 * u^2 + v^2) * (2 * u^2 * v^4 + 3 * v^6 + s^2 * u^2 * (u^2 + 2 * v^2)^2) * q + (s^2 * u^2 + v^2) * (2 * u^2 * v^4 +  3 * v^6 + s^2 * u^2 * (u^2 + 2 * v^2)^2) * q^2))/(pi * v^3 * (u^2 + v^2)^3 * (s^2 * u^2 + v^2)^2) + (
                                                                                                                                                    exp(-((mu - q)^2/( 2 * (u^2 + v^2)))) * (u^4 - u^2 * v^2 - 2 * v^4 + v^2 * (mu - q)^2) * (mu - q) * erf((s * u * (-mu + q))/(sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2 *pi)* (u^2 + v^2)^(7/2)) + l2[,4]
    l2[,5] <- (exp(-(((1 + (s^2 * u^2)/v^2) * (mu - q)^2)/(2 * (u^2 + v^2)))) * s * (-(s * u - v) * (s * u + v) * (u^2 + v^2)^3 -
                                                                                       mu^2 * (s^2 * u^2 + v^2) * ((-1 + s^2) * u^4 + (-1 + 2 * s^2) * u^2 * v^2 + v^4) + 2 * mu * (s^2 * u^2 + v^2) * ((-1 + s^2) * u^4 + (-1 + 2 * s^2) * u^2 * v^2 +
                                                                                                                                                                                                          v^4) * q - (s^2 * u^2 + v^2) * ((-1 + s^2) * u^4 + (-1 + 2 * s^2) * u^2 * v^2 + v^4) * q^2))/(pi* (u^2 + v^2)^3 * (s^2 * u^2 + v^2)^2) + (
                                                                                                                                                                                                            exp(-((mu - q)^2/(2 * (u^2 + v^2))))* u * v * (-3 * (u^2 + v^2) + (mu - q)^2) * (mu - q) * erf((s * u * (-mu + q))/(sqrt(2) * v * sqrt(u^2 + v^2))))/(sqrt(2 *pi) * (u^2 + v^2)^(7/2)) + l2[,5]
    l2[,6] <-(exp(-(((1 + (s^2 * u^2)/v^2) * (mu - q)^2)/(2 * (u^2 + v^2))))*s * u * v * (2 * s^2 * (u^2 + v^2)^3 + mu^2 * (s^2 * u^2 + v^2) * (-u^2 + (-2 + s^2) * v^2) -  2 * mu * (s^2 * u^2 + v^2) * (-u^2 + (-2 + s^2) * v^2) * q + (s^2 * u^2 +
                                                                                                                                                                                                                                            v^2) * (-u^2 + (-2 + s^2) * v^2) * q^2))/(pi* (u^2 + v^2)^3 * (s^2 * u^2 + v^2)^2) + (exp(-((mu - q)^2/(2 * (u^2 + v^2)))) * (mu - q) * (mu^2 * u^2 - 2 * u^4 + v^4 - 2 * mu * u^2 * q +
                                                                                                                                                                                                                                                                                                                                                                                       u^2 * (-v^2 + q^2)) * erf((s * u * (-mu + q))/(sqrt(2)* v * sqrt(u^2 + v^2))))/(sqrt(2 *pi) * (u^2 + v^2)^(7/2)) + l2[,6]

    #Set gradient and hessian as attributes
    attr(out,"gradient")<-l1
    attr(out,"hessian")<-l2
  }

  #Correct for numerical inaccuracy
  out[out>1-1e-14]<-1-1e-14
  out[out<0+1e-14]<-0+1e-14

  #Compute upper tail
  if(!lower.tail){
    out<-1-out
  }

  #Logarithmize output
  if(log.p){
    out<-log(out)
  }

  #Return output
  return(out)
}

#' @describeIn dnormhnorm quantile function for the normal-halfnormal distribution.
#' @param q vector of quantiles.
#' @export
qnormhnorm <- function(p, mu=0, sigma_v=1, sigma_u=1, s=-1, lower.tail = TRUE, log.p = FALSE){
  #Quantile function of the normhnorm distribution

  #Check for correct inputs
  if (any(sigma_v <= 0)){
    stop(paste("sigma_v must be positive", "\n", ""))
  }

  if (any(sigma_u <= 0)){
    stop(paste("sigma_u must be positive", "\n", ""))
  }

  if (any(!s%in%c(-1,1))){
    stop(paste("s must be {-1, 1}", "\n", ""))
  }

  #Transform parameters
  para<-reparametrize(mu=mu, sigma_v=sigma_v, sigma_u=sigma_u, s=s)

  #Compute upper tail
  if(!lower.tail){
    p<-1-p
  }

  #Compute quantiles
  out<-c()
  if(is.matrix(p)){
    N<-nrow(p)
  } else {
    N<-length(p)
  }

  # for(n in 1:N){
  #   out[n]<-sn::qsn(p=p[n], xi=para$xi[n], omega=para$omega[n], alpha=para$alpha[n])
  # }
  out<-vec_qsn(p=p, xi=para$xi, omega=para$omega, alpha=para$alpha)

  #Logarithmize output
  if(log.p){
    out<-log(out)
  }

  #Return output
  return(out)
}

#' @describeIn dnormhnorm random number generation for the normal-halfnormal distribution.
#' @param n number of observations.
#' @export
rnormhnorm <- function(n, mu=0, sigma_v=1, sigma_u=1, s=-1){
  #Function to generate n random numbers from the normhnorm distribution

  #Check for correct inputs
  if (any(sigma_v <= 0)){
    stop(paste("sigma_v must be positive", "\n", ""))
  }

  if (any(sigma_u <= 0)){
    stop(paste("sigma_v must be positive", "\n", ""))
  }

  if (any(!s%in%c(-1,1))){
    stop(paste("s must be {-1, 1}", "\n", ""))
  }

  #Transform parameters
  para<-reparametrize(mu=mu, sigma_v=sigma_v, sigma_u=sigma_u, s=s)

  out<-sn::rsn(n=n, xi=para$xi, omega=para$omega, alpha=para$alpha)

  #Remove attributes
  attributes(out) <- NULL

  #Return output
  return(out)
}
