% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S3_predict.R
\name{predict.lm_robust}
\alias{predict.lm_robust}
\title{Predict method for \code{\link{lm_robust}} object}
\usage{
\method{predict}{lm_robust}(object, newdata, se.fit = FALSE,
  interval = c("none", "confidence", "prediction"), alpha = 0.05,
  na.action = na.pass, pred.var = NULL, weights, ...)
}
\arguments{
\item{object}{an object of class 'lm_robust'}

\item{newdata}{a data frame in which to look for variables with which to predict}

\item{se.fit}{logical. Whether standard errors are required, default = FALSE}

\item{interval}{type of interval calculation. Can be abbreviated, default = none}

\item{alpha}{numeric denoting the test size for confidence intervals}

\item{na.action}{function determining what should be done with missing
values in newdata. The default is to predict NA.}

\item{pred.var}{the variance(s) for future observations to be assumed for
prediction intervals.}

\item{weights}{variance weights for prediction. This can be a numeric
vector or a bare (unquoted) name of the weights variable in the supplied
newdata.}

\item{...}{other arguments, unused}
}
\description{
Predict method for \code{\link{lm_robust}} object
}
\details{
Produces predicted values, obtained by evaluating the regression
function in the frame `newdata`` for fits from \code{lm_robust} and
\code{lm_lin}. If the logical se.fit is TRUE, standard errors of the
predictions are calculated. Setting intervals specifies computation of
confidence or prediction (tolerance) intervals at the specified level,
sometimes referred to as narrow vs. wide intervals.

The equation used for the standard error of a prediction given a row of
data \eqn{x} is:

\eqn{\sqrt(x \Sigma x')},

where \eqn{\Sigma} is the estimated variance-covariance matrix from
\code{lm_robust}.

The prediction intervals are for a single observation at each case in
`newdata` with error variance(s) `pred.var`. The the default is to assume
that future observations have the same error variance as those used for
fitting, which is gotten from the fit \code{\link{lm_robust}} object. If
weights is supplied, the inverse of this is used as a scale factor. If the
fit was weighted, the default is to assume constant prediction variance,
with a warning.
}
\examples{

# Set seed
set.seed(42)

# Simulate data
n <- 10
dat <- data.frame(y = rnorm(n), x = rnorm(n))

# Fit lm
lm_out <- lm_robust(y ~ x, data = dat)
# Get predicted fits
fits <- predict(lm_out, newdata = dat)
# With standard errors and confidence intervals
fits <- predict(lm_out, newdata = dat, se.fit = TRUE, interval = "confidence")

# Use new data as well
new_dat <- data.frame(x = runif(n, 5, 8))
predict(lm_out, newdata = new_dat)

# You can also supply custom variance weights for prediction intervals
new_dat$w <- runif(n)
predict(lm_out, newdata = new_dat, weights = w, interval = "prediction")

}
