\name{garchModelling}

\alias{garchModelling}

\alias{garchSim}
\alias{aparchSim}

\alias{garchFit}

\alias{fGARCH}

\alias{predict.fGARCH}
\alias{print.fGARCH}
\alias{summary.fGARCH}
\alias{print.summary.fGARCH}
\alias{plot.fGARCH}

\alias{fitted.values.fGARCH}
\alias{residuals.fGARCH}


\title{Univariate GARCH Time Series Modelling}


\description{

	This ia a collection of functions to simulate artificial GARCH 
	time series processes, to fit the parameters of univariate time 
	series to GARCH models, to perform a diagnostic analysis of the 
	fit, and to predict future values of the time series.
	
	The family of GARCH time series models includes the following 
	processes:
	
	\tabular{rll}{
	1	\tab	garch 	
		\tab    generalized AR conditional heteroskedastic models, \cr
	2	\tab    aparch 
	    \tab    asymmetretic power ARCH models. }
	
	The functions are written entirely in R which makes them open 
	sourced to further extensions and highly controllable for the 
	implementation.  
}


\usage{
garchSim(model = list(omega = 1e-6, alpha = 0.1, beta = 0.8, mu = 0), 
	n = 100, innov = NULL, n.start = 100, start.innov = NULL, 
	rand.gen = rnorm, \dots)
aparchSim(model = list(omega = 1.0e-4, alpha = 0.1, gamma = 0.1, 
	beta = 0.8, mu = 0, delta = 1), n = 1000, innov = NULL, n.start = 100, 
	start.innov = NULL, rand.gen = rnorm, \dots)	
		
garchFit(formula.mean = ~arma(0, 0), formula.var = ~garch(1, 1), series = x,
	fixed=NA, pre.mean=NA, pre.var=NA, h.start=NA, llh.start=NA, init=NA, 
	delta.est=FALSE, delta.par=NA,
	cond.dist=c("dnorm", "dt"), dist.est=FALSE, dist.par=NA, trace=FALSE, 
	algorithm=c("nlm", "optim"), ...)

\method{predict}{fGARCH}(object)

\method{print}{fGARCH}(object, \dots)
\method{print}{fGARCH}(object, \dots)
\method{summary}{fGARCH}(object)
\method{print.summary}{fGARCH}(object)
}


\arguments{

	\item{ }{\emph{Simulation:\cr}}
 	 
  	\item{model}{
		List of GARCH model parameters:\cr
		\code{omega} the constant coefficient of the variance equation, 
			must be defined; 
		\code{alpha} the vector of autoregressive coefficients, must be 
			defined; 
		\code{beta} the vector of variance coefficients, optional; 
		\code{h0}.
		\cr
		List of APARCH model parameters:\cr
		\code{omega} the omega parameter, a numeric value;
		\code{alpha} the "ar" coefficients, a numeric vector;
		\code{gamma} the asymmetry coefficients, a numeric vector of same 
		  	length as "alpha";
		\code{alpha.lags} an integer vector denoting to which lags the 
		  	"alpha" coefficients belong;
		\code{beta} the "ma" coefficients, a numeric vector;
		\code{beta.lags} an integer vector denoting to which lags the 
		  	"beta" coefficients belong;
		\code{delta} the "delta" exponent.
		The default model is Taylor Schwert GARCH(1,1) model.
		}
  	\item{n}{
		length of output series, an integer value.
		}
  	\item{innov}{
		a univariate time series or vector of innovations to 
		produce the series. If not provided, \code{innov} will 
		be generated using \code{rand.gen}. Missing values are 
		not allowed. A numeric vector.
		}		
  	\item{n.start}{
		the number of start-up values. The start-up innovations 
		will be generated by \code{rand.gen} if \code{start.innov} 
		is not provided. An integer value by default 100.
		Missing values are not allowed.
		}
  	\item{start.innov}{	
		a univariate time series or vector of innovations to be 
		used as start up values. Missing values are not allowed.
		A numeric vector.
		}
  	\item{rand.gen}{
		optional, a function to generate the innovations.
		\cr\cr
		}	
		
	\item{ }{\emph{Parameter Estimation:\cr}
	
	\item{formula.mean}{
		...
		}
	\item{formula.var}{
		...
		}			
	\item{series}{
		an univariate vector or time series.
		}
	\item{pre.mean}{
		...
		}		
	\item{pre.var}{
		...
		}		
	\item{h.start}{
		...
		}
	\item{llh.start}{
		...
		}
	\item{init}{
		...
		}	
	\item{delta.est, delta.par}{
		...
		}
	\item{cond.dist, dist.est, dist.par}{
		...
		}
	\item{algorithm}{
		...
		}
	\item{trace}{
		...
		\cr\cr
		}
    
		
    \item{ }{\emph{Methods:\cr}
    
	
	\item{object}{
		a fitted GARCH or APARCH object of class \code{"fGARCH"}.
		}

}


\details{
  
	\bold{Models:}
	Two families of models can be investigated: The GARCH family and
	the APARCH family. The GARCH family includes Engle's ARCH model
	and Bollerslev's GARCH model. The second family includes 
	
	
	\bold{GARCH Simulation:}
	\cr
	\code{garchSim} simulates a GARCH(p,q) process, a heteroskedastic 
	time series with \code{p} autoregressive and \code{q} moving  
	average coefficients as introduced by Bollerslev [1986]. The  
	default time series is generated from the standard GARCH(1,1)  
	process with coefficients \code{omega = 1e-6}, \code{alpha = 0.1},  
	and \code{beta = 0.8}. The mean \code{mu} is zero by default.
	To simulate Engle's [1982] ARCH(p) time series model one has to  
	set \code{beta = 0}.
	\cr
  	
  	\bold{APARCH Simulation:}
  	\cr
  	\code{aparchSim} allows to estimate the family of APARCH models,
  	where APARCH abbreviates "Asymmetric Power ARCH". The model was
  	introduced by Ding, Granger and Engle [1993].
  	\cr
	For Engle's ARCH(p) model one has to set \code{beta.lags=1}, beta=0, 
   	and \code{opt$beta=FALSE}. 
   	\cr
	For Bollerslev's GARCH(p,q) model, just use it.
	\cr
	For Taylor-Schwert's TS-GARCH(p,q) model one has to set \code{delta=1}.
	\cr
	For Gosten et al.'s GJR(p,q) model one has to set \code{gamma} and 
   	\code{opt.gamma=TRUE}.
   	\cr
	For the extimation of parameters from asymmetric models and models 
   	with unknown \code{delta} exponents different from 2 one has to set 
   	the arguments \code{opt$gamma=TRUE} and \code{opt$delta=TRUE}.
	In addition from three types of innovations, normal distributed,
   	Student-t and symmetric stable can be chosen.
   	\cr
   	
   	\bold{Specifying the Model:}
	\cr	
	\code{garchFit} uses a Quasi-Newton optimizer to find the maximum
  	likelihood estimates of the conditionally normal model. The first
  	max(p,q) values are assumed to be fixed. The optimizer uses a hessian
  	approximation computed from the BFGS update.  Only a Cholesky factor
  	of the Hessian approximation is stored. For more details see Dennis
  	et al. (1981), Dennis and Mei (1979), Dennis and More (1977), and
  	Goldfarb (1976). The gradient is either computed analytically or
  	using a numerical approximation.
  	\cr
  	\code{aparchFit} allows to estimate a whole family of ARCH models.
	For Engle's ARCH(p) model one has to set \code{beta.lags=1}, beta=0, 
   	and \code{opt$beta=FALSE}. 
	For Bollerslev's GARCH(p,q) model, just use it.
	For a Taylor-Schwert TS-GARCH(p,q) model one has to set \code{delta=1}.
	For Gosten et al.'s GJR(p,q) model one has to set \code{gamma} and 
   	code{opt.gamma=TRUE}.
	For the extimation of parameters from asymmetric models and models 
   	with unknown \code{delta} exponents different from 2 one has to set 
   	the arguments \code{opt$gamma=TRUE} and \code{opt$delta=TRUE}.
	In addition from three types of innovations, normal distributed,
   	Student-t and symmetric stable can be chosen.
  	
  	\bold{Initializing the Series:}
  	\cr
  	
  	\bold{Fitting Submodels:}
  	\cr
  		
  	\bold{Selecting the Conditional Distribution:}
  	\cr
  	
  	\bold{Selecting the Optimization Algorithm:}
  	\cr
  	
  	\bold{Fit the GARCH Model:}
  	\cr
  	
  	\bold{Perform a Diagnostic Analysis}
  	\cr
  	
  	\bold{Predicting the next Values:}
  	\cr
  	
  	\bold{APARCH Modelling:}
  	\cr
  	
}


\value{

	\code{garchSim}\cr
	\code{aparchSim}\cr
	 return objects of class \code{ts}.
	 \cr

	\code{garchFit}\cr
	returns a list of class \code{fGARCH} with at least 
	the following components:

  	\item{order}{
		the order of the fitted model.}
  	\item{coef}{
		estimated GARCH coefficients for the fitted model.}
  	\item{n.likeli}{
		the negative log-likelihood function evaluated at the
		coefficient estimates (apart from some constant).}
  	\item{n.used}{
		the number of observations of \code{x}.}
  	\item{residuals}{
		the series of residuals.}
  	\item{fitted.values}{
		the bivariate series of conditional standard
		deviation predictions for \code{x}.}
  	\item{series}{
		the name of the series \code{x}.}
  	\item{frequency}{
		the frequency of the series \code{x}.}
  	\item{call}{
		the call of the \code{garch} function.}
  	\item{asy.se.coef}{
		the asymptotic-theory standard errors of the
		coefficient estimates.}
	
}


\references{

Bera A.K., Higgins M.L. (1993);
	\emph{ARCH Models: Properties, Estimation and Testing},
	Journal of Economic Surveys 7, 305--362.
  
Bollerslev T. (1986);
	\emph{Generalized Autoregressive Conditional Heteroscedasticity},
	Journal of Econometrics} 31, 307--327.

Engle R.F. (1982);
	\emph{Autoregressive Conditional Heteroscedasticity with Estimates 
	of the Variance of United Kingdom Inflation},
	Econometrica 50, 987--1008.

Dennis J.E., Gay D.M., Welsch R.E. (1981);
	\emph{Algorithm 573 -- An Adaptive Nonlinear Least-Squares Algorithm};
	ACM Transactions on Mathematical Software 7, 369--383.

Dennis J.E., Mei H.H.W. (1979);
	\emph{Two New Unconstrained Optimization Algorithms which use 
	Function and Gradient Values},
	J. Optim. Theory Applic. 28, 453--482.

Dennis J.E., More J.J. (1977);
	\emph{Quasi-Newton Methods, Motivation and Theory};
	SIAM Revue 19, 46--89.

Goldfarb D. (1976);
	\emph{Factorized Variable Metric Methods for Unconstrained Optimization},
	Math. Comput. 30, 796--811.
	
}


\author{

    Diethelm Wuertz for this \R-port.
    
}


\seealso{
  
	\code{\link{armaModelling}}.
	
}


\examples{
## 	GARCH:
	# Model Bollerslev's GARCH(1,1) Process:
   	# Simulate a time series of length 1000:
   	ts <- garchSim(model=list(omega=1e-6, alpha=0.1, beta=0.8, 
   	  	h0=1e-7), n=1000)
   	Continue <- readline("Press any key > ") 
   	# Estimate the parameters:
   	fit <- garchFit(ts, order=c(1,1))
   	print(fit)
   	Continue <- readline("Press any key > ") 
   	# Summary and Diagnostic Analysis:
   	summary(fit)
   	# 5 Steps ahead Forecasts:
   	Continue <- readline("Press any key > ") 
   	predict(fit, n.ahead=5, se.fit=TRUE)
    Continue <- readline("Press any key > ") 

## 	APARCH:
	# Taylor-Schwert TS-GARCH(1,5[1,5]) subset model, delta=1, fixed:
   	# Simulate a time series of length 1000:
   	ts <- aparchSim(model=list(omega=1e-6, alpha=0.1, gamma=0, 
   		alpha.lags=1, beta=c(0.5, 0.3), beta.lags=c(1, 5), delta=1),
        innov=rnorm(5000), start.innov=rnorm(1000))
    ts.plot(ts, main="Subset TS GARCH(1,5) - Mormal")
    acf(abs(ts), lag.max=20)
    # Estimate the parameters:
    aparchFit(ts, order=list(alpha.lags = 1, beta.lags = c(1, 5), delta = 1),
        opt=list(gamma=FALSE, delta=FALSE, disparm=FALSE), doprint=FALSE)
    # For more examples type: xmpfSeries() ...
}


\keyword{Time Series}
\keyword{fSeries}

