% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fairness_check.R
\name{fairness_check}
\alias{fairness_check}
\title{Fairness check}
\usage{
fairness_check(
  x,
  ...,
  protected = NULL,
  privileged = NULL,
  cutoff = NULL,
  label = NULL,
  epsilon = NULL,
  verbose = TRUE,
  colorize = TRUE
)
}
\arguments{
\item{x}{object created with \code{\link[DALEX]{explain}} or of class \code{fairness_object}}

\item{...}{possibly more objects created with \code{\link[DALEX]{explain}} and/or objects of class \code{fairness_object}}

\item{protected}{factor, protected variable (also called sensitive attribute), containing privileged and unprivileged groups}

\item{privileged}{factor/character, one value of \code{protected}, in regard to what subgroup parity loss is calculated}

\item{cutoff}{numeric, vector of cutoffs (thresholds) for each value of protected variable, affecting only explainers.}

\item{label}{character, vector of labels to be assigned for explainers, default is explainer label.}

\item{epsilon}{numeric, boundary for fairness checking}

\item{verbose}{logical, whether to print information about creation of fairness object}

\item{colorize}{logical, whether to print information in color}
}
\value{
An object of class \code{fairness_object} which is a list with elements:
\itemize{
\item parity_loss_metric_data - data.frame containing parity loss for various fairness metrics. Created with following metrics:
\itemize{

\item{TPR}{ - True Positive Rate (Sensitivity, Recall)}
\item{TNR}{ - True Negative Rate (Specificity)}
\item{PPV}{ - Positive Predictive Value (Precision)}
\item{NPV}{ - Negative Predictive Value}
\item{FNR}{ - False Negative Rate}
\item{FPR}{ - False Positive Rate}
\item{FDR}{ - False Discovery Rate}
\item{FOR}{ - False Omission Rate}
\item{TS}{ - Threat Score}
\item{STP}{ - Statistical Parity}
\item{ACC}{ - Accuracy}
\item{F1}{ - F1 Score}
\item{MCC}{ - Matthews correlation coefficient}
}

\item{groups_data}{ - metrics across levels in protected variable}
\item{groups_confusion_matrices}{ - confusion matrices for each subgroup}
\item{explainers}{ - list of \code{DALEX} explainers used to create object}
\item{cutoffs}{ - list of cutoffs for each explainer and subgroup}
\item{fairness_check_data}{ - \code{data.frame} used for for plotting \code{fairness_object}}
\item{...}{ - other parameters passed to function}
}
}
\description{
Fairness check creates \code{fairness_object} which measures different fairness metrics and wraps data, explainers and parameters in useful object. This is fundamental object in this package.
It enables to visualize fairness metrics and models in many ways and compare models on both fairness and performance level. Fairness check acts as merger and wrapper for explainers and fairness objects.
While other fairness objects values are not changed, fairness check assigns cutoffs and labels to provided explainers so same explainers with changed labels/cutoffs might be gradually added to fairness object.
Users through print and plot methods may quickly check values of most popular fairness metrics. More on that topic in details.
}
\details{
Metrics used are made for each subgroup, then base metric score is subtracted leaving loss of particular metric.
If absolute loss is greater than epsilon than such metric is marked as "not passed". It means that values of metrics should be within (-epsilon,epsilon) boundary.
Epsilon value can be adjusted to user's needs. There are some metrics that might be derived from existing metrics (For example Equalized Odds - equal TPR and FPR for all subgroups).
That means passing 5 metrics in fairness check asserts that model is even more fair. In \code{fairness_check} models must always predict positive result. Not adhering to this rule
may lead to misinterpretation of the plot. More on metrics and their equivalents:
\url{https://fairware.cs.umass.edu/papers/Verma.pdf}
\url{https://en.wikipedia.org/wiki/Fairness_(machine_learning)}

Parity loss is computed as follows:
M_parity_loss = sum(abs(metric - base_metric))

where:

M - some metric mentioned above

metric - vector of metrics from each subgroup

base_metric - scalar, value of metric for base subgroup
}
\examples{
data("german")

y_numeric <- as.numeric(german$Risk) -1

lm_model <- glm(Risk~.,
                data = german,
                family=binomial(link="logit"))

rf_model <- ranger::ranger(Risk ~.,
                           data = german,
                           probability = TRUE,
                           num.trees = 200)

explainer_lm <- DALEX::explain(lm_model, data = german[,-1], y = y_numeric)
explainer_rf <- DALEX::explain(rf_model, data = german[,-1], y = y_numeric)

fobject <- fairness_check(explainer_lm, explainer_rf,
                          protected = german$Sex,
                          privileged = "male")

plot(fobject)

}
\references{
Zafar,Valera, Rodriguez, Gummadi (2017)  \url{https://arxiv.org/pdf/1610.08452.pdf}

Hardt, Price, Srebro (2016) \url{https://arxiv.org/pdf/1610.02413.pdf}

Verma, Rubin (2018) \url{https://fairware.cs.umass.edu/papers/Verma.pdf}
}
