#' @title Read Protein Surface File
#' @name read_prot
#'
#' @description Reads and processes data from protein surface files generated by the 'occluded_surface' function. The results enable important analyses of protein behavior at the atomic level.
#'
#' @param file Path or name of protein surface file (.srf).
#'
#' @details The function reads a surface file containing data calculated using the 'OS' or 'FIBOS' methodology. The file should be in the format generated by the 'occluded_surface' function.
#'
#' @return A data frame with the following columns:
#'   \describe{
#'     \item{ATOM}{atomic contact identifiers}
#'     \item{NUMBER_POINTS}{number of points (integer) on atomic surface}
#'     \item{AREA}{surface area (numeric)}
#'     \item{RAYLENGTH}{average ray length (numeric)}
#'     \item{DISTANCE}{average distance of contacts (numeric)}
#'   }
#'
#' @author Carlos Henrique da Silveira (carlos.silveira@unifei.edu.br)
#' @author Herson Hebert Mendes Soares (hersonhebert@hotmail.com)
#' @author Joao Paulo Roquim Romanelli (joaoromanelli@unifei.edu.br)
#' @author Patrick Fleming (Pat.Fleming@jhu.edu)
#'
#' @importFrom readr read_fwf
#' @importFrom dplyr filter
#' @importFrom dplyr rename
#' @importFrom stringr str_count
#' @importFrom tidyr separate
#'
#' @keywords internal

utils::globalVariables(c("X1", "X2", "X3", "X4", "X5", "X6"))
read_prot = function(file){
  if (!file.exists(file)) {
    stop("File not found: ", file)
  }
  
  dado = read_fwf(file, show_col_types = FALSE)
  dado = filter(dado, X1 == "INF")
  dado$X7 = NULL
  dado = rename(dado, INF = X1, ATOM = X2, NUMBER_POINTS = X3, AREA = X4, RAYLENGTH = X5, DISTANCE = X6)
  dado$NUMBER_POINTS = gsub("\\s\\pts", "", dado$NUMBER_POINTS)
  dado$AREA = gsub("\\s\\A2", "", dado$AREA)
  dado$RAYLENGTH = gsub("\\s\\Rlen", "", dado$RAYLENGTH)
  dado$NUMBER_POINTS = as.integer(dado$NUMBER_POINTS)
  dado$AREA = as.double(dado$AREA)
  dado$RAYLENGTH = as.double(dado$RAYLENGTH)
  dado$DISTANCE = as.double(dado$DISTANCE)
  dado$INF = NULL
  return(dado)
}