% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{running_centered}
\alias{running_centered}
\alias{running_scaled}
\alias{running_sharpe}
\alias{running_tstat}
\alias{running_zscored}
\title{Compare data to moments computed over a sliding window.}
\usage{
running_centered(v, window = NULL, na_rm = FALSE, min_df = 0L,
  lookahead = 0L, restart_period = 100L)

running_scaled(v, window = NULL, na_rm = FALSE, min_df = 0L,
  lookahead = 0L, restart_period = 100L)

running_zscored(v, window = NULL, na_rm = FALSE, min_df = 0L,
  lookahead = 0L, restart_period = 100L)

running_sharpe(v, window = NULL, na_rm = FALSE, compute_se = FALSE,
  min_df = 0L, restart_period = 100L)

running_tstat(v, window = NULL, na_rm = FALSE, min_df = 0L,
  restart_period = 100L)
}
\arguments{
\item{v}{a vector}

\item{window}{the window size. if given as finite integer or double, passed through.
If \code{NULL}, \code{NA_integer_}, \code{NA_real_} or \code{Inf} are given, equivalent
to an infinite window size. If negative, an error will be thrown.}

\item{na_rm}{whether to remove NA, false by default.}

\item{min_df}{the minimum df to return a value, otherwise \code{NaN} is returned.
This can be used to prevent \emph{e.g.} Z-scores from being computed on only 3
observations. Defaults to zero, meaning no restriction, which can result in 
infinite Z-scores during the burn-in period.}

\item{lookahead}{for some of the operations, the value is compared to 
mean and standard deviation possibly using 'future' or 'past' information
by means of a non-zero lookahead. Positive values mean data are taken from
the future.}

\item{restart_period}{the recompute period. because subtraction of elements can cause
loss of precision, the computation of moments is restarted periodically based on 
this parameter. Larger values mean fewer restarts and faster, though less accurate
results. Note that the code checks for negative second and fourth moments and
recomputes when needed.}

\item{compute_se}{for \code{running_sharpe}, return an extra column of the
standard error, as computed by Mertens' correction.}
}
\value{
a vector the same size as the input consisting of the adjusted version of the input.
When there are not sufficient (non-nan) elements for the computation, \code{NaN} are returned.
}
\description{
Computes moments over a sliding window, then adjusts the data accordingly, centering, or scaling,
or z-scoring, and so on.
}
\details{
Given the length \eqn{n} vector \eqn{x}, for
a given index \eqn{i}, define \eqn{x^{(i)}}{x^(i)}
as the vector of 
\eqn{x_{i-window+1},x_{i-window+2},...,x_{i}}{x_(i-window+1),x_(i-window+2),...,x_i},
where we do not run over the 'edge' of the vector. In code, this is essentially
\code{x[(max(1,i-window+1)):i]}. Then define \eqn{\mu_i}{mu_i}, \eqn{\sigma_i}{sigma_i}
and \eqn{n_i}{n_i} as, respectively, the sample mean, standard deviation and number of
non-NA elements in \eqn{x^{(i)}}{x^(i)}. 

We compute output vector \eqn{m} the same size as \eqn{x}. 
For the 'centered' version of \eqn{x}, we have \eqn{m_i = x_i - \mu_i}{m_i = x_i - mu_i}.
For the 'scaled' version of \eqn{x}, we have \eqn{m_i = x_i / \sigma_i}{m_i = x_i / sigma_i}.
For the 'z-scored' version of \eqn{x}, we have \eqn{m_i = (x_i - \mu_i) / \sigma_i}{m_i = (x_i - mu_i) / sigma_i}.
For the 't-scored' version of \eqn{x}, we have \eqn{m_i = \sqrt{n_i} \mu_i / \sigma_i}{m_i = sqrt(n_i) mu_i / sigma_i}.

We also allow a 'lookahead' for some of these operations.
If positive, the moments are computed using data from larger indices;
if negative, from smaller indices. Letting \eqn{j = i + lookahead}{j = i + lookahead}:
For the 'centered' version of \eqn{x}, we have \eqn{m_i = x_i - \mu_j}{m_i = x_i - mu_j}.
For the 'scaled' version of \eqn{x}, we have \eqn{m_i = x_i / \sigma_j}{m_i = x_i / sigma_j}.
For the 'z-scored' version of \eqn{x}, we have \eqn{m_i = (x_i - \mu_j) / \sigma_j}{m_i = (x_i - mu_j) / sigma_j}.
}
\note{
The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.
}
\examples{

if (require(moments)) {
    set.seed(123)
    x <- rnorm(5e1)
    window <- 10L
    rm1 <- t(sapply(seq_len(length(x)),function(iii) { 
                  xrang <- x[max(1,iii-window+1):iii]
                  c(sd(xrang),mean(xrang),length(xrang)) },
                  simplify=TRUE))
    rcent <- running_centered(x,window=window)
    rscal <- running_scaled(x,window=window)
    rzsco <- running_zscored(x,window=window)
    rshrp <- running_sharpe(x,window=window)
    rtsco <- running_tstat(x,window=window)
    rsrse <- running_sharpe(x,window=window,compute_se=TRUE)
    stopifnot(max(abs(rcent - (x - rm1[,2])),na.rm=TRUE) < 1e-12)
    stopifnot(max(abs(rscal - (x / rm1[,1])),na.rm=TRUE) < 1e-12)
    stopifnot(max(abs(rzsco - ((x - rm1[,2]) / rm1[,1])),na.rm=TRUE) < 1e-12)
    stopifnot(max(abs(rshrp - (rm1[,2] / rm1[,1])),na.rm=TRUE) < 1e-12)
    stopifnot(max(abs(rtsco - ((sqrt(rm1[,3]) * rm1[,2]) / rm1[,1])),na.rm=TRUE) < 1e-12)
    stopifnot(max(abs(rsrse[,1] - rshrp),na.rm=TRUE) < 1e-12)

    rm2 <- t(sapply(seq_len(length(x)),function(iii) { 
                  xrang <- x[max(1,iii-window+1):iii]
                  c(kurtosis(xrang)-3.0,skewness(xrang)) },
                  simplify=TRUE))
    mertens_se <- sqrt((1 + ((2 + rm2[,1])/4) * rshrp^2 - rm2[,2]*rshrp) / rm1[,3])
    stopifnot(max(abs(rsrse[,2] - mertens_se),na.rm=TRUE) < 1e-12)
}

}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\url{https://www.semanticscholar.org/paper/Numerically-stable-single-pass-parallel-statistics-Bennett-Grout/a83ed72a5ba86622d5eb6395299b46d51c901265}

Cook, J. D. "Accurately computing running variance."
\url{http://www.johndcook.com/standard_deviation.html}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{http://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation}
}
\seealso{
\code{\link{scale}}
}

