% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rotate.R
\name{rotate}
\alias{rotate}
\title{Automatic Factor Rotation from Random Starting Configurations}
\usage{
rotate(R = NULL, numFactors = NULL, facMethod = "fals",
  lambda = NULL, rotate = NULL, targetMatrix, numberStarts = 10,
  digits = NULL, keepAll = TRUE, rotateControl = NULL,
  faControl = NULL)
}
\arguments{
\item{R}{(Matrix) A correlation matrix.}

\item{numFactors}{(Numeric) The number of factors to extract for subsequent rotation.}

\item{facMethod}{(Character) The method used for factor extraction. The supported options are "fals" for unweighted least squares, "faml" for maximum likelihood, and "fapa" for iterated principal axis factoring. The default method is "fals".
\itemize{
  \item \strong{"fals"}: Factors are extracted using the unweighted least squares estimation procedure using the \code{\link[fungible]{fals}} function.
  \item \strong{"faml"}: Factors are extracted using the maximum likelihood estimation procedure using the \code{\link[stats]{factanal}} function.
  \item \strong{"fapa"}: Factors are extracted using the iterated principal axis factoring estimation procedure using the \code{\link{fapa}} function.
}}

\item{lambda}{(Matrix) An unrotated factor-structure matrix to be rotated.}

\item{rotate}{(Character) Designate which rotation algorithm to apply. The following are available rotation options: "oblimin", "quartimin", "targetT", "targetQ", "oblimax", "entropy", "quartimax", "varimax", "simplimax", "bentlerT", "bentlerQ", "tandemI", "tandemII", "geominT", "geominQ", "cfT", "cfQ", "infomaxT", "infomaxQ", "mccammon", "bifactorT", "bifactorQ".}

\item{targetMatrix}{(Matrix) If a target rotation is desired, provide the target to be rotated toward. Note that a matrix of numeric values will yield a fully-specified Procrustes solution. To conduct Browne's partially-specified Procrustes rotation, ensure that all factor loadings to be freely estimated are designated by "NA" values. All non-salient values are typically specified as 0 (zero).}

\item{numberStarts}{(Numeric) The number of random (orthogonal) starting configurations to use. The default value is 10 random starts}

\item{digits}{(Numeric) The number of digits to round all output to. The default is no rounding.}

\item{keepAll}{(Logical) A logical value indicating whether to keep the solutions for each random starting configuration (i.e., all factor loadings, factor correlations, and their evaluated discrepancy function). The default value is TRUE.}

\item{rotateControl}{(List) A list of control values to pass to the factor rotations in the \code{\link{rotate}} function.
\itemize{
  \item \strong{gamma}: (Numberic) This is a tuning parameter (between 0 and 1, inclusive) for an oblimin rotation.  See the GPArotation library's oblimin documentation for more details.
  \item \strong{delta}: (Numberic) This is a tuning parameter for the geomin rotation. It adds a small number (default = .01) to factor loadings before computing the geometric means in the discrepancy function.
  \item \strong{kappa}: (Numeric) The main parameterization of the Crawford-Ferguson family of factor rotations. Specific values correspond to certain rotations. For instance (adapted from GPArotation's help page), 0=Quartimax, 1/p=varimax, m/(2*p)=Equamax, (m-1)/(p+m-2)=Parsimax, 1=Factor parsimony. Note that 'p' is the number of factor indicators and 'm' is the number of common factors. The default value is 0 (Quartimax).
  \item \strong{k}: (Numeric) A specific parameter of the simplimax rotation. The default value is nrow(lambda).
  \item \strong{epsilon}: (Numeric) The rotational convergence criterion to use. The default value is 1e-5.
  \item \strong{power}: (Numeric) Raise factor loadings the the n-th power in the promax rotation. The default value is 4.
  \item \strong{norm}: (Logical) A logical value indicating whether to compute Kaiser normalization. Default is FALSE
  \item \strong{maxITR}: (Numeric) The maximum number of allowed iterations for convergence before stopping the rotation. The default value is 15,000 iterations.
}}

\item{faControl}{(List) A list of optional parameters passed to the factor extraction (\code{\link{faX}}) function.
\itemize{
  \item \strong{treatHeywood}: (Logical) In fals, if treatHeywood is true, a penalized least squares function is used to bound the communality estimates below 1.0. The default is TRUE.
  \item \strong{nStart}: (Numeric) In faml, determine the number of starting values to try. The default is 10 start values.
  \item \strong{maxCommunality}: (Numeric) In faml, set the maximum communality value for the estimated solution. The default maximum is .995.
  \item \strong{epsilon}: (Numeric) In fapa, the numeric threshold designating when the algorithm has converged. The default value is 1e-4.
  \item \strong{communality}: (Character) In fapa, the routine requires an initial communality estimate. Select how communalities are initially estimated. The default is squared multiple correlation ("SMC").
  \itemize{
    \item \strong{"SMC"}: Initial communalities are estimated by taking the squared multiple correlations of each indicator after regressing the indicator on the remaining variables. The following equation is employed to find the squared multiple correlation: \eqn{1 - 1 / diag(R^-1)}.
    \item \strong{"maxRsqr"}: Initial communalities equal the largest squared correlation in each column of the correlation matrix.
    \item \strong{"unity"}: Initial communalities equal 1.0 for all variables.
  }
  \item \strong{maxITR}: (Numeric) In fapa, the maximum number of iterations to reach convergence. The default is 15,000
}}
}
\value{
The rotate function will produce a lot of output in addition to the rotated factor pattern matrix and the factor correlations.
\itemize{
\item \strong{loadings}: (Matrix) The rotated factor solution with the lowest* evaluated discrepancy function. *This solution has the lowest discrepancy function \emph{of the examined random starting configurations}. It is not guarenteed to find the "true" global minimum. Note that multiple (or even all) local solutions can have the same discrepancy functions.
\item \strong{Phi}: (Matrix) The factor correlations of the rotated factor solution with the lowest* evaluated discrepancy function (see * in the loadings description above)
\item \strong{rotateControl}: (List) A list of the control parameters passed to the rotation (\code{\link{rotate}}) function.
\item \strong{faControl}: (List) A list of the control parameters passed to the factor extraction (\code{\link{faX}}) function.
\item \strong{localLoadings}: (List) A list of all of the local factor pattern matrices. If keepAll is FALSE, this will return NULL.
\item \strong{localPhi}: (List) A list of all of the local factor correlation matrices. If keepAll is FALSE, this will return NULL.
\item \strong{localDiscrepancy}: (List) A list of all of the minimized discrepancy functions. If keepAll is FALSE, this will return NULL.
\item \strong{minDiscrepancy}: (Scalar) A value, ranging from 1:numberStarts, indicating which of the local solutions has the smallest discrepancy function value. If keepAll is FALSE, this will return NULL.
}
}
\description{
This function conducts factor rotations from a user-specified number of random (orthogonal) starting configurations. Based on the resulting discrepancy function, the function will determine the number of local minima and, among these local solutions, will find the "global minimum" (i.e., the minimal discrepancy value from a finite number of solutions).
}
\details{
\itemize{
  \item \strong{Global Minimum}: This function uses several random starting configurations for factor rotations in an attempt to find the global minimum solution. However, this function is not guaranteed to find the global minimum. Furthermore, the global minimum solution need not be more psychologically interpretable than any of the local solutions (cf. Rozeboom, 1992). As is recommended, our function returns the purality of local solutions so users can make their own judgements.
}
}
\examples{
## Generate an orthgonal factor model
lambda <- matrix(c(.41, .00, .00,
                   .45, .00, .00,
                   .53, .00, .00,
                   .00, .66, .00,
                   .00, .38, .00,
                   .00, .66, .00,
                   .00, .00, .68,
                   .00, .00, .56,
                   .00, .00, .55),
                 nrow = 9, ncol = 3, byrow = TRUE)

## Generate factor correlation matrix
Phi <- matrix(.50, nrow = 3, ncol = 3)
diag(Phi) <- 1

## Model-implied correlation (covariance) matrix
R <- lambda \%*\% Phi \%*\% t(lambda)
diag(R) <- 1

## 100 promax rotations from least squares factor extraction
Out1 <- rotate(R             = R,
               numFactors    = 3,
               facMethod     = "fals",
               rotate        = "promaxQ",
               numberStarts  = 100,
               keepAll       = TRUE,
               rotateControl = list(power = 4,
                                    norm  = TRUE),
               faControl     = list(treatHeywood = TRUE),
               digits        = 2)$loadings

## 10 oblimin rotations from maximum likelihood factor extraction
Out2 <- rotate(R             = R,
               numFactors    = 3,
               facMethod     = "faml",
               rotate        = "oblimin",
               numberStarts  = 10,
               keepAll       = FALSE,
               rotateControl = list(gamma   = 0,
                                    epsilon = 1e-6),
               faControl     = list(nStart         = 10,
                                    maxCommunality = .99),
               digits        = 2)$loadings

}
\references{
Browne, M. W. (2001). An overview of analytic rotation in exploratory factor analysis. \emph{Multivariate Behavioral Research, 36}(1), 111-150.

Mansolf, M., & Reise, S. P. (2016). Exploratory bifactor analysis: The Schmid-Leiman orthogonalization and Jennrich-Bentler analytic rotations. \emph{Multivariate Behavioral Research, 51}(5), 698-717.

Rozeboom, W. W. (1992). The glory of suboptimal factor rotation: Why local minima in analytic optimization of simple structure are more blessing than curse. \emph{Multivariate Behavioral Research, 27}(4), 585-599.
}
\author{
\itemize{
  \item Casey Giordano (Giord023@umn.edu)
  \item Niels G. Waller (nwaller@umn.edu)
}
}
