% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fsIndeterminacy.R
\name{fsIndeterminacy}
\alias{fsIndeterminacy}
\title{Understanding Factor Score Indeterminacy with Finite Dimensional Vector Spaces}
\usage{
fsIndeterminacy(
  Lambda = NULL,
  Phi = NULL,
  N = NULL,
  X = NULL,
  SeedX = NULL,
  SeedBasis = NULL,
  SeedW = NULL,
  SeedT = 1,
  DoFCorrection = TRUE,
  Print = "short",
  Digits = 3,
  Example = FALSE
)
}
\arguments{
\item{Lambda}{(Matrix) A p x m matrix of factor loadings.}

\item{Phi}{(Matrix) An m x m factor correlation matrix.}

\item{N}{(Integer) The desired sample size.}

\item{X}{(Matrix) an optional N x p matrix of observed scores. Note that the observed scores
are expected to fit the factor model (as they will if they are generated 
from simFA and Population = TRUE is specified). Default (\code{X = NULL}).}

\item{SeedX}{(Integer) Starting seed for generating the matrix of observed scores, X.}

\item{SeedBasis}{(Integer) Starting seed for generating a basis for all scores.}

\item{SeedW}{(Integer) Starting seed for generating a weight matrix that is 
used to construct those parts of the factor scores that lie outside of span(X).}

\item{SeedT}{(Integer) Starting seed for generating a rotation matrix that 
creates a new set of factor scores from an existing set of scores such that 
the new set also perfectly fits the factor model.}

\item{DoFCorrection}{(Logical) Degrees of freedom correction.  If DoFCorrection = TRUE
then var(x) = 1/(N-1) * t(x) %*% x; else var(x) = 1/N * t(x) %*% x. 
Default (\code{DoFCorrection = TRUE}).}

\item{Print}{(Character) If \code{Print = "none"} no summary information
will be printed.  If \code{Print = "short"} then basic output for evaluating
the factor scores will be printed. If \code{Print = "long"} extended output
will be printed. Default (\code{Print = "short"}).}

\item{Digits}{(Integer) Sets the number of significant digits to print when
printing is requested.}

\item{Example}{(Logical) If Example = TRUE the program will
execute the orthogonal two factor model  described  in Waller (2021).}
}
\value{
\itemize{
 \item \strong{"Sigma"}: The p x p model implied covariance matrix.
 \item \strong{"X"}:  An N x p data matrix for the observed variables. 
 \item \strong{"Fhat"}: An N x (m + p) matrix of regression factor score estimates. 
 \item \strong{"Fi"}:  A possible set of common and unique factor scores.
 \item \strong{"Fj"}: The set of factor scores that are minimally correlated with Fi. 
 \item \strong{"Fk"}: Another set of common and unique factor scores. 
 Note that in a 1-factor model, Fk = Fi.
 \item \strong{"Fl"}: The set of factor scores that are minimally correlated with Fk. 
 Note that in a 1-factor model, Fj = Fl. 
 \item \strong{"Ei"}: Residual scores for Fi.
 \item \strong{"Ej"}: Residual scores for Fj.
 \item \strong{"Ek"}: Residual scores for Fk.
 \item \strong{"El"}: Residual scores for Fl. 
 \item \strong{"L"}: The factor loading super matrix.
 \item \strong{"C"}: The factor correlation super matrix.
 \item \strong{"V"}: A (non unique) basis for R^N.
 \item \strong{"W"}: Weight matrix for generating  Zi.
 \item \strong{"Tmat"}: The orthogonal transformation matrix used to construct Fk  from Fi .
 \item \strong{"B}:  The matrix that takes Ei to Ek (Ek = Ei B).
 \item \strong{"Bstar"} In an orthogonal factor model, Bstar takes Fi to Fk (Fk = Fi Bstar). 
              In an oblique model the program returns Bstar=NULL.
 \item \strong{"P"}: The matrix that imposes the proper covariance structure on Ei.
 \item \strong{"SeedX"}: Starting seed for X.   
 \item \strong{"SeedBasis"}: Starting seed for the basis. 
 \item \strong{"SeedW"}: Starting seed for weight matrix W. 
 \item \strong{"SeedT"}: Starting seed for rotation matrix T.
 \item \strong{"Guttman"}:  Guttman indeterminacy measures for the common and unique factors.
 \item \strong{"CovFhat"}: Covariance matrix of estimated factor scores.
 }
}
\description{
This function illustrates the algebra of factor score indeterminacy 
using concepts from finite dimensional vector spaces. Given any factor loading 
matrix, factor correlation matrix, and desired sample size, the program will 
compute a matrix of observed scores and multiple sets of factors
scores.  Each set of (m common and p unique) factors scores will fit the model 
perfectly.
}
\examples{
# ---- Example 1: ----
# To run the example in Waller (2021) enter:
out1 <- fsIndeterminacy(Example = TRUE)

# ---- Example 1: Extended Version: ----

N <- 10 # number of observations
# Generate Lambda: common factor loadings 
#          Phi: Common factor correlation matrix

Lambda <- matrix(c(.8,  0,
                   .7,  0,
                   .6,  0,
                    0, .5,
                    0, .4,
                    0, .3), 6, 2, byrow=TRUE)

out1  <- fsIndeterminacy(Lambda,
                         Phi = NULL,    # orthogonal model
                         SeedX = 1,     # Seed for X
                         SeedBasis = 2, # Seed for Basis
                         SeedW = 3,     # Seed for Weight matrix
                         SeedT = 5,     # Seed for Transformation matrix
                         N = 10,        # Number of subjects
                         Print = "long",
                         Digits = 3)

# Four sets of factor scores
  Fi <- out1$Fi
  Fj <- out1$Fj
  Fk <- out1$Fk
  Fl <- out1$Fl

# Estimated Factor scores
  Fhat <- out1$Fhat

# B wipes out Fhat (in an orthogonal model)
  B <- out1$B
  round( cbind(Fhat[1:5,1:2], (Fhat \%*\% B)[1:5,1:2]), 3) 

# B takes Ei -> Ek
  Ei <- out1$Ei
  Ek <- out1$Ek
  Ek - (Ei \%*\% B)

# The Transformation Approach
# Bstar takes Fi --> Fk
  Bstar <- out1$Bstar
  round( Fk - Fi \%*\% Bstar, 3)

# Bstar L' = L'
  L <- out1$L
  round( L \%*\% t(Bstar), 3)[,1:2]  


# ---- Example 3 ----
# We choose a different seed for T

out2  <- fsIndeterminacy(Lambda , 
                        Phi = NULL, 
                        X = NULL,
                        SeedX = 1,     # Seed for X 
                        SeedBasis = 2, #  Seed for Basis
                        SeedW = 3,     #  Seed for Weight matrix
                        SeedT = 4,     # Seed for Transformation matrix
                        N,             
                        Print = "long",
                        Digits = 3,
                        Example = FALSE)
 Fi <- out2$Fi
 Fj <- out2$Fj
 Fk <- out2$Fk
 Fl <- out2$Fl
 X  <- out2$X
 
# Notice that all sets of factor scores are model consistent 
 round( t( solve(t(Fi) \%*\% Fi) \%*\% t(Fi) \%*\% X) ,3)
 round( t( solve(t(Fj) \%*\% Fj) \%*\% t(Fj) \%*\% X) ,3)
 round( t( solve(t(Fk) \%*\% Fk) \%*\% t(Fk) \%*\% X) ,3)
 round( t( solve(t(Fl) \%*\% Fl) \%*\% t(Fl) \%*\% X) ,3)
 
# Guttman's Indeterminacy Index
round( (1/N * t(Fi) \%*\% Fj)[1:2,1:2], 3)

}
\references{
Guttman, L.  (1955).  The determinacy of factor score matrices 
 with applications for five other problems of common factor theory.  
 \emph{British Journal of Statistical Psychology, 8}, 65-82.

Ledermann, W.  (1938).  The orthogonal transformation of a factorial matrix 
 into itself.  \emph{Psychometrika, 3}, 181-187.

Schönemann, P. H. (1971). The minimum average correlation 
between equivalent sets of uncorrelated factors. \emph{Psychometrika, 36}, 
21-30.

Steiger, J. H. and Schonemann, P. H.  (1978).  In Shye, S. (Ed.), 
 \emph{A history of factor indeterminacy} (pp. 136--178). San  Francisco: Jossey-Bass.

Waller, N. G. (2021) Understanding factor indeterminacy through the lens of finite 
dimensional vector spaces. Manuscript under review.
}
\seealso{
Other Factor Analysis Routines: 
\code{\link{BiFAD}()},
\code{\link{Box26}},
\code{\link{GenerateBoxData}()},
\code{\link{Ledermann}()},
\code{\link{SLi}()},
\code{\link{SchmidLeiman}()},
\code{\link{faAlign}()},
\code{\link{faEKC}()},
\code{\link{faIB}()},
\code{\link{faLocalMin}()},
\code{\link{faMB}()},
\code{\link{faMain}()},
\code{\link{faScores}()},
\code{\link{faSort}()},
\code{\link{faStandardize}()},
\code{\link{faX}()},
\code{\link{fals}()},
\code{\link{fapa}()},
\code{\link{fareg}()},
\code{\link{orderFactors}()},
\code{\link{print.faMB}()},
\code{\link{print.faMain}()},
\code{\link{promaxQ}()},
\code{\link{summary.faMB}()},
\code{\link{summary.faMain}()}
}
\author{
Niels G. Waller (nwaller@umn.edu)
}
\concept{Factor Analysis Routines}
