% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CompleteRdev.R
\name{CompleteRdev}
\alias{CompleteRdev}
\title{Complete a Partially Specified Correlation Matrix by the Method of Differential Evolution}
\usage{
CompleteRdev(
  Rna,
  NMatrices = 1,
  MaxDet = FALSE,
  MaxIter = 200,
  delta = 1e-08,
  PRINT = FALSE,
  Seed = NULL
)
}
\arguments{
\item{Rna}{(matrix) An n x n incomplete correlation matrix.  Missing entries must
be specified by \code{NA} values.}

\item{NMatrices}{(integer) \code{CompleteRDEV} will complete \code{NMatrices}
correlation matrices.}

\item{MaxDet}{(logical) If MaxDet = TRUE then the correlation matrix will
be completed with entries that maximize the determinant of R.}

\item{MaxIter}{(integer) The maximum number of iterations
(i.e., generations) allowed. Default \code{MaxIter = 200}.}

\item{delta}{(numeric > 0) A number that controls the convergence
accuracy of the differential evolution algorithm. Default \code{delta = 1E-8}.}

\item{PRINT}{(logical) When PRINT = TRUE the algorithm convergence status is printed.
Default  \code{PRINT = FALSE}.}

\item{Seed}{(integer) Initial random number seed. Default (\code{Seed = NULL}).}
}
\value{
\code{CompleteRdev} returns the following objects:
\itemize{
   \item  \strong{R} (matrix) A PSD completed correlation matrix.
   \item  \strong{converged}: (logical) a logical that indicates the convergence status of the optimizaton.
   \item \strong{iter} (integer) The number of cycles needed to reach converged solution.
}
}
\description{
This function completes a partially specified
correlation matrix by the method of differential evolution.
}
\examples{
## Example 1: Generate random 4 x 4 Correlation matrices.
  Rmiss <- matrix(NA, nrow = 4, ncol = 4)
  diag(Rmiss) <- 1

  out <- CompleteRdev(Rna = Rmiss,
                      NMatrices = 4,
                      PRINT = TRUE,
                      Seed = 1)

  print( round( out$R[[1]] , 3) )

\dontrun{
# Example 2: Complete a partially specified R matrix.
# Example from Georgescu, D. I., Higham, N. J., and
#              Peters, G. W.  (2018).

Rmiss <- matrix(
     c( 1,  .25, .6,  .55, .65,  0,  .4,   .6,  .2,  .3,
       .25, 1,    0,   0,   0,   0,  NA,   NA,  NA,  NA,
       .6,  0,   1,   .75, .75,  0,  NA,   NA,  NA,  NA,
       .55, 0,   .75, 1,   .5,   0,  NA,   NA,  NA,  NA,
       .65, 0,   .75,  .5, 1,    0,  NA,   NA,  NA,  NA,
        0,  0,    0,   0,   0,  1,   NA,   NA,  NA,  NA,
        .4, NA,   NA,  NA,  NA,  NA, 1,   .25, .25,  .5,
        .6, NA,   NA,  NA,  NA,  NA, .25,  1,  .25,  0,
        .2, NA,   NA,  NA,  NA,  NA, .25,  .25, 1,   0,
        .3, NA,   NA,  NA,  NA,  NA, .5,    0,   0,  1), 10,10)

# Complete Rmiss with values that maximize
# the matrix determinant (this is the MLE solution)
 set.seed(123)
 out <- CompleteRdev(Rna = Rmiss,
                     MaxDet = TRUE,
                     MaxIter = 1000,
                     delta = 1E-8,
                     PRINT = FALSE)

cat("\nConverged = ", out$converged,"\n")
print( round(out$R, 3))
print( det(out$R))
print( eigen(out$R)$values, digits = 5)
}

}
\references{
Ardia, D., Boudt, K., Carl, P., Mullen, K.M., Peterson, B.G. (2011) Differential
Evolution with DEoptim. An Application to Non-Convex Portfolio Optimization.
URL The R Journal, 3(1), 27-34.
URL https://journal.r-project.org/archive/2011-1/RJournal_2011-1_Ardia~et~al.pdf.

Georgescu, D. I., Higham, N. J., and Peters, G. W.  (2018).  Explicit
solutions to correlation matrix completion problems, with
an application to risk management and insurance.  Royal Society Open
Science, 5(3), 172348.

Mauro, R.  (1990).  Understanding L.O.V.E. (left out variables
error): a method for estimating the effects of omitted variables.
Psychological Bulletin, 108(2), 314-329.

Mishra, S. K.  (2007).  Completing correlation matrices
of arbitrary order by differential evolution method of global optimization:
a Fortran program.  Available at SSRN 968373.

Mullen, K.M, Ardia, D., Gil, D., Windover, D., Cline, J. (2011). DEoptim: An
R Package for Global Optimization by Differential Evolution. Journal of Statistical Software, 40(6), 1-26. URL http://www.jstatsoft.org/v40/i06/.

Price, K.V., Storn, R.M., Lampinen J.A. (2005) Differential Evolution - A Practical Approach to Global Optimization. Berlin Heidelberg:
Springer-Verlag. ISBN 3540209506.

Zhang, J. and Sanderson, A. (2009) Adaptive Differential
Evolution Springer-Verlag. ISBN 978-3-642-01526-7
}
\author{
Niels G. Waller
}
