checkValidWeights <- function(weights) {
	if(!is.numeric(weights)) {
		stop("Weights have to be numeric!")
	}
	if(any(is.na(weights) | is.infinite(weights))) {
		warning("Some of the weights are not real numbers. NA, NaN, Inf and -Inf are not supported.")
	}
	if(any(0 > weights | weights > 1 + .Machine$double.eps ^ 0.25)) {
		warning("Invalid weights: weights must be between 0 and 1")
	}
	if(sum(weights) > 1 + .Machine$double.eps ^ 0.25) {
		warning("Invalid weights: the sum of all weights must be less than 1")
	}
}

# Converts a string like "5+3*e+5*e^2" to the tupel representation c(5,3,5) 
parseEpsPolynom <- function(s) {
	env <- new.env(parent = baseenv())
	assign("epsilon", polynom(), envir=env)
	p <- try(eval(parse(text=s), envir=env), silent = TRUE)
	if (class(p)=="try-error") {
		stop("String does not represent a polynom in epsilon.")
	}
	if(is.numeric(p)) {
		return(p)
	} else {
    	return(coef(p))
	}
}

getDebugInfo <- function() {
	graphs <- ls(pattern="\\.InitialGraph*", all.names=TRUE, envir=globalenv())
	if (exists(".tmpGraph")) {
		graphs <- c(graphs, ".tmpGraph")
	}
	graphInfo <- c()
	for (graph in graphs) {
		.DebugGraph <- get(graph, envir=globalenv())
		graphTXT <- paste(capture.output(print(.DebugGraph)), collapse="\n")
		matrixTXT <- paste("m <-",paste(capture.output(dput(graph2matrix(.DebugGraph))), collapse="\n"),"\n")
		weightsTXT <- paste("w <-",paste(capture.output(dput(getWeights(.DebugGraph))), collapse="\n"),"\n")
		createTXT <- paste("graph <- matrix2graph(m)", "setWeights(graph, w)", sep="\n")
		graphInfo <- c(graphInfo, paste(graphTXT, matrixTXT, weightsTXT, createTXT, sep="\n"))
	}
	if (length(graphInfo)!=0) {
		return(paste(graphInfo, collapse="\n\n"))
	}
	return("Graph not available.")
}

bdiagNA <- function(...) {	
	if (nargs() == 0) 
		return(matrix(nrow=0, ncol=0))
	if (nargs() == 1 && !is.list(...)) 
		return(as.matrix(...))
	asList <- if (nargs() == 1 && is.list(...)) list(...)[[1]] else list(...)
	if (length(asList) == 1) 
		return(as.matrix(asList[[1]]))
	n <- 0
	for (m in asList) {
		if (!is.matrix(m)) {
			stop("Only matrices are allowed as arguments.")
		}
		if (dim(m)[1]!=dim(m)[2]) {
			stop("Only quadratic matrices are allowed.")
		}
		n <- n + dim(m)[1]	
	}
	M <- matrix(NA, nrow=n, ncol=n)
	k <- 0
	for (m in asList) {
		for (i in 1:dim(m)[1]) {
			for (j in 1:dim(m)[1]) {
				M[i+k,j+k] <- m[i,j]
			}
		}
		k <- k + dim(m)[1]	
	}	
	return(M)
}

requireLibrary <- function(package) {
	if(!require(package, character.only=TRUE)) {
		answer <- readline(paste("Package ",package," is required - should we install it (y/n)? ", sep=""))
		if (substr(answer, 1, 1) %in% c("y","Y")) {
			if (package %in% c("graph", "Rgraphviz")) {	
				source("http://www.bioconductor.org/biocLite.R")
				biocLite <- get("biocLite", envir=globalenv())
				biocLite(package)
			} else {
				install.packages(package)
				require(package, character.only=TRUE)
			}
		} else {
			stop(paste("Required package ",package," should not be installed.", sep=""))
		}
	}
}

triangle <- function(min, peak, max) {
	
}

# Depending of the number of hypotheses it is calculated 
# which standard designs could be a possibility and how many 
# groups they would have.
getAvailableStandardDesigns <- function(n) {
	designs <- c()
	numberOfGroups <- c()
	possibleDesigns = c("Dunnett", "Tukey", "Sequen", "AVE", "Changepoint", "Williams", "Marcus", "McDermott", "UmbrellaWilliams", "GrandMean")
	for (design in possibleDesigns) {
		for (i in 2:(2*n)) {
			m <- try(contrMat(n=rep(10, i), type=design), silent = TRUE)
			if (!("try-error" %in% class(m)) && dim(m)[1]==n) {
				designs <- c(designs, design)
				numberOfGroups <- c(numberOfGroups, i)
			}
		}
	}	
	return(list(designs, numberOfGroups))
}

getCorrMat <- function(n, type) {
	m <- contrMat(n, type)	
	m <- m%*%diag(1/sqrt(n))%*%t(m)
	s <- diag(1/sqrt(diag(m)))
	r <- s%*%m%*%s
	return(as.numeric(r))
}

# Given a character string x only the letters are returned (upper and lower case)
removeSymbols <- function(x, numbers=TRUE) {
	pattern <-ifelse(numbers, "[a-z]|[0-9]", "[a-z]")
	m <- gregexpr(pattern, x, ignore.case=TRUE)
	return(unlist(lapply(regmatches(x, m), paste, collapse="")))
}

nextAvailableName <- function(x) {
	if (exists(x)) {
		i <- 1
		while (exists(paste(x, i, sep=""))) {
			i <- i + 1
		}
		x <- paste(x, i, sep="")
	}
	return(x)
}

# The following code of regmatches under GPL is taken directly from the R base package. 
# Copyright (C) 2012 The R Foundation for Statistical Computing
# It was introduced in R 2.14.0 - but for backward-compatibility we also include it in gMCP.

regmatches <- function (x, m, invert = FALSE) {
	if (length(x) != length(m)) 
		stop(gettextf("%s and %s must have the same length", 
						sQuote("x"), sQuote("m")), domain = NA)
	ili <- is.list(m)
	useBytes <- if (ili) 
				any(unlist(lapply(m, attr, "useBytes")))
			else any(attr(m, "useBytes"))
	if (useBytes) {
		asc <- iconv(x, "latin1", "ASCII")
		ind <- is.na(asc) | (asc != x)
		if (any(ind)) 
			Encoding(x[ind]) <- "bytes"
	}
	if (!ili && !invert) {
		so <- m[ind <- (!is.na(m) & (m > -1L))]
		eo <- so + attr(m, "match.length")[ind] - 1L
		return(substring(x[ind], so, eo))
	}
	y <- if (invert) {
				Map(function(u, so, ml) {
							if ((n <- length(so)) == 1L) {
								if (is.na(so)) 
									return(character())
								else if (so == -1L) 
									return(u)
							}
							beg <- if (n > 1L) {
										eo <- so + ml - 1L
										if (any(eo[-n] >= so[-1L])) 
											stop(gettextf("need non-overlapping matches for %s", 
															sQuote("invert = TRUE")), domain = NA)
										c(1L, eo + 1L)
									}
									else {
										c(1L, so + ml)
									}
							end <- c(so - 1L, nchar(u))
							substring(u, beg, end)
						}, x, m, if (ili) 
									lapply(m, attr, "match.length")
								else attr(m, "match.length"), USE.NAMES = FALSE)
			}
			else {
				Map(function(u, so, ml) {
							if (length(so) == 1L) {
								if (is.na(so) || (so == -1L)) 
									return(character())
							}
							substring(u, so, so + ml - 1L)
						}, x, m, lapply(m, attr, "match.length"), USE.NAMES = FALSE)
			}
	names(y) <- names(x)
	y
}