% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{createColorRamp}
\alias{createColorRamp}
\title{Create a color ramp}
\usage{
createColorRamp(
  start_index,
  start_color,
  end_index,
  end_color,
  palette_interp = "RGB"
)
}
\arguments{
\item{start_index}{Integer start index (raster value).}

\item{start_color}{Integer vector of length three or four.
A color entry value to start the ramp (e.g., RGB values).}

\item{end_index}{Integer end index (raster value).}

\item{end_color}{Integer vector of length three or four.
A color entry value to end the ramp (e.g., RGB values).}

\item{palette_interp}{One of "Gray", "RGB" (the default), "CMYK" or "HLS"
describing interpretation of \code{start_color} and \code{end_color} values
(see \href{https://gdal.org/en/stable/user/raster_data_model.html#color-table}{GDAL
Color Table}).}
}
\value{
Integer matrix with five columns containing the color ramp from
\code{start_index} to \code{end_index}, with raster index values in column 1 and
color entries in columns 2:5).
}
\description{
\code{createColorRamp()} is a wrapper for \code{GDALCreateColorRamp()} in the GDAL
API. It automatically creates a color ramp from one color entry to another.
Output is an integer matrix in color table format for use with
\code{\link[=GDALRaster]{GDALRaster$setColorTable()}}.
}
\note{
\code{createColorRamp()} could be called several times, using \code{rbind()} to
combine multiple ramps into the same color table. Possible duplicate rows
in the resulting table are not a problem when used in
\code{GDALRaster$setColorTable()} (i.e., when \code{end_color} of one ramp is the
same as \code{start_color} of the next ramp).
}
\examples{
# create a color ramp for tree canopy cover percent
# band 5 of an LCP file contains canopy cover
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
ds <- new(GDALRaster, lcp_file)
ds$getDescription(band=5)
ds$getMetadata(band=5, domain="")
ds$close()

# create a GTiff file with Byte data type for the canopy cover band
# recode nodata -9999 to 255
tcc_file <- calc(expr = "ifelse(CANCOV == -9999, 255, CANCOV)",
                 rasterfiles = lcp_file,
                 bands = 5,
                 var.names = "CANCOV",
                 fmt = "GTiff",
                 dtName = "Byte",
                 nodata_value = 255,
                 setRasterNodataValue = TRUE)

ds_tcc <- new(GDALRaster, tcc_file, read_only=FALSE)

# create a color ramp from 0 to 100 and set as the color table
colors <- createColorRamp(start_index = 0,
                          start_color = c(211, 211, 211),
                          end_index = 100,
                          end_color = c(0, 100, 0))

print(colors)
ds_tcc$setColorTable(band=1, col_tbl=colors, palette_interp="RGB")
ds_tcc$setRasterColorInterp(band=1, col_interp="Palette")

# close and re-open the dataset in read_only mode
ds_tcc$open(read_only=TRUE)

plot_raster(ds_tcc, interpolate=FALSE, legend=TRUE,
            main="Storm Lake Tree Canopy Cover (\%)")
ds_tcc$close()

deleteDataset(tcc_file)
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster$getColorTable()}},
\code{\link[=GDALRaster]{GDALRaster$getPaletteInterp()}}
}
