% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_spike.R
\name{geom_spike}
\alias{geom_spike}
\title{Spike plot (ggplot2 geom)}
\usage{
geom_spike(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  ...,
  arrow = NULL,
  orientation = NA,
  normalize = "all",
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{stat}{The statistical transformation to use on the data for this
layer, either as a \code{ggproto} \code{Geom} subclass or as a string naming the
stat stripped of the \code{stat_} prefix (e.g. \code{"count"} rather than
\code{"stat_count"})}

\item{position}{Position adjustment, either as a string, or the result of a call to a position adjustment function.
Setting this equal to \code{"dodge"} (\code{\link[=position_dodge]{position_dodge()}}) or \code{"dodgejust"} (\code{\link[=position_dodgejust]{position_dodgejust()}}) can be useful if
you have overlapping geometries.}

\item{...}{Other arguments passed to \code{\link[=layer]{layer()}}. These are often aesthetics, used to set an aesthetic
to a fixed value, like \code{colour = "red"} or \code{linewidth = 3} (see \strong{Aesthetics}, below). They may also be
parameters to the paired geom/stat.}

\item{arrow}{\code{\link[grid:arrow]{grid::arrow()}} giving the arrow heads to use on the spike, or \code{NULL} for no arrows.}

\item{orientation}{Whether this geom is drawn horizontally or vertically. One of:
\itemize{
\item \code{NA} (default): automatically detect the orientation based on how the aesthetics
are assigned. Automatic detection works most of the time.
\item \code{"horizontal"} (or \code{"y"}): draw horizontally, using the \code{y} aesthetic to identify different
groups. For each group, uses the \code{x}, \code{xmin}, \code{xmax}, and \code{thickness} aesthetics to
draw points, intervals, and slabs.
\item \code{"vertical"} (or \code{"x"}): draw vertically, using the \code{x} aesthetic to identify different
groups. For each group, uses the \code{y}, \code{ymin}, \code{ymax}, and \code{thickness} aesthetics to
draw points, intervals, and slabs.
}
For compatibility with the base ggplot naming scheme for \code{orientation}, \code{"x"} can be used as an alias
for \code{"vertical"} and \code{"y"} as an alias for \code{"horizontal"} (\pkg{ggdist} had an \code{orientation} parameter
before base ggplot did, hence the discrepancy).}

\item{normalize}{How to normalize heights of functions input to the \code{thickness} aesthetic. One of:
\itemize{
\item \code{"all"}: normalize so that the maximum height across all data is \code{1}.
\item \code{"panels"}: normalize within panels so that the maximum height in each panel is \code{1}.
\item \code{"xy"}: normalize within the x/y axis opposite the \code{orientation} of this geom so
that the maximum height at each value of the opposite axis is \code{1}.
\item \code{"groups"}: normalize within values of the opposite axis and within each
group so that the maximum height in each group is \code{1}.
\item \code{"none"}: values are taken as is with no normalization (this should probably
only be used with functions whose values are in [0,1], such as CDFs).
}}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with a warning. If \code{TRUE}, missing
values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}
}
\value{
A \link[ggplot2:ggplot2-ggproto]{ggplot2::Geom} representing a spike geometry which can
be added to a \code{\link[=ggplot]{ggplot()}} object.
rd_slabinterval_aesthetics(geom_name),
}
\description{
Geometry for drawing "spikes" (optionally with points on them) on top of
\code{\link[=geom_slabinterval]{geom_slabinterval()}} geometries: this geometry understands the scaling and
positioning of the \code{thickness} aesthetic from \code{\link[=geom_slabinterval]{geom_slabinterval()}}, which
allows you to position spikes and points along a slab.
}
\details{
This geometry consists of a "spike" (vertical/horizontal line segment) and a
"point" (at the end of the line segment). It uses the \code{thickness} aesthetic
to determine where the endpoint of the line is, which allows it to be used
with \code{\link[=geom_slabinterval]{geom_slabinterval()}} geometries for labeling specific values of the
thickness function.
}
\section{Aesthetics}{

The spike \code{geom} has a wide variety of aesthetics that control
the appearance of its two sub-geometries: the \strong{spike} and the \strong{point}.

\strong{Positional aesthetics}

\itemize{
\item{\code{x}: x position of the geometry}
\item{\code{y}: y position of the geometry}
}

\strong{Spike-specific (aka Slab-specific) aesthetics}

\itemize{
\item{\code{thickness}: The thickness of the slab at each \code{x} value (if \code{orientation = "horizontal"}) or
\code{y} value (if \code{orientation = "vertical"}) of the slab.}
\item{\code{side}: Which side to place the slab on. \code{"topright"}, \code{"top"}, and \code{"right"} are synonyms
which cause the slab to be drawn on the top or the right depending on if \code{orientation} is \code{"horizontal"}
or \code{"vertical"}. \code{"bottomleft"}, \code{"bottom"}, and \code{"left"} are synonyms which cause the slab
to be drawn on the bottom or the left depending on if \code{orientation} is \code{"horizontal"} or
\code{"vertical"}. \code{"topleft"} causes the slab to be drawn on the top or the left, and \code{"bottomright"}
causes the slab to be drawn on the bottom or the right. \code{"both"} draws the slab mirrored on both
sides (as in a violin plot).}
\item{\code{scale}: What proportion of the region allocated to this geom to use to draw the slab. If \code{scale = 1},
slabs that use the maximum range will just touch each other. Default is \code{0.9} to leave some space.}
}

\strong{Color aesthetics}

\itemize{
\item{\code{colour}: (or \code{color}) The color of the \strong{spike} and \strong{point} sub-geometries.}
\item{\code{fill}: The fill color of the \strong{point} sub-geometry.}
\item{\code{alpha}: The opacity of the \strong{spike} and \strong{point} sub-geometries.}
\item{\code{colour_ramp}: (or \code{color_ramp}) A secondary scale that modifies the \code{color}
scale to "ramp" to another color. See \code{\link[=scale_colour_ramp]{scale_colour_ramp()}} for examples.}
\item{\code{fill_ramp}: A secondary scale that modifies the \code{fill}
scale to "ramp" to another color. See \code{\link[=scale_fill_ramp]{scale_fill_ramp()}} for examples.}
}

\strong{Line aesthetics}

\itemize{
\item{\code{linewidth}: Width of the line used to draw the \strong{spike} sub-geometry.}
\item{\code{size}: Size of the \strong{point} sub-geometry.}
\item{\code{stroke}: Width of the outline around the \strong{point} sub-geometry.}
\item{\code{linetype}: Type of line (e.g., \code{"solid"}, \code{"dashed"}, etc) used to draw the \strong{spike}.}
}

\strong{Other aesthetics} (these work as in standard \code{geom}s)

\itemize{
\item{\code{width}}
\item{\code{height}}
\item{\code{group}}
}

See examples of some of these aesthetics in action in \code{vignette("slabinterval")}.
Learn more about the sub-geom override aesthetics (like \code{interval_color}) in the
\link[ggdist]{scales} documentation. Learn more about basic ggplot aesthetics in
\code{vignette("ggplot2-specs")}.
}

\examples{
library(ggplot2)
library(distributional)
library(dplyr)

# geom_spike is easiest to use with distributional or
# posterior::rvar objects
df = tibble(
  d = dist_normal(1:2, 1:2), g = c("a", "b")
)

# annotate the density at the mean of a distribution
df \%>\% mutate(
  mean = mean(d),
  density(d, list(density_at_mean = mean))
) \%>\%
  ggplot(aes(y = g)) +
  stat_slab(aes(xdist = d)) +
  geom_spike(aes(x = mean, thickness = density_at_mean)) +
  # need shared thickness scale so that stat_slab and geom_spike line up
  scale_thickness_shared()

# annotate the endpoints of intervals of a distribution
# here we'll use an arrow instead of a point by setting size = 0
arrow_spec = arrow(angle = 45, type = "closed", length = unit(4, "pt"))
df \%>\% mutate(
  median_qi(d, .width = 0.9),
  density(d, list(density_lower = .lower, density_upper = .upper))
) \%>\%
  ggplot(aes(y = g)) +
  stat_halfeye(aes(xdist = d), .width = 0.9, color = "gray35") +
  geom_spike(
    aes(x = .lower, thickness = density_lower),
    size = 0, arrow = arrow_spec, color = "blue", linewidth = 0.75
  ) +
  geom_spike(
    aes(x = .upper, thickness = density_upper),
    size = 0, arrow = arrow_spec, color = "red", linewidth = 0.75
  ) +
  scale_thickness_shared()

}
\seealso{
See \code{\link[=stat_spike]{stat_spike()}} for the stat version, intended for
use on sample data or analytical distributions.

Other slabinterval geoms: 
\code{\link{geom_interval}()},
\code{\link{geom_pointinterval}()},
\code{\link{geom_slab}()}
}
\concept{slabinterval geoms}
