% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glinvci.R, R/graph.R
\name{glinv}
\alias{glinv}
\alias{print.glinv}
\alias{lik.glinv}
\alias{grad.glinv}
\alias{hess.glinv}
\alias{plot.glinv}
\title{Construct an GLInv model with respect to user-specified parametrisation}
\usage{
glinv(
  tree,
  x0,
  X,
  parfns = NULL,
  pardims = NULL,
  regimes = NULL,
  parjacs = NULL,
  parhess = NULL,
  repar = NULL
)

\method{print}{glinv}(x, ...)

\method{lik}{glinv}(mod, ...)

\method{grad}{glinv}(
  mod,
  num_threads = 2L,
  numDerivArgs = list(method = "Richardson", method.args = list(d = 0.5, r = 3)),
  ...
)

\method{hess}{glinv}(
  mod,
  num_threads = 2L,
  numDerivArgs = list(method = "Richardson", method.args = list(d = 0.5, r = 3)),
  store_gaussian_hessian = FALSE,
  ...
)

\method{plot}{glinv}(x, internal_nodes = TRUE, ...)
}
\arguments{
\item{tree}{A tree of class \code{\link[ape]{phylo}}.}

\item{x0}{A vector representing the root's trait vector. Must not contain \code{NA} and \code{NaN}.}

\item{X}{Optional. A matrix of trait values, in which \code{X[p,n]} stores the p-th dimension
of the multivariate trait of the n-th tip of the phylogeny. \code{NA} and \code{NaN}
has special meanings (See Details).}

\item{parfns}{A list of functions that maps from the user-parametrisation to the underlying Gaussian parameters.
Each of them returns a vector of concatenated \eqn{(\Phi, w, V')}, where \eqn{V'} is the lower triangular
part of \eqn{V}, and accepts four arguments: a vector of parameters whose length is specified
by the \code{pardims} argument to the \code{glinv_gauss} function, the branch length leading to the currently processing node, 
a vector of factors with three levels indicating which dimensions are missing or lost in the mother of
the current node, and a vector of factors with the same three levels indicating missingness of the current
node.}

\item{pardims}{A vector of integers, which has the same amount elements as the length of parfns.
\code{pardims[i]} indicates the length of the parameter vector that \code{parfns[i]} accepts.}

\item{regimes}{A list of length-two integer vectors. Each of these length-two vectors specifies an evolutionary regime
and consists of a named element \code{start}, which specifies the node ID at which an evolutionary regime 
starts, and another named element \code{fn}, which is an index of \code{parfns}, indicating which parametrisation
function this evolutionary regime should use.}

\item{parjacs}{A list of functions, which has the same amount elements as that of \code{parfns}.
\code{parjacs[i]} accepts the same arguments as \code{parfns[i]} and returns the Jacobian of \code{parfns[i]}.}

\item{parhess}{A list of functions, which has the same amount elements as that of \code{parfn[i]}.
\code{parhess[i]} accepts the same arguments as \code{parfns[i]} and returns a list of three 3D arrays,
named \code{Phi}, \code{w}, \code{V} respectively inside the list. \code{((parhess[[i]])(...))$Phi[m,i,j]} contains
the cross second-order partial derivative of \eqn{\Phi_m} (here we treat the matrix \eqn{\Phi} as a
column-major-flattened vector) with respect to the \eqn{i}-th and\eqn{j}-th user parameters;
while \code{((parhess[[i]])(...))$w[m,i,j]} and \code{((parhess[[i]])(...))$V[m,i,j]}
analogously contains second-order derivative of \eqn{w_m} and \eqn{V'_m}.}

\item{repar}{Optional. One or a list of object returned by \code{get_restricted_ou}. This is a convenient
short-cut alternative to supplying \code{pardims}, \code{parfns}, \code{parjacs}, and
\code{parhess} one-by-one.}

\item{x}{An object of class \code{glinv}.}

\item{...}{Not used.}

\item{mod}{An object of class \code{glinv}.}

\item{num_threads}{Number of threads to use.}

\item{numDerivArgs}{Arguments to pass to \code{numDeriv::\link[numDeriv]{jacobian}}. Only used the user did not specify the
\code{parjacs} arguments when creating \code{mod}.}

\item{store_gaussian_hessian}{If \code{TRUE} and \code{method} is not \code{mc}, the returned list will contain
a (usually huge) Hessian matrix \code{gaussian_hessian} with respect to the Gaussian
parameters \eqn{\Phi, w, V'}. This option significantly increases the amount of memory
the function uses, in order to store the matrix.}

\item{internal_nodes}{Boolean, whether to plot the internal nodes's numbers}
}
\value{
The \code{glinv} function returns a model object of S3 class \code{glinv}. Elements are:
               
                 \item{rawmod}{An object of class \code{glinv_gauss}.}
                 \item{regimes}{Identical to the \code{regimes} argument.}
                 \item{regtags}{An integer vector of the same length as the number of nodes. The \eqn{i}-th element is
                                the regime ID (corresponding to the index in the \code{regimes} argument to the \code{glinv_gauss} function) of
                                node \eqn{i}. \code{NA} at the root.}
                 \item{misstags}{A factor matrix with three ordered levels, \code{LOST}, \code{OK}, and \code{MISSING}. Each column
                                 corresponds to a node and row to a trait dimension.}
                 \item{nparams}{The sum of the \code{pardims} argument, an integer.}
                 \item{pardims}{Identical to the \code{pardims} arguemnt.}
                 \item{parfntags}{An integer vector of the same length as the number of nodes. The \eqn{i}-th element is
                                  the index of \code{parfns} that corresponds to node \eqn{i}. \code{NA} at the root.}
                 \item{parfns}{Identical to the \code{parfns} argument.}
                 \item{parjacs}{Identical to the \code{parjacs} argument.}
                 \item{parhess}{Identical to the \code{parhess} argument.}
                 \item{parsegments}{A \eqn{K}-by-2 matrix of integer indicies, where \eqn{K} is the length of \code{parfns}.
                                    If \code{v} is a vector that \code{\link{lik.glinv}} accepts, then
                                    \code{v[parsegments[k,1]:parsegments[k,2]]} is the parameter vector should \code{parfns[[k]]}
                                    accept.}
                 \item{gausssegments}{A \eqn{N}-by-2 matrix of integer indicies, where \eqn{N} is the number of nodes.
                                      If \code{w} is a vector that \code{\link{lik.glinv_gauss}} accepts, then
                                      \code{w[gausssegments[i,1]:gausssegments[i,2]]} is the concatenated \eqn{(\Phi, w, V')}
                                      corresponding to node \eqn{i}.}
                 \item{gaussparams_fn}{A function that accepts a parameter vector of length \code{nparams} and returns a
                                       parameter vector of length \code{rawmod$nparams}. When called, this function
                                       traverses the tree, calls the functions in parfns on each node, and assemble 
                                       the results into a format that \code{\link{lik.glinv_gauss}} accepts.}
                 \item{gaussparams_jac}{A function that accepts a parameter vector of length \code{nparams} and returns a
                                       \eqn{p}-by-\eqn{q} Jacobian matrix, where \eqn{p} is \code{rawmod$nparams} and \eqn{q}
                                       is \code{nparams} in this object. When called, this function traverses
                                       the tree, calls the functions in \code{parjacs} on each node, and row-concatenates the
                                       result in an order consistent with what \code{\link{lik.glinv_gauss}} accepts.}
                 \item{X}{The original data (trait) matrix in a "normalized" format.}
}
\description{
The \code{glinv} function construct an object of class \code{glinv}, which represents a GLInv model with respect
to a user-specified parametrisation.

The \code{lik.glinv} function returns a function which accepts a parameter vector, which is of length \code{mod$nparams},
and returns the log-likelihood.

The \code{grad.glinv} function returns a function which accepts a parameter vector, which is of length \code{mod$nparams},
and returns the gradient of log-likelihood with respect to this parametrisation.

The \code{hess.glinv} function returns a function which accepts a parameter vector, which is of length \code{mod$nparams},
and returns the Hessian matrix of log-likelihood with respect to this parametrisation.
}
\details{
Models for \code{glinv} assumes one or more evolutionary regimes exists in the phylogeny. The \code{regimes} parameters defines
how many regimes there are, where do the regimes start, and what parameterisation function it has. If \code{regimes} were
NULL then a single regime starting from the root node is assumed. Multiple regimes could share the same parametrisation
function (and thus the same parameters) by specifying the same index; therefore the number of regimes may differs from 
the number of parametrisation functions. One and only one regime must start from the root of the phylogeny.

If \code{X} contains \code{NA} in the \eqn{p}-th dimension of the \eqn{i}-th tip (whose node ID is also \eqn{i}) then \eqn{X_pi} is
tagged \code{MISSING}. No other tags of any other nodes are changed. The \eqn{p}-th dimension of any node \eqn{j}, regardless of
whether or not it is an internal node or a tips, is tagged \code{LOST} if and only if the \eqn{p}-th dimension of all tips inside
the clade started at \eqn{j} are \code{NaN}. Any entry that is neither \code{LOST} nor \code{MISSING} are tagged \code{OK}. These
tags are then passed into the user-defined functions \code{parfns} etc. as arguments; therefore the user is free to specify how
these tags are handled. \code{x0} cannot contain missing values, and the vectors of missingness tags passed to \code{parfns}, for
any nodes, are always of the same length as \code{x0}.

Before this package calls the functions in \code{parhess}, it adds, into the function's environment, a variable named \code{INFO__} 
which contains some extra information.

Passing a single function to \code{parfns} is equivalent to passing a singleton list; and the same is true for \code{parjacs},
\code{parhess}, and \code{pardims}.
}
\examples{
\donttest{
### --- STEP 1: Making an example tree and trait data
ntips = 200
k     = 2                 # No. of trait dimensions
tr    = ape::rtree(ntips) 
X     = matrix(rnorm(k*ntips), k, ntips)
x0    = rnorm(k)

### --- STEP 2: Making a model object. We use OU as an example.
###             Assume H is a positively definite diagonal matrix.
mod = glinv(tr, x0, X,
            parfns = list(ou_logdiagH(ou_haltlost(oupar))),
            pardims = list(nparams_ou_diagH(k)),
            parjacs = list(dou_logdiagH(dou_haltlost(oujac))),
            parhess = list(hou_logdiagH(hou_haltlost(ouhess))))

### --- STEP 3: Try getting the likelihood, gradient etc.
H     = matrix(c(1,0,0,-1), k)
theta = c(0,0)
sig   = matrix(c(0.5,0,0,0.5), k)
sig_x = t(chol(sig))
# glinvci ALWAYS assumes diagonals of sig_x is in log scale.
diag(sig_x) = log(diag(sig_x))
par_init = c(H=diag(H),theta=theta,sig_x=sig_x[lower.tri(sig_x,diag=TRUE)])
print(par_init)
print(lik(mod)(par_init))
print(grad(mod)(par_init))
print(hess(mod)(par_init))

### --- STEP 4: Fitting a model
fitted = fit(mod, par_init)
print(fitted)

### --- STEP 5: Estimating variance-covariance of the MLE
v_estimate = varest(mod, fitted)

### --- STEP 6: Get marginal confidence intervals
print(marginal_ci(v_estimate, lvl=0.95)) 
}
}
\references{
Mitov V, Bartoszek K, Asimomitis G, Stadler T (2019). “Fast likelihood calculation for multivariate Gaussian phylogenetic models with shifts.” Theor. Popul. Biol.. https://doi.org/10.1016/j.tpb.2019.11.005.
}
