#' Deploy an R script with an optional schedule
#'
#' Will create a build to run an R script in Cloud Build with an optional schedule from Cloud Scheduler
#'
#' @inheritParams cr_buildstep_r
#' @inheritParams cr_build
#' @inheritParams cr_schedule
#' @inheritParams cr_build_schedule_http
#' @param r_image The R docker environment executing the R code
#' @param run_name What name the R code will identify itself as.  If \code{NULL} one is autogenerated.
#' @param pre_steps Other \link{cr_buildstep} to run before the R code executes
#' @param post_steps Other \link{cr_buildstep} to run after the R code executes
#' @param ... Other arguments passed through to \link{cr_buildstep_r}
#' @details
#'
#' If \code{schedule=NULL} then the R script will be run immediately on Cloud Build via \link{cr_build}.
#'
#' If \code{schedule} carries a cron job string (e.g. \code{"15 5 * * *"}) then the build will be scheduled via Cloud Scheduler to run as described in \link{cr_build_schedule_http}
#'
#' The R script will execute within the root directory of which \link{Source} you supply, usually created via \link{cr_build_source}.  Bear in mind if the source changes then the code scheduled may need updating.
#'
#' The \code{r_image} dictates what R libraries the R environment executing the code of \code{r} will have, via the underlying Docker container usually supplied by rocker-project.org.  If you want custom R libraries beyond the default, create a docker container with those R libraries installed (perhaps via \link{cr_deploy_docker})
#'
#' @return If scheduling then a \link{Job}, if building immediately then a \link{Build}
#' @family Deployment functions
#' @export
#'
#' @seealso If you want to run R code upon certain events like GitHub pushes, look at \link{cr_buildtrigger}
#'
#' @examples
#'
#' r_lines <- c("list.files()",
#'              "library(dplyr)",
#'              "mtcars %>% select(mpg)",
#'              "sessionInfo()")
#' source <- cr_build_source(RepoSource("googleCloudStorageR",
#'                                      branchName = "master"))
#'
#' \dontrun{
#' cr_project_set("my-project")
#' cr_region_set("europe-west1")
#' cr_email_set("123456@projectid.iam.gserviceaccount.com")
#'
#' # check the script runs ok
#' cr_deploy_r(r_lines, source = source)
#'
#' # schedule the script
#' cr_deploy_r(r_lines, schedule = "15 21 * * *", source = source)
#' }
#'
cr_deploy_r <- function(r,
                        schedule = NULL,
                        source = NULL,
                        run_name = NULL,
                        r_image = "rocker/verse",
                        pre_steps = NULL,
                        post_steps = NULL,
                        timeout = 600L,
                        ...,
                        email = cr_email_get(),
                        region = cr_region_get(),
                        projectId = cr_project_get(),
                        launch_browser=interactive()){

  if(is.null(run_name)){
    run_name <- paste0("cr_rscript_", format(Sys.time(), "%Y%m%s%H%M%S"))
  }

  myMessage(paste("Deploy R script", run_name, "to Cloud Build"),
            level = 3)

  build <- cr_build_yaml(
    steps = c(pre_steps,
              cr_buildstep_r(r = r,
                             name = r_image,
                             id = run_name,
                             ...),
              post_steps)
  )

  br <- cr_build_make(build,
                      source = source,
                      timeout = timeout)

  if(!is.null(schedule)){
    # a cloud build you would like to schedule
    myMessage(paste("Scheduling R script on cron schedule:", schedule),
              level = 3)

    https <- cr_build_schedule_http(br,
                                   email = email,
                                   projectId = projectId)

    brs <- cr_schedule(schedule,
                       name=run_name,
                       region = region,
                       description = run_name,
                       httpTarget = https)
    return(brs)
  }

  # build it now
  br1 <- cr_build(br, launch_browser=launch_browser)

  cr_build_wait(br1, projectId = projectId)

}



#' Deploy a local Dockerfile to be built on ContainerRegistry
#'
#' Build a local Dockerfile in the cloud. See googleCloudRunner website for help how to generate Dockerfiles.
#'
#' @param local The folder containing the Dockerfile to build
#' @param remote The folder on Google Cloud Storage
#' @param dockerfile An optional Dockerfile built to support the script.  Not needed if 'Dockerfile' exists in folder.  If supplied will be copied into deployment folder and called "Dockerfile"
#' @param bucket The GCS bucket that will be used to deploy code source
#' @param image_name The name of the docker image to be built either full name starting with gcr.io or constructed from the image_name and projectId via \code{gcr.io/{projectId}/{image_name}}
#' @param ... Other arguments passed to \link{cr_buildstep_docker}
#' @inheritParams cr_buildstep_docker
#' @inheritParams cr_build
#' @export
#' @family Deployment functions
#'
#' @details
#'
#' This lets you deploy local folders with Dockerfiles, automating saving the source on Google Cloud Storage.
#'
#' To deploy builds on git triggers and sources such as GitHub, see the examples of \link{cr_buildstep_docker} or the use cases on the website
#'
#' @examples
#'
#' \dontrun{
#' cr_project_set("my-project")
#' cr_region_set("europe-west1")
#' cr_email_set("123456@projectid.iam.gserviceaccount.com")
#' cr_bucket_set("my-bucket")
#'
#' b <- cr_deploy_docker(system.file("example/", package="googleCloudRunner"))
#'
#' }
cr_deploy_docker <- function(local,
                             image_name = remote,
                             dockerfile = NULL,
                             remote = basename(local),
                             tag = "$BUILD_ID",
                             timeout = 600L,
                             bucket = cr_bucket_get(),
                             projectId = cr_project_get(),
                             launch_browser = interactive(),
                             ...){
  assert_that(
    dir.exists(local)
  )

  myMessage("Building ", local, " folder for Docker image: ",
            image_name, level = 3)

  myMessage(paste("Configuring Dockerfile"), level = 3)
  find_dockerfile(local, dockerfile = dockerfile)

  assert_that(
    is.readable(file.path(local, "Dockerfile"))
  )

  image <- make_image_name(image_name, projectId = projectId)

  build_yaml <- cr_build_yaml(
    steps = cr_buildstep_docker(image,
                                tag = tag,
                                location = ".",
                                dir=paste0("deploy/", remote),
                                projectId = projectId,
                                ...),
    images = image)

  image_tag <- paste0(image, ":", tag)
  myMessage(paste("#Deploy docker build for image: \n", image_tag),
            level = 3)

  remote_tar <- remote
  if(!grepl("tar\\.gz$", remote)){
    remote_tar <- paste0(remote, ".tar.gz")
  }
  gcs_source <- cr_build_upload_gcs(local,
                                    remote = remote_tar,
                                    bucket = bucket)

  docker_build <- cr_build(build_yaml,
                           source = gcs_source,
                           launch_browser = launch_browser,
                           timeout=timeout)

  b <- cr_build_wait(docker_build, projectId = projectId)

  myMessage(image_tag, level = 3)

  b
}



#' Deploy a cloudbuild.yml for a pkgdown website of an R package
#'
#' This builds a pkgdown website each time the trigger fires and deploys it to git
#'
#' @inheritParams cr_buildstep_pkgdown
#' @inheritParams cr_buildstep_gitsetup
#' @param steps extra steps to run before the pkgdown website steps run
#' @param cloudbuild_file The cloudbuild yaml file to write to
#'
#' @details
#'
#' The trigger repository needs to hold an R package configured to build a pkgdown website.
#'
#' For GitHub, the repository will also need to be linked to the project you are building within, via \url{https://console.cloud.google.com/cloud-build/triggers/connect}
#'
#' The git ssh keys need to be deployed to Google KMS for the deployment of the website - see \link{cr_buildstep_git} - this only needs to be done once per Git account.  You then need to commit the encrypted ssh key (by default called \code{id_rsa.enc})
#'
#' @seealso Create your own custom deployment using \link{cr_buildstep_pkgdown} which this function uses with some defaults.
#' @family Deployment functions
#' @export
#' @examples
#'
#' pd <- cr_deploy_pkgdown(secret = "my_git_secret")
#' pd
#' file.exists("cloudbuild-pkgdown.yml")
#' unlink("cloudbuild-pkgdown.yml")
#'
cr_deploy_pkgdown <- function(steps = NULL,
                              secret,
                              github_repo = "$_GITHUB_REPO",
                              cloudbuild_file = "cloudbuild-pkgdown.yml",
                              git_email = "googlecloudrunner@r.com",
                              env = NULL,
                              build_image = 'gcr.io/gcer-public/packagetools:master',
                              post_setup = NULL,
                              post_clone = NULL){


  build_yaml <-
    cr_build_yaml(steps = c(steps,
                   cr_buildstep_pkgdown(github_repo,
                                      git_email = git_email,
                                      secret = secret,
                                      env = env,
                                      build_image = build_image,
                                      post_setup = post_setup,
                                      post_clone = post_clone))
         )

  build <- cr_build_make(build_yaml)

  cr_build_write(build, file = cloudbuild_file)
  usethis::ui_line()
  usethis::ui_info("Complete deployment of pkgdown Cloud Build yaml:")
  usethis::ui_todo(c(
            "Go to https://console.cloud.google.com/cloud-build/triggers and
            make a build trigger pointing at this file in your repo:
            {cloudbuild_file} "))

  if(grepl("^\\$_",github_repo)){
    usethis::ui_info(c("Build Trigger substitution variable settings:",
                       "_GITHUB_REPO = username/repo"))
  }

  usethis::ui_info(c("Ignored files filter (glob): docs/**"))


  invisible(build)

}

#' Deploy a cloudbuild.yml for R package tests and upload to Codecov
#'
#' This tests an R package each time you commit, and uploads the test coverage results to Codecov
#'
#' @inheritParams cr_buildstep_packagetests
#' @param steps extra steps to run before the \link{cr_buildstep_packagetests} steps run (such as decryption of auth files)
#' @param cloudbuild_file The cloudbuild yaml file to write to
#' @param ... Other arguments passed to \link{cr_build_make}
#'
#' @details
#'
#' The trigger repository needs to hold an R package configured to do tests upon.
#'
#' For GitHub, the repository will need to be linked to the project you are building within, via \url{https://console.cloud.google.com/cloud-build/triggers/connect}
#'
#' If your tests need authentication details, add these via \link{cr_buildstep_decrypt} to the \code{steps} argument, which will prepend decrypting the authentication file before running the tests.
#'
#' If you want codecov to ignore some files then also deploy a .covrignore file to your repository - see covr website at \url{https://covr.r-lib.org/} for details.
#'
#' @seealso Create your own custom deployment using \link{cr_buildstep_packagetests} which this function uses with some defaults
#' @family Deployment functions
#' @export
#' @examples
#'
#' pd <- cr_deploy_packagetests()
#' pd
#'
#' # add a decryption step for an auth file
#' cr_deploy_packagetests(
#'   steps = cr_buildstep_secret("my_secret", "auth.json"),
#'   env = c("NOT_CRAN=true", "MY_AUTH_FILE=auth.json"),
#'   timeout = 1200
#' )
#'
#' unlink("cloudbuild-tests.yml")
#'
cr_deploy_packagetests <- function(
  steps = NULL,
  cloudbuild_file = "cloudbuild-tests.yml",
  env = c("NOT_CRAN=true"),
  test_script = NULL,
  codecov_script = NULL,
  codecov_token = "$_CODECOV_TOKEN",
  build_image = 'gcr.io/gcer-public/packagetools:master',
  ...){


  build_yaml <-
    cr_build_yaml(steps = c(steps,
                            cr_buildstep_packagetests(
                              test_script = test_script,
                              codecov_script = codecov_script,
                              codecov_token = codecov_token,
                              build_image = build_image,
                              env = env)
                            ),
                  ...
                  )

  cr_build_write(build_yaml, file = cloudbuild_file)

  usethis::ui_line()
  usethis::ui_info("Complete deployment of tests Cloud Build yaml:")
  usethis::ui_todo(c(
    "Go to https://console.cloud.google.com/cloud-build/triggers and
            make a build trigger pointing at this file in your repo:
            {cloudbuild_file} "))
  usethis::ui_info(c("Build Trigger substitution variable settings:",
                     "_CODECOV_TOKEN = your-codecov-token",
                     "Ignored files filter (glob): docs/** and vignettes/**"))

  invisible(build_yaml)

}



