library(gpindex)
# Some data for tests
set.seed(1111)
p1 <- runif(15, 1, 5)
p0 <- rlnorm(15)
q1 <- rnorm(15)^2
q0 <- rnorm(15, mean = 2)^2
pb <- rlnorm(15)
qb <- runif(15, 2, 4)

#---- Tests for arithmetic indexes ----
all.equal(arithmetic_index("Carli")(p1, p0),
          mean(p1 / p0))
all.equal(arithmetic_index("Dutot")(p1, p0),
          sum(p1) / sum(p0))
all.equal(arithmetic_index("Laspeyres")(p1, p0, q0),
          sum(p1 * q0) / sum(p0 * q0))
all.equal(arithmetic_mean(p1 / p0, index_weights("HybridPaasche")(p0, q1)),
          sum(p1 * q1) / sum(p0 * q1))
all.equal(arithmetic_index("Palgrave")(p1, p0, q1),
          weighted.mean(p1 / p0, p1 * q1 / sum(p1 * q1)))
all.equal(arithmetic_index("Drobisch")(p1, p0, q1, q0),
          0.5 * sum(p1 * q0) / sum(p0 * q0) + 0.5 * sum(p1 * q1) / sum(p0 * q1))
all.equal(arithmetic_index("Unnamed")(p1, p0, q1, q0),
          weighted.mean(p1 / p0, 0.5 * p0 * q0 / sum(p0 * q0) + 0.5 * p1 * q1 / sum(p1 * q1)))
all.equal(arithmetic_index("Walsh1")(p1, p0, q1, q0),
          sum(p1 * sqrt(q0 * q1)) / sum(p0 * sqrt(q0 * q1)))
all.equal(arithmetic_index("MarshallEdgeworth")(p1, p0, q1, q0),
          sum(p1 * (q0 + q1) / 2) / sum(p0 * (q0 + q1) / 2))
all.equal(arithmetic_index("GearyKhamis")(p1, p0, q1, q0),
          sum(p1 * 1 / (0.5 / q0 + 0.5 / q1)) / sum(p0 * 1 / (0.5 / q0 + 0.5 / q1)))
all.equal(arithmetic_index("Lowe")(p1, p0, qb),
          sum(p1 * qb) / sum(p0 * qb))
all.equal(arithmetic_index("Young")(p1, p0, pb, qb),
          weighted.mean(p1 / p0, pb * qb / sum(pb * qb)))

#---- Tests for geometric indexes ----
all.equal(geometric_index("Jevons")(p1, p0),
          prod((p1 / p0)^(1 / length(p0))))
all.equal(geometric_index("Laspeyres")(p1, p0, q0),
          geometric_mean(p1 / p0, p0 * q0 / sum(p0 * q0)))
all.equal(geometric_index("Paasche")(p1, p0, q1),
          geometric_mean(p1 / p0, p1 * q1 / sum(p1 * q1)))
all.equal(geometric_index("Young")(p1, p0, pb, qb),
          geometric_mean(p1 / p0, pb * qb / sum(pb * qb)))
all.equal(geometric_index("Tornqvist")(p1, p0, q1, q0),
          geometric_mean(p1 / p0, 0.5 * p0 * q0 / sum(p0 * q0) +
                           0.5 * p1 * q1 / sum(p1 * q1)))
all.equal(geometric_index("Vartia1")(p1, p0, q1, q0),
          prod((p1 / p0)^(logmean(p0 * q0, p1 * q1) /
                            logmean(sum(p0 * q0), sum(p1 * q1)))))
geometric_index("Vartia1")(p1, p0, q1, q0) ==
  geometric_index("MontgomeryVartia")(p1, p0, q1, q0)
all.equal(geometric_index("Vartia2")(p1, p0, q1, q0),
          geometric_mean(p1 / p0, logmean(p0 * q0 / sum(p0 * q0), p1 * q1 / sum(p1 * q1))))
geometric_index("Vartia2")(p1, p0, q1, q0) ==
  geometric_index("SatoVartia")(p1, p0, q1, q0)
all.equal(geometric_index("Walsh2")(p1, p0, q1, q0),
          geometric_mean(p1 / p0, sqrt(p1 * q1 * p0 * q0)))

#---- Tests for harmonic indexes ----
all.equal(harmonic_index("Coggeshall")(p1, p0),
          harmonic_mean(p1 / p0))
all.equal(harmonic_index("Laspeyres")(p1, p0, q0),
          harmonic_mean(p1 / p0, p0 * q0 / sum(p0 * q0)))
all.equal(harmonic_index("Paasche")(p1, p0, q1),
          harmonic_mean(p1 / p0, p1 * q1 / sum(p1 * q1)))
all.equal(arithmetic_index("Laspeyres")(p1, p0, q0),
          harmonic_mean(p1 / p0, index_weights("HybridLaspeyres")(p1, q0)))
all.equal(harmonic_index("Young")(p1, p0, pb, qb),
          harmonic_mean(p1 / p0, pb * qb / sum(pb * qb)))

#---- Tests for the other indexes ----
all.equal(fisher_index(p1, p0, q1, q0),
          geometric_mean(c(sum(p1 * q0) / sum(p0 * q0), sum(p1 * q1) / sum(p0 * q1))))
all.equal(hlp_index(p1, p0, q1, q0),
          harmonic_mean(c(sum(p1 * q0) / sum(p0 * q0), sum(p1 * q1) / sum(p0 * q1))))
all.equal(lm_index(1.5)(p1, p0, q0), 
          generalized_mean(-0.5)(p1 / p0, p0 * q0 / sum(p0 * q0)))
all.equal(cswd_index(p1, p0), 
          sqrt(arithmetic_mean(p1 / p0) * harmonic_mean(p1 / p0)))
all.equal(cswdb_index(p1, p0, q1, q0), 
          sqrt(arithmetic_mean(p1 / p0) / arithmetic_mean(q1 / q0) * arithmetic_mean(p1 * q1 / (p0 * q0))))
all.equal(bw_index(p1, p0), 
          arithmetic_mean(sqrt(p1 / p0)) * harmonic_mean(sqrt(p1 / p0)))
all.equal(stuvel_index(4, 4)(p1, p0, q1, q0), stuvel_index(1, 1)(p1, p0, q1, q0))
stuvel_index(4, 4)(p1, p0, q1, q0) != stuvel_index(2, 1)(p1, p0, q1, q0)
all.equal(arithmetic_agmean_index(0.25)(p1, p0, q0), 
          0.25 * geometric_index("Laspeyres")(p1, p0, q0) + 0.75 * laspeyres_index(p1, p0, q0))
all.equal(geometric_agmean_index(0.25)(p1, p0, q0), 
          geometric_index("Laspeyres")(p1, p0, q0)^0.25 * laspeyres_index(p1, p0, q0)^0.75)
all.equal(round(lehr_index(c(4, 2), c(2, 1), c(1, 16), c(8, 8)), 4), 1.6154)

#---- Tests for quantity indexes ----
all.equal(fisher_index(p1, p0, q1, q0),
          quantity_index(fisher_index)(p1, p0, q1, q0))
all.equal(lm_index(1.5)(p1, p0, q0), 
          quantity_index(lm_index(1.5))(p1, p0, p0 = q0))
all.equal(jevons_index(p1, p0), 
          quantity_index(jevons_index)(q1 = p1, q0 = p0))
all.equal(laspeyres_index(q1, q0, p0), 
          quantity_index(laspeyres_index)(q1, q0 = q0, p0))
all.equal(index_weights("Vartia1")(p1, p0, q1, q0), 
          quantity_index(index_weights("Vartia1"))(q1 = p1, p0, p1 = q1, q0))
all.equal(fisher_index(q1, p1 = p1, p0 = p0, q0) * quantity_index(fisher_index)(q1, p1 = p1, p0 = p0, q0),
          sum(p1 * q1) / sum(p0 * q0))

#---- Tests for weights ----
all(sum(index_weights("Vartia1")(p1, p0, q0, q1)) < 1)
length(index_weights("Carli")(integer(0))) == 0
